package Direct::Creatives;

use Direct::Modern;

use Mouse;
use Carp qw/croak/;
use List::Util qw/any/;
use List::MoreUtils qw/uniq each_array/;
use Scalar::Util qw/blessed/;
use Yandex::Validate qw/is_valid_id/;
use Yandex::ListUtils qw/as_array/;
use List::UtilsBy qw/partition_by/;

use Direct::AdGroups2::Performance;
use Direct::Model::BannerPerformance;

use Direct::Model::Creative;
use Direct::Model::CreativeGroup;
use Direct::Model::Campaign;
use Direct::Model::Creative::Manager;
use Direct::Model::Creative::Factory;
use Direct::Model::Creative::Constants;
use Direct::Creatives::Dict;

use Direct::Banners::Performance;
use Direct::Validation::BannersPerformance;

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Log;

use Yandex::I18n;
use BannerStorage qw//;
use BannerStorage::Dict;
use Client::ClientFeatures qw/has_feature/;
use Common qw/get_user_camps_name_only/;
use MailNotification;
use PrimitivesIds qw/get_clientid/;
use TextTools qw/iget_or/;

use base qw/Exporter/;
our @EXPORT = qw//;
our @EXPORT_OK = qw/
    get_used_creative_ids
    search_performance_creatives
    sync_creatives
    sync_creative_screenshots
/;


has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::Creative]');

=head2 get_available_template_ids_by_role 
 
Получить список доступных шаблонов в зависимости от роли и фичи клиента

=cut 
 
sub get_available_template_ids_by_role { 
    my ($role, $client_id) = @_; 
 
    my $templates = Direct::Model::Creative->all_templates(for_available_operations => 1);
    my @tmpl_ids = grep {(!($templates->{$_}->{only_for_super}//0) || $role eq 'super')
        # фильтрация шаблонов по фиче
        && (! defined $templates->{$_}->{feature_name} || Client::ClientFeatures::has_feature($client_id, $templates->{$_}->{feature_name}))}
        (keys %$templates), (keys %{BannerStorage::Dict::get_dict('template')});
    
    return [ uniq @tmpl_ids ];
} 

=head2 get_existing_template_ids 

Получить список id шаблонов, уже существующих у пользователя креативов 

=cut

sub get_existing_template_ids { 
    my %in = @_; 
    my ($uid, $creative_ids) = @in{qw/uid creative_ids/}; 

    return [] unless $uid; 
    my $client_id = get_clientid(uid => $uid); 

    my $existing_templates = get_all_sql(PPC(ClientID => $client_id), 
            [qq/SELECT template_id, COUNT(*) as count FROM perf_creatives/, 
            WHERE => {
                ClientID => $client_id,
                ($creative_ids && @$creative_ids ? (creative_id => $creative_ids) : ()),
                template_id__is_not_null => 1,
                creative_type__in => ['performance', 'bannerstorage'],
             },
            GROUP => 'BY template_id'] 
    ); 
    my ($total, $ids) = (0, []); 
    foreach (@$existing_templates) { 
        $total += $_->{count}; 
        push @$ids, $_->{template_id}; 
    } 
    return { 
      ids    => $ids, 
      total  => $total, 
  };
}

=head2 _prepare_filter_name

$uid - user id
$value - строка (имя креатива) или число (номер креатива)

Если $value похоже на число, то генерируется условие where creative_id between NN and NN or creative_id between NN0 and NN9 ...
Если $value похоже на строку - генерируется условие name__like "%$value%"

=cut

sub _prepare_filter_name
{
    my ($client_id, $value) = @_;
    my @filters;
    if ($value =~ /^[1-9][0-9]*$/) {
        my $max = get_one_field_sql(
                        PPC(ClientID => $client_id),
                        "SELECT MAX(creative_id) FROM perf_creatives WHERE ClientID = ?",
                        $client_id
                    );
        my @intervals = _find_int_intervals(int($value), $max);
        push @filters, ( _TEXT => join ' OR ', map { sprintf "pc.creative_id between %d AND %d", @$_ } @intervals ) if @intervals
    }
    push @filters, ( 'pc.name__like' => '%' . $value . '%' );
    return @filters;
}

=head2 _prepare_filters

$uid - user id
$filters - массив фильтров:
    [ { filter => 'filter_type', value => 'filter_value' }, ... ]

Возвращает хеш для использования в get_...sql функциях с требуемой фильтрацией

=cut

sub _prepare_filters
{
    my ($client_id, $filters) = @_;

    my %where = ('pc.creative_type' => 'performance');
    my %filters_by_type = ();
    for my $f (@$filters) {
        push @{$filters_by_type{$f->{filter}}}, @{as_array($f->{value})};
    }
    
    my @cids = ();
    
    # на верхнем уровне условия соединяются по AND
    my @and_conditions;
    while (my ($ftype, $fvalues) = each %filters_by_type) {
        my @or_conditions;
        # на втором уровне условия соединяются по OR: (name like '%foo%' OR name like '%bar%') AND (statusModerate = 'Yes')
        if ($ftype eq 'name') {
            push @or_conditions, map { _prepare_filter_name($client_id, $_) } @$fvalues;
            push @or_conditions, ( 'pc.group_name' =>  @$fvalues );
        } elsif ($ftype eq 'id') {
            push @or_conditions, ( creative_id => [ grep { is_valid_id($_) } @$fvalues ] );
        } elsif ($ftype eq 'status_moderate') {
            push @or_conditions, map {
                $_ eq 'Yes' ? ( _OR => [
                        'pc.statusModerate' => 'Yes',
                        'pc.layout_id__is_null' => 1,
                        'pc.layout_id__in' => Direct::Model::Creative::Constants::OBSOLETE_LAYOUT_IDS()
                    ] )
                : $_ =~ /No|Wait|New/ ? ( _AND => [
                        'pc.statusModerate__in' => $_ eq 'Wait' ? ['Ready', 'Sending', 'Sent']
                                                   : $_ eq 'New' ? ['New', 'Error']
                                                   : [$_],
                        'pc.layout_id__is_not_null' => 1,
                        'pc.layout_id__not_in' => Direct::Model::Creative::Constants::OBSOLETE_LAYOUT_IDS()
                    ] )
                : ()
            } @$fvalues;
        } elsif ($ftype eq 'group_id') {
            push @or_conditions, ( creative_group_id => [ grep { is_valid_id($_) } @$fvalues ] );
            if (@$fvalues == 1 && $fvalues->[0] == 0) {
                # особый случай - group_id == 0 - "устаревше креативы"
                push @or_conditions, ( creative_group_id__is_null => 1 );
            }
        } elsif ($ftype eq 'business_type') {
            foreach my $feed_business_type (@$fvalues) {
                my $creative_business_types = $Direct::Validation::BannersPerformance::FEEDS_TO_SMART_CREATIVES_COMPATIBILITY{$feed_business_type};
                foreach my $creative_business_type (@$creative_business_types) {
                    push @or_conditions, ( business_type => [$creative_business_type] );
                }
            }
        } elsif ($ftype eq 'theme') {
            push @or_conditions, ( theme_id => [ grep { is_valid_id($_) } @$fvalues ] );
        } elsif ($ftype eq 'layout') {
            push @or_conditions, ( layout_id => [ grep { is_valid_id($_) } @$fvalues ] );
        } elsif ($ftype eq 'size') {
            push @or_conditions, map { my ($w, $h) = ($_ =~ /(\d+)x(\d+)/); defined $w && defined $h ? ( _AND => [ width__int => $w, height__int => $h ] ) : () } @$fvalues;
        } elsif ($ftype eq 'create_time') {
            push @or_conditions, ( group_create_time => $fvalues );
        } elsif ($ftype eq 'campaigns') {
            push @cids, grep { is_valid_id($_) } @$fvalues;
        } else {
            die "unknown filter $ftype";
        }
        push @and_conditions, ( _OR => \@or_conditions ) if @or_conditions;
    }

    if (@cids) {
        my $creative_ids = get_one_column_sql(PPC(ClientID => $client_id), [
            "select distinct creative_id from banners_performance",
            where => {
                cid => \@cids,
            }
        ]);
        push @and_conditions, ( 'pc.creative_id' => $creative_ids );
    }

    $where{_AND} = \@and_conditions if @and_conditions;

    return \%where;
}


=head2 search_performance_creatives($uid, $exclude_creative_ids, %options)

Поиск креативов пользователя ($uid) по имени и/или creative_id

Параметры:
    $uid - владелец креативов
    $exclude_creative_ids - [] массив id креативов которые нужно исключить из поиска
                            (аргумент опциональный, может быть == undef) 
    %options - см. в _get_creative
    
Результат:
    см. в _get_creative

=cut

sub _find_int_intervals {
    my ($search_value, $max) = @_;
    # returns pairs of ints that starts with $search_value, below $max:
    # _find_int_intervals(42, 10000) -> [ [42,42], [420,429] ]
    my @intervals;
    my ($begin, $end) = ($search_value, $search_value + 1); 
    while ($begin <= $max) {
        push @intervals, [$begin, $end - 1];
        $begin *= 10;
        $end *= 10;
    }
    return @intervals;
}

sub search_performance_creatives {
    my ($uid, $exclude_creative_ids, %options) = @_;

    my $client_id = get_clientid(uid => $uid);
    my $filter = _prepare_filters($client_id, $options{filter});
    
    if (defined $exclude_creative_ids && @$exclude_creative_ids) {
        $filter->{creative_id__not_in} = $exclude_creative_ids
    }

    return _get_creatives($uid, $filter,
            map {
                exists $options{$_} ? ($_ => $options{$_}) : () 
            } qw/with_campaigns per_page page lead_creative_id with_rejection_reasons sort order group filter get_status_moderate_stats mask_obsolete_creatives/
    );
}

=head2 get_by($class, $key => $ids, $uid, %options)

Поиск креативов пользователя ($uid)

Параметры:
    $key - ключ для поиска - creative_id или group_creative_id
    $ids - значения $key для поиска, массив или число
    $uid - владелец креативов
    %options - см. в _get_creative

=cut

sub get_by {

    my ($class, $key, $vals, $uid, %options) = @_;

    croak "only `creative_id/creative_group_id` keys are supported" unless $key =~ /^(creative_id|creative_group_id)$/;

    $vals = [$vals // ()] if ref($vals) ne 'ARRAY';
    return $class->new([]) if !@$vals;

    my $items = _get_creatives($uid, {
        ( $key eq 'creative_id' ? ('pc.creative_id' => $vals) : () ),
        ( $key eq 'creative_group_id' ? ( 'pc.creative_group_id' => $vals ) : () ),
        'pc.creative_type' => 'performance'
    }, map { exists $options{$_} ? ($_ => $options{$_}) : () } qw/with_campaigns with_rejection_reasons/)->{creatives};
    return $class->new(items => $items);
}

=head2 _get_creatives($uid, $filter, %options)

Поиск креативов пользователя ($uid) по критериям($filter)

Параметры:
    $uid - владелец креативов
    $filter - $where условие отбора 
    %options
        with_campaigns - 1|0 получить информацию в каких кампаниях креатив используется
        with_rejection_reasons - 1|0 получить информацию в причинах отклонения
        get_status_moderate_stats - 1|0 получить статистику по модерации
        mask_obsolete_creatives - 1|0 подменять устаревшим креативам статус модерации на Yes
        per_page - постраничная выборка  
        page - текущая выбираемая страница (индексация страниц начиная с 1)
        lead_creative_id - сортировать по данному номеру баннера (будет [0] элементом в результате, при page == 1) 
        sort, order - параметры сортировки (sort => 'creative_id', order => 'asc|desc')
        group => 1 - возвращать не креативы, а группы креативов
        filter => [ { field => value }, ... ] - дополнительная фильтрация

Результат - хеш с полями:
    creatives => массив [Direct::Model::Creative], если group => 0
    groups => массив [Direct::Model::CreativeGroup], уесли group => 1
    total_count => N - общее число креативов, если задано per_page & page

=cut

=head2 get_client_creative_items_with_type(\@types, $chief_uid, \@ids)

    Возвращает креативы определенных типов по списоку id-шников внутри uid-а главного представителя
    Ids может быть не указан, тогда вернуться все креативы клиента

=cut

=head2 get_creatives_with_type(\@types, $uid, \@ids)

    обертка к _get_creatives вызывающая его с нужным фильтром, возвращает
    креативы на пользователе по списку creative_id, но только определенных
    указанных типов если ids скаляр, то ищется один креатив

=cut

sub get_creatives_with_type {
    my ($types, $uid, $ids) = @_;

    croak "types must be and non empty array ref" unless ref($types) eq 'ARRAY' && @$types;

    my $filter = {
        'pc.creative_type' => $types
    };
    if(defined $ids) {
        $filter->{'pc.creative_id'} = ref($ids) eq 'ARRAY' ? $ids : [ $ids ];
        croak "ids can't be empty list" unless @{$filter->{'pc.creative_id'}};
    }

    return _get_creatives($uid, $filter)->{creatives};
}

sub _get_creatives {
    my ($uid, $filter, %options) = @_;

    my $client_id = get_clientid(uid => $uid);

    $filter->{'pc.ClientID'} = $client_id;

    my ($by_page_sql, $calc_rows, @bind);
    if (exists $options{per_page} || exists $options{page}) {
        $by_page_sql = "LIMIT ? OFFSET ?";
        push @bind, $options{per_page}, $options{per_page} * ($options{page} - 1);
        $calc_rows = 'SQL_CALC_FOUND_ROWS';
    }

    my (@select_fields, @join_tables);
    if ($options{with_rejection_reasons}) {
        push @select_fields, 'mr.reason AS _mod_reason_yaml';
        push @join_tables, "LEFT JOIN mod_reasons mr ON mr.id = pc.creative_id AND mr.type = 'perf_creative'";
    }

    if ($options{group}) {
        do_sql(PPC(ClientID => $client_id), "SET SESSION group_concat_max_len = 1000000");
        push @select_fields,
            'count(*) as group_creatives_count',
            'group_concat(creative_id order by creative_id) as group_creative_ids',
            'group_concat(business_type order by creative_id) as group_business_types',
            'group_concat(coalesce(theme_id, \'\') order by creative_id) as group_theme_ids';
    }

    $calc_rows ||= '';
    push @select_fields, map { "pc.$_" } Direct::Model::Creative->get_db_columns_list('perf_creatives');
    if ($options{group}) {
        push @select_fields, map { "pc.$_" } Direct::Model::CreativeGroup->get_db_columns_list('perf_creatives');
    }

    my $order_sql = 'ORDER BY pc.creative_id';
    if ($options{lead_creative_id}) {
        $order_sql = 'ORDER BY IF(pc.creative_id = ?, -1, pc.creative_id)';
    } elsif ($options{sort} ) {
        my $sort_field = Direct::Model::Creative->field2sql($options{sort}) or die "invalid sort field '$options{sort}'";
        $order_sql = "ORDER BY $sort_field ".($options{order} eq 'desc' ? 'DESC' : 'ASC');
    }

    my $group_by_sql = '';
    if ($options{group}) {
        $group_by_sql = 'GROUP BY creative_group_id';
    }

    my $creatives = get_all_sql(PPC(ClientID => $client_id), [
        "SELECT $calc_rows", (join ',', @select_fields),
        "FROM perf_creatives pc", @join_tables,
        WHERE => {
            %$filter
        }, 
        $group_by_sql, $order_sql, $by_page_sql || ''
    ],
    ($options{lead_creative_id} ? $options{lead_creative_id} : ()), 
    @bind
    );

    my $total_count;
    if ($by_page_sql) {
        $total_count = select_found_rows(PPC(ClientID => $client_id));
    }

    if ($options{with_campaigns}) {
        my (@creative_ids, %id2creative);
        for (@$creatives) {
            push @creative_ids, $_->{creative_id};
            $id2creative{$_->{creative_id}} = $_;
            $_->{used_in_camps} = [];
        }
        my $used_in_camps = get_all_sql(PPC(ClientID => $client_id), [
            "SELECT DISTINCT bp.creative_id, b.cid FROM banners_performance bp join banners b USING(bid)",
            WHERE => {'bp.creative_id' => \@creative_ids}
        ]);
        my %cid2creatives;
        for (@$used_in_camps) {
            push @{$cid2creatives{$_->{cid}}}, $id2creative{$_->{creative_id}};
        }
        my $camps = Direct::Model::Campaign->from_db_hash_multi(
            get_user_camps_name_only($uid, {cid => [keys %cid2creatives]}),
        );
        foreach my $camp (@$camps) {
            push @{$_->{used_in_camps}}, $camp for @{$cid2creatives{$camp->id}};
        }
    }

    my $groups_total_counts = {};
    if ($options{group}) {
        my @group_ids = map { $_->{creative_group_id} } @$creatives;
        $groups_total_counts = get_hash_sql(PPC(ClientID => $client_id), [
            "select ifnull(creative_group_id,0), count(*) from perf_creatives",
            where => { _OR => [ creative_group_id => \@group_ids, creative_group_id__is_null => 1 ], ClientID => $client_id },
            "group by ifnull(creative_group_id,0)",
        ]);
    }

    my $moderate_status;
    if ($options{get_status_moderate_stats}) {
        $moderate_status = _get_creatives_status_mod($client_id, $options{mask_obsolete_creatives});
    }

    my $other_ids = [];
    if (exists $options{page} && !$options{group}) {
        $other_ids = get_all_sql(PPC(ClientID => $client_id), [
            "select creative_id as id, business_type, creative_group_id from perf_creatives pc", @join_tables, where => {
                %$filter,
            },
            $order_sql,
            "LIMIT ? OFFSET ?",
        ], $options{lead_creative_id}//(), $total_count, $options{per_page} * $options{page});
    }

    if ($options{mask_obsolete_creatives}) {
        my %obsolete_layout_ids = map { $_ => undef } @{ Direct::Model::Creative::Constants::OBSOLETE_LAYOUT_IDS() };
        for (@$creatives) {
            $_->{statusModerate} = 'Yes' if $_->{creative_type} eq 'performance' && (!$_->{layout_id} || exists $obsolete_layout_ids{ $_->{layout_id} });
        }
    }

    my (@models, $_cache);
    for my $creative (@$creatives) {
        my $model;
        if ($options{group}) {
            $creative->{creatives_data} = [];
            my $it = each_array(
                @{[ split /,/, $creative->{group_creative_ids} ]},
                @{[ split /,/, $creative->{group_business_types} ]},
                @{[ split /,/, $creative->{group_theme_ids} ]},
            );
            while (my ($id, $bt, $tid) = $it->()) {
                push @{$creative->{creatives_data}}, { id => $id, business_type => $bt, theme_id => (($tid // '') eq '' ? undef : $tid) };
            }

            $creative->{total_counts} = $groups_total_counts->{$creative->{creative_group_id}//0};
            $model = Direct::Model::CreativeGroup->from_db_hash($creative, \$_cache);
        } else {
            $model = Direct::Model::Creative::Factory->create($creative, \$_cache, ($options{with_campaigns} ? (with => 'UsedInCamps') : ()));
        }
        push @models, $model;
    }

    return {
        ( $options{group} ? ( groups => \@models ) : ( creatives => \@models ) ),
        total_count => $total_count,
        status => $moderate_status,
        other_ids => $other_ids,
    }
}

=head2 _get_creatives_status_mod

$client_id - ClientID
Считает статистику по статусам модерации для всех креативов клиента
Возвращает хеш с полями:
    accepted => N -- количество принятых на модерации
    wait => N -- на модерации
    rejected => N -- отклоненные на модерации
    draft => N -- черновики (статусы New и Error)

=cut

sub _get_creatives_status_mod
{
    my ($client_id, $mask_obsolete_creatives) = @_;
    my $moderate_status = {};
    my $stats_row = get_hash_sql(PPC(ClientID => $client_id), [
        "select
            statusModerate, count(*) as cnt
        from perf_creatives",
        where => {
            ClientID => $client_id,
            creative_type => 'performance',
            ($mask_obsolete_creatives
                ? (
                    layout_id__is_not_null => 1,
                    layout_id__not_in => Direct::Model::Creative::Constants::OBSOLETE_LAYOUT_IDS(),
                ) : ()),
        },
        'GROUP BY statusModerate'
    ]);
    my $count_obsolete = $mask_obsolete_creatives ? get_one_field_sql(PPC(ClientID => $client_id), [
        "SELECT count(*) AS cnt
         FROM perf_creatives",
         WHERE => {
             ClientID => $client_id,
             creative_type => 'performance',
             _OR => [
                 layout_id__is_null => 1,
                 layout_id__in => Direct::Model::Creative::Constants::OBSOLETE_LAYOUT_IDS(),
             ],
         }
    ]) : 0;
    my $count_all = $count_obsolete;
    $moderate_status->{accepted} = $count_obsolete;
    while (my ($status, $count) = each %$stats_row) {
        if ($status eq 'Yes') {
            $moderate_status->{accepted} += $count;
        } elsif ($status =~ /Ready|Sending|Sent/) {
            $moderate_status->{wait} += $count;
        } elsif ($status eq 'No') {
            $moderate_status->{rejected} += $count;
        } elsif ($status eq 'New' || $status eq 'Error') {
            $moderate_status->{draft} += $count;
        } else {
            warn "unknown status moderate '$status'";
        }
        $count_all += $count;
    }
    $moderate_status->{all} = $count_all;

    return $moderate_status;
}

=head2 get_used_creative_ids($adgroup_ids)

Получить id креативов используемых в группах - $adgroup_ids

Результат:
    {adgroup_id => [creative_id, creative_id]}

=cut

sub get_used_creative_ids {
    my ($adgroup_ids) = @_;

    my %creative_ids_by_gid;

    my $rows = get_all_sql(PPC(pid => $adgroup_ids), [
        "SELECT b.pid, bperf.creative_id FROM banners b JOIN banners_performance bperf USING(bid)",
        WHERE => {'b.pid' => SHARD_IDS},
    ]);
    push @{$creative_ids_by_gid{$_->{pid}}}, $_->{creative_id} for @$rows;

    return \%creative_ids_by_gid;
}


=head2 sync_creatives($creatives)

Актуализировать креативы с BannerStorage

Параметры:
    $creatives - [{creative_id => 123, client_id => 580}] массив креативов

Результат:
    $creatives - массив id креативов, которые по факту были получены из BS и сохранены в директе

=cut

sub sync_creatives {
    my ($sync_creatives) = @_;

    my ($creative_ids, $creative2client) = _extract_client_ids($sync_creatives);

    my $creatives = BannerStorage::receive_creatives($creative_ids);
    foreach my $c (@$creatives) {
        $c->client_id($creative2client->{$c->id});
    }

    for my $creative (@$creatives) {
        if ($creative->status_moderate eq 'Yes') {
            $creative->rejection_reason_ids([]);
        } elsif (defined $creative->rejection_reason_ids && $creative->rejection_reason_ids > 0) {
            $creative->has_banner_storage_reasons(1)
        }
    }

    Direct::Model::Creative::Manager->new(items => $creatives)->create_or_update;
    return [map {$_->id} @$creatives];
}

=head2 sync_creative_screenshots($creatives)

Актуализировать скриншоты креативов с BannerStorage 

Параметры:
    $creatives - [{creative_id => 123, client_id => 580}] массив креативов
   
Результат:
    массив ($synced_creatives, $skipped_creatives)
    $synced_creatives - [Direct::Model::Creative] массив креативов, которые по факту были получены из BS и сохранены в директе
    $skipped_creatives - [Direct::Model::Creative] массив креативов, которые были пропущены при обработке ( в сторедже нет скриншотов)

=cut

sub sync_creative_screenshots {
    my ($sync_creatives) = @_;

    my ($creative_ids, $creative2client) = _extract_client_ids($sync_creatives);

    my @creatives = @{ BannerStorage::receive_creatives($creative_ids) // [] };

    my ($synced_creatives, $skipped_creatives) = ([], []);
    foreach my $crtv (@creatives){
        $crtv->client_id($creative2client->{$crtv->id});
        push @{ $crtv->has_screenshot ? $synced_creatives : $skipped_creatives }, $crtv;
    }

    Direct::Model::Creative::Manager->new(items => $synced_creatives)->update_preview_url();

    return ($synced_creatives, $skipped_creatives);
}

sub _extract_client_ids {
    my ($sync_creatives) = @_;

    my (@creative_ids, %creative2client);
    for (@$sync_creatives) {
       push @creative_ids, $_->{creative_id};
       $creative2client{$_->{creative_id}} = $_->{client_id};
    }
    return (\@creative_ids, \%creative2client);
}

=head2 fill_in_clients($creatives)

Дополнить креативы данными о владельце
Параметры:
    $creatives - массив хешей [ {}, .. {} ] с данными о креативах, хеши креативов обязательно должны содержать поле creative_id

Функция меняет исходный массив креативов, добавляя client_id к параметрам каждого из них.

=cut

sub fill_in_clients {
    my $creatives = shift;
    my $creative2client = get_hash_sql(PPCDICT, [
        "SELECT creative_id, ClientID FROM shard_creative_id",
        WHERE => {creative_id => [map { blessed $_ ? $_->id : $_->{creative_id} } @$creatives]}
    ]);
    
    for (@$creatives) {
        if (blessed $_) {
            croak( 'Unhandled object type:', blessed $_ ) unless $_->isa('Direct::Model::Creative');
            $_->client_id($creative2client->{$_->id} // 0);
        }
        else {
            $_->{client_id} = $creative2client->{$_->{creative_id}} || 0;
        }
    }
    return;
}

sub _log_and_die{
    my ($log, $message) = @_;
    
    $log->out($message);
    die {message => $message};
}

=head2 create_child_creatives($child2parent)

Создать в директе дочерние креативы с привязкой, аналогичной родительским (DIRECT-55628)
$child2parent - hashref : {child_creative_id => parent_creative_id}

Опции:
    skip_cid - список кампаний, в которых не нужно создавать баннеры

Возвращает hashref:
{
        created => { child_creative_id => 1|0, ... }, - дочерний креатив создан
        errors => { child_creative_id => errorMessage, ... }, 
        warnings => { child_creative_id => warningMessage, ... },
}

=cut


sub create_child_creatives {
    my ($in, %opt) = @_;
    
    my %child2parent = %$in;
    
    my %errors;
    
    my $log = Yandex::Log->new(log_file_name => 'CreateChildCreatives.log', date_suf => "%Y%m%d");
    $log->out('got pairs (parent=>child): '.join( '; ', map {$child2parent{$_} .'=>'. $_ } keys %child2parent));
    
    my $parents = BannerStorage::receive_creatives([values %child2parent]) // [];

    my $children  = BannerStorage::receive_creatives( [keys %child2parent] );
    _log_and_die( $log, sprintf( 'Unable to get creatives - parents: %s, children: %s', 0+@$parents, 0+@$children ) ) unless @$parents && @$children;
    
    fill_in_clients($parents);

    my %parent_id_to_child_id;
    foreach my $child (keys %child2parent){
        my $parent =  $child2parent{$child};
        $parent_id_to_child_id{ $parent } //= [];
        push @{$parent_id_to_child_id{ $parent }}, $child;
    }
    
    foreach my $parent (@$parents){
        $child2parent{$_} = $parent foreach @{ $parent_id_to_child_id{$parent->id} };
    }

    my %child_by_id = map { $_->id => $_ } @$children;

    my %skipped;
    foreach my $child (@$children){
        my $parent = $child2parent{$child->id};
        if ( ref $parent) {
            $child->client_id($parent->client_id);
            # Уже принятые в bannerstorage креативы не перемодерируем
            if ($child->status_moderate ne 'Yes') {
                $child->status_moderate('New');
            }
        }
        else {
            $errors{$child->id} .= 'parent creative not found',
            $skipped{ $child->id } = 1;
        }
    }

    my $existing_creatives = get_hash_sql(PPC( creative_id => [keys %child_by_id]), [
        qq/SELECT creative_id, 1 FROM perf_creatives/,
            WHERE => { creative_id => SHARD_IDS }
    ]);

    foreach my $child (@$children){
        next if $skipped{$child->id};
        $errors{ $child->id } = 'creative already exists' if $existing_creatives->{ $child->id };

        $skipped{ $child->id } = 1 if $errors{ $child->id };
    }

    my $to_create = [ grep { !$skipped{ $_->id } } @$children];
    
    Direct::Model::Creative::Manager->new(items => $to_create)->create_or_update;
    
    my %created = map { $_->id => 1 } @$to_create;
    my %parent_ids = map { $_->id } @$parents;

    my $parent_id2sum_geo = get_hash_sql(PPC(creative_id => [%parent_ids]), [
        qq/SELECT creative_id, sum_geo FROM perf_creatives/,
            WHERE => { creative_id => SHARD_IDS }
    ]);
   
    my $creative2pid = get_all_sql(PPC(creative_id => [%parent_ids]), [
        'SELECT
            bp.creative_id,
            bp.pid,
            b.statusArch,
            b.statusShow
        FROM banners_performance bp
        JOIN banners b USING(bid)',
        WHERE => {
            ($opt{skip_cid} ? ('bp.cid__not_in' => $opt{skip_cid}) : ()),
            'bp.creative_id' => SHARD_IDS,
        },
    ]);

    my %creative_id2pids = partition_by {$_->{creative_id}} @$creative2pid;
    my @pids = uniq map {$_->{pid}} @$creative2pid;
    my $adgroups = Direct::AdGroups2::Performance->get(\@pids, extended => 1)->items_by;
    my $campaigns = get_hashes_hash_sql(PPC(cid => [uniq map { $_->campaign_id } values %$adgroups ]), [
        qq/SELECT cid, archived, statusModerate, uid FROM campaigns/,  WHERE => { cid => SHARD_IDS }
    ]);

    my %warnings;
    foreach my $parent_id (keys %creative_id2pids) {
        foreach my $item (@{$creative_id2pids{$parent_id}}){
            my $pid = $item->{pid};
            foreach my $child_id (@{$parent_id_to_child_id{$parent_id}}){
                next if $skipped{$child_id};
                my $adgroup = $adgroups->{$pid};
                unless ($adgroup->has_old){
                    my $adgroup_old = $adgroup;
                    $adgroup = $adgroup_old->clone;
                    $adgroup->old($adgroup_old);
                }
                
                my $campaign = $campaigns->{$adgroup->campaign_id};
                if ( $campaign->{archived} eq 'Yes' ){
                    $warnings{$child_id} = 'campaign archived';
                    next;
                }
                my $creative = $child_by_id{$child_id};
                # Проставляем sum_geo из родительского креатива в perf_creatives, ибо в BannerStorage актуальность sum_geo уже не поддерживается
                $creative->sum_geo($parent_id2sum_geo->{ $parent_id });
                my $new_banner = Direct::Model::BannerPerformance->new(
                    id => 0,
                    client_id => $adgroup->client_id,
                    campaign_id => $adgroup->campaign_id,
                    adgroup => $adgroup,
                    adgroup_id => $adgroup->id,
                    creative => $creative,
                    creative_id => $creative->id,
                    status_show => $item->{statusShow},
                    status_archived => $item->{statusArch},
                );
                my $vr = Direct::Validation::BannersPerformance::validate_add_performance_banners([$new_banner], $adgroup);
                unless($vr->is_valid){
                    $warnings{$child_id } = join( '; ',  @{$vr->get_error_descriptions} );
                    next;
                }
                # В группе без условий показа баннер можен быть только черновиком
                # см Banners.pm "Cannot create non-draft banner in AdGroup without show conditions"
                if ($adgroup->status_moderate ne 'New' && $adgroup->has_show_conditions
                    && $campaign->{statusModerate} ne 'New' && $child2parent{$child_id}->status_moderate ne 'New') {
                    if ($creative->status_moderate eq 'Yes') {
                        $new_banner->status_moderate('Yes');
                    } else {
                        $new_banner->status_moderate('Ready');
                    }
                }
                else {
                    $warnings{$child_id} = 'creative created as draft';
                }
                unless ($adgroup->has_banners) { $adgroup->banners([$new_banner]) }
                else { push @{$adgroup->banners}, $new_banner }
            
                $adgroup->banners_count(scalar @{$adgroup->banners});
                $adgroups->{$pid} = $adgroup;
            }
        }
    }
    
    #Теперь создаем баннеры
    foreach my $adgroup (values %$adgroups){
        next unless $adgroup->has_banners;
        
        my $new_banners = [ grep {!$_->id } @{$adgroup->banners()} ];
        next unless @$new_banners;
        
        my $chief_uid = $campaigns->{$adgroup->campaign_id}->{uid};
        MailNotification::save_UID_host($chief_uid);

        do_in_transaction {
            Direct::Banners::Performance->new($new_banners)->create($chief_uid);
        }
    }
    
    $log->out('created:',  join('; ', keys %created));
    $log->out('warnings:', join('; ', map { $_.':'.$warnings{$_}} keys %warnings));
    $log->out('errors:',   join('; ', map { $_.':'.$errors{$_}} keys %errors));
    
    return {
        created => \%created,
        errors => \%errors,
        warnings => \%warnings,
    }
}

=head2 get_filters

Получить доступные для клиента фильтры креативов
%options:
    ClientID

=cut

sub get_filters
{
    my %opt = @_;
    my $client_id = $opt{ClientID};

    my $filters = {};
    do_sql(PPC(ClientID => $client_id), "SET SESSION group_concat_max_len = 1000000");
    my $filter_values = get_one_line_sql(PPC(ClientID => $client_id), [
        "select
            group_concat(distinct business_type) as business_type,
            group_concat(distinct theme_id) as themes,
            group_concat(distinct layout_id) as layouts,
            group_concat(distinct concat_ws('x', width, height)) as sizes,
            group_concat(distinct group_create_time) as create_times
        from perf_creatives pc",
        where => {
            ClientID => $client_id,
            creative_type => 'performance',
        },
        'group by ClientID'
    ]);

    my @cids = @{ get_one_column_sql(PPC(ClientID => $client_id), [
        "select distinct cid
        from banners_performance bp
        join perf_creatives pc using(creative_id)",
        where => { ClientID => $client_id, creative_type => 'performance' }
    ]) };
    my $cid2camp_name = get_hash_sql(PPC(ClientID => $client_id), [
        "select cid, name from campaigns",
        where => {
            cid => \@cids
        }
    ]);
    $filters->{business_type} = {
        name => iget('Тип бизнеса'),
        values => [
            map { 
                { name => iget_or(Direct::Creatives::Dict::get_business_type($_)->{name}), value => $_ } 
            }
            sort split /,/, $filter_values->{business_type}
        ],
    };
    $filters->{theme} = {
        name => iget('Тематика'),
        values => [
            map {
                { name => iget_or(BannerStorage::Dict::get_dict('theme')->{$_}->{name}), value => $_ }
            } sort split /,/, ($filter_values->{themes}//'')
        ],
    };
    $filters->{layout} = {
        name => iget('Макет'),
        values => [
            map {
                my $layout = BannerStorage::Dict::get_dict('layout')->{$_}; { name => iget_or($layout->{name}), value => $_, img_src => $layout->{img_src} }
            } sort split /,/, $filter_values->{layouts}//''
        ],
    };
    $filters->{size} = {
        name => iget('Размер'),
        values => [
            map {
                { name => $_ eq $Direct::Model::Creative::ADAPTIVE_SIZE ? $Direct::Model::Creative::ADAPTIVE_SIZE_TITLE : $_, value => $_ }
            } sort split /,/, $filter_values->{sizes}
        ],
    };
    $filters->{create_time} = {
        name => iget('Время создания'),
        values => [
            map {
                { name => $_, value => $_ },
            } sort split /,/, ($filter_values->{create_times}//'')
        ],
    };
    $filters->{campaigns} = {
        name => iget('Привязка к кампании'),
        values => [
            map { { name => "№$_ — $cid2camp_name->{$_}", value => $_ } } @cids
        ],
    };
    return $filters;
}

1;
