use strict;
use warnings;
use utf8;

=head1 NAME
    
    Direct::Defect - базовый класс "ошибка" 
    
=head1 SYNOPSIS

    use Direct::Defect;
    
    my $fault = new Direct::Defect(
        code => 10002, name => 'ReqField',
        text => 'Short human readable text',
        type => 'error',
        description => 'Some logn description text'
    );
    
    $fault->code; # 1002 - числовой код ошибки
    $fault->name; # ReqField - текстовый код ошибки (человекочитаемый)
    
    $fault->type; # тип ошибки, допустимые значения error | warning
    $fault->is_error; # 1 | 0
    $fault->is_warning; # 1 | 0
    
    $fault->text; # короткое текстовое описание ошибки
    $fault->description; # длинный поясняющий текст для ошибки (содержит описание причин возникновения ошибки)
    
=cut

package Direct::Defect;

use TextTools qw//;

use Mouse;

=head2 type

    Возвращает 'error' или 'warning'

=head2 code

    Уникальный код ошибки - число, ошибки и ворнинги могут пересекаться по коду

=head2 name

    Текстовый идентификатор кода, состоит из тех же символов что и название
    процедуры вперле, т.к. используется для генерации имени конструктора ошибки

=head2 text

    Текст ошибки

=head2 description

    Описание (уточнение ошибки), может содержать шаблон для дальнейшей обработки через TextTools::process_text_template
    Мутабельно, для того, чтобы шаблон можно было развернуть позже

=cut

has 'type' => (is => 'ro', isa => 'Str', required => 1);
has 'code' => (is => 'ro', isa => 'Int', required => 1);
has 'name' => (is => 'ro', isa => 'Str', required => 1);
has 'suffix' => (is => 'ro', isa => 'Str', default => '');
has 'text' => (is => 'ro', isa => 'Str', required => 1);
has 'description' => (is => 'rw', isa => 'Str', default => '');

=head2 is_error

    Возвращает истину если объект типа error

=cut

sub is_error {
    my $self = shift;
    return $self->type eq 'error';
}

=head2 is_warning

    Возвращает истину если объект типа warning

=cut

sub is_warning {
    my $self = shift;
    return $self->type eq 'warning';
}

=head2 process_description($values)

    Подставляем в шаблон из description значения из $values = { 'placeholder' => 'value', .. },
    если задано, иначе ничего не делает

=cut

sub process_description {
    my ($self, $values) = @_;

    return unless $values;

    $self->description( TextTools::process_text_template($self->description, %$values) );
    return;
}

=head2 TO_JSON

    Блеснутый объект будет проигнорирован в to_json, соответственно ошибка
    выброшенная через попадет в лог как null. -convert_blessed_universally
    исправляет ситуацию, но это эксперементальная фича и она ставится глобально
    на весь процесс. Пример где это стрельнуло https://st.yandex-team.ru/DIRECT-54937#1467276554000

=cut

sub TO_JSON {
    my $self = shift;
    return {
        type => $self->type,
        code => $self->code,
        name => $self->name,
        text => $self->text,
        description => $self->description,
    }
}

__PACKAGE__->meta->make_immutable;

1;
