package Direct::DynamicConditions;

use Mouse;
with qw/Direct::Role::Copyable/;

use Direct::Modern;

use Settings;

use Direct::Model::DynamicCondition;
use Direct::Model::DynamicCondition::Manager;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::HashUtils qw/hash_cut hash_merge/;

use ShardingTools qw/choose_shard_param/;
use LogTools;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::DynamicCondition]');
has 'data'  => (is => 'ro', isa => 'HashRef', init_arg => undef, lazy => 1, default => sub { +{}; });

around BUILDARGS => sub { my ($orig, $class) = (shift, shift); $class->$orig(@_ == 1 ? (items => $_[0]) : @_) };

=head2 manager_class
=cut

sub manager_class { 'Direct::Model::DynamicCondition::Manager' }

=head2 WEB_FIELD_NAMES

Название полей в web интерфейсе для расшифровки результатов валидации.

=cut

sub WEB_FIELD_NAMES {(
    condition_name      => {field => sprintf('"%s"', iget('Название условия нацеливания'))},
    price               => {field => sprintf('"%s"', iget('Цена на поиске'))},
    price_context       => {field => sprintf('"%s"', iget('Цена на сети'))},
    autobudget_priority => {field => sprintf('"%s"', iget('Приоритет автобюджета'))},
)}

my %KEYS_MAP = (id => 'dyn_id', gid => 'adgroup_id', pid => 'adgroup_id', cid => 'campaign_id');

=head2 get_by($key, $vals, %options)

По заданному критерию возвращает instance с выбранными условиями нацеливания.

Параметры:
    $key      -> по какому ключу выбирать: campaign_id/adgroup_id/dyn_cond_id
    $vals     -> scalar|arrayref; значение ключа
    %options:
        'shard_key'  -> (если $key равен 'dyn_cond_id'); идентификатор для выбора номера шарда,
                            должен быть одним из @{$ShardingTools::DEFAULT_MAPPING_SHARD_KEYS}
        with_deleted -> выбирать также удаленные условия нацеливания (без ставок!)
        filter       -> (HashRef); дополнительный фильтр
        order_by     -> сортировка по: dyn_cond_id (по умолчанию) / dyn_id
        limit/offset -> параметры для постраничной выборки
        for_update   -> выборка для последующего обновления (блокирует чтение)

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;

    my $self = $class->new(items => []);
    return $self if !defined $vals || (ref($vals) eq 'ARRAY' && !@$vals);

    my @shard;
    my %where = %{$options{filter} // {}};
    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};
    if ($key eq 'adgroup_id') {
        @shard = (pid => $vals);
        $where{'dc.pid'} = SHARD_IDS;
    } elsif ($key eq 'campaign_id') {
        @shard = (cid => $vals);
        $where{'g.cid'} = SHARD_IDS;
    } elsif ($key eq 'dyn_id') {
        @shard = $options{shard} ? (shard => $options{shard}) : choose_shard_param(\%options);
        $where{'bd.dyn_id'} = $vals;
    } elsif ($key eq 'dyn_cond_id') {
        @shard = $options{shard} ? (shard => $options{shard}) : choose_shard_param(\%options);
        $where{'dc.dyn_cond_id'} = $vals;
    } else {
        croak "Unknown get_by key: `$key`";
    }

    my $with_deleted = $options{with_deleted};
    my (@select_columns, @from_tables);

    push @select_columns,
        Direct::Model::DynamicCondition->get_db_columns(bids_dynamic => 'bd', prefix => ''),
        Direct::Model::DynamicCondition->get_db_columns(dynamic_conditions => 'dc', prefix => ''),
        (qw/g.cid/),
        ($with_deleted ? 'dc.dyn_cond_id, dc.pid' : ()),
        'feeds.feed_id, feeds.feed_type, feeds.business_type';

    push @from_tables,
        'dynamic_conditions dc',
        ($with_deleted ? 'LEFT ' : '').'JOIN bids_dynamic bd ON (bd.dyn_cond_id = dc.dyn_cond_id)',
        'JOIN phrases g ON (g.pid = dc.pid)',
        'JOIN adgroups_dynamic gd ON (gd.pid = g.pid)',
        'LEFT JOIN feeds on feeds.feed_id = gd.feed_id';

    croak "unknown `order_by` option: ".$options{order_by} if $options{order_by} && $options{order_by} !~ /^dyn(?:_cond)?_id$/;
    my $order_by = ($options{order_by} // 'dyn_cond_id') eq 'dyn_cond_id' ? 'dc.dyn_cond_id' : 'bd.dyn_id';

    my $dyn_cond_rows = get_all_sql(PPC(@shard), [
        sprintf('SELECT %s FROM %s', join(', ', @select_columns), join(' ', @from_tables)),
        where => \%where,
        "ORDER BY $order_by",
        $options{limit} ? (
            limit => $options{limit}, $options{offset} ? (offset => $options{offset}) : (),
        ) : (),
        $options{for_update} ? ("FOR UPDATE") : (),
    ]);

    # Если не получили ставки из bids_dynamic, то удалим лишние колонки
    delete @$_{qw/dyn_id price price_context autobudgetPriority statusBsSynced opts/} for grep { !$_->{dyn_id} } @$dyn_cond_rows;
    if (!$options{with_additional}) {
        # поле from_tab должно отдаваться только при установленной опции with_additional
        delete @$_{qw/from_tab/} for grep { !$_->{dyn_id} } @$dyn_cond_rows;
    }
    
    for my $row (@$dyn_cond_rows) {
        if ($row->{feed_id} && $row->{feed_type}) {
            $row->{filter_type} = $row->{business_type} . '_' . $row->{feed_type};
        }
        else {
            $row->{filter_type} = 'dynamic_conditions';
        }
    }

    push @{$self->items}, Direct::Model::DynamicCondition->from_db_hash($_, \{}) for @$dyn_cond_rows;

    return $self;
}

=head2 items_by($key)

Возвращает структуру с условиями нацеливания, вида:
    $key //eq 'id' => {$dyn_cond1->id => $dyn_cond1, $dyn_cond2->id => $dyn_cond2, ...};
    $key eq 'gid'  => {$adgroup_id1 => [$dyn_cond1, $dyn_cond2, ...], adgroup_id2 => [$dyn_cond3, ...], ...};

=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';
    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};

    my %result;
    if ($key eq 'dyn_id') {
        $result{$_->id} = $_ for @{$self->items};
    } elsif ($key eq 'dyn_cond_id') {
        $result{$_->dyn_cond_id} = $_ for @{$self->items};
    } elsif ($key eq 'adgroup_id') {
        push @{$result{ $_->adgroup_id }}, $_ for @{$self->items};
    } else {
        croak "Unknown items_by key: `$key`";
    }

    return \%result;
}

=head2 prepare_create

Подготовка списка условий нацеливания к созданию.

=cut

sub prepare_create {
    my ($self) = @_;

    for my $dyn_cond (@{$self->items}) {
        # Если условие добавляется в группу не-черновик, то сбросим статус синхронизации на группе
        # Также, сбросим статус синхронизации всех баннеров группы (DIRECT-46222)
        if ($dyn_cond->adgroup->status_moderate ne 'New') {
            $dyn_cond->do_bs_sync_banners(1);
            $dyn_cond->do_bs_sync_adgroup(1);
            $dyn_cond->do_freeze_autobudget_alert(1);

            # Обновим статус генерации фраз в BannerLand только если условие не выключено
            if (
                $dyn_cond->adgroup->status_bl_generated eq 'No' && !($dyn_cond->has_is_suspended && $dyn_cond->is_suspended)
            ) {
                $dyn_cond->do_set_adgroup_bl_status('Processing');
            }
        }

        $dyn_cond->do_update_adgroup_last_change(1);
    }

    return $self;
}

=head2 create

Создание списка условий нацеливания в БД.

=cut

sub create {
    my ($self) = @_;
    
    # Выдача идентификаторов
    my $ids = get_new_id_multi(dyn_id => scalar(@{$self->items}));

    $_->id(shift @$ids) for @{$self->items};

    $self->prepare_create();
    $self->prepare_logging('create');
    Direct::Model::DynamicCondition::Manager->new(items => $self->items)->create();
    $self->do_logging();

    return;
}

=head2 prepare_update

Подготовка списка условий нацеливания к обновлению.

=cut

sub prepare_update {
    my ($self) = @_;

    for my $dyn_cond (@{$self->items}) {
        # Если группа не черновик и изменилось содержимое/приостановка условия,
        #   то сбросим статусы синхронизации группы/баннеров
        if (
            ($dyn_cond->is_condition_changed || $dyn_cond->is_is_suspended_changed) &&
            $dyn_cond->adgroup->status_moderate ne 'New'
        ) {
            $dyn_cond->do_bs_sync_banners(1);
            $dyn_cond->do_bs_sync_adgroup(1);

            # Обновим статус генерации фраз в BannerLand только если условие не выключено
            if ($dyn_cond->adgroup->status_bl_generated eq 'No' && !$dyn_cond->is_suspended) {
                $dyn_cond->do_set_adgroup_bl_status('Processing');
            }
        }

        # При изменении имени/содержимого/приостановки условия -- обновим время изменения группы
        if ($dyn_cond->is_condition_name_changed || $dyn_cond->is_condition_changed || $dyn_cond->is_is_suspended_changed) {
            $dyn_cond->do_update_adgroup_last_change(1);
        }

        # При изменении цен/приоритета_автобюджета условия отправляются отдельным транспортом
        if ($dyn_cond->is_price_changed || $dyn_cond->is_price_context_changed || $dyn_cond->is_autobudget_priority_changed) {
            $dyn_cond->status_bs_synced('No');
        }
    }

    return $self;
}

=head2 update(%options)

Обновление списка условий нацеливания в БД.

Параметры:
    %options:
        log_price__type -> тип логирования для цен

=cut

sub update {
    my ($self, %options) = @_;

    $self->prepare_update();
    $self->prepare_logging('update', %{hash_cut \%options, qw/log_price__type/});
    Direct::Model::DynamicCondition::Manager->new(items => $self->items)->update();
    $self->do_logging();

    return;
}

=head2 copy_extra

Копирование условий нацеливания.

Параметры:
    см. Direct::Role::Copyable

Результат:
    $src_dyn_id2dst_dyn_id - отображение исходного dyn_id в новый dyn_id
    $src_pid2src_dyn_cond_ids - {pid => [dyn_cond_id1, dyn_cond_id2 ...]} отображение исходной группы в исходные id условий

=cut

sub copy_extra {
    my ($self, $from_client_id, $source, $to_client_id, $destination, $src_id2dst_id, %options) = @_;
    my %src_pid2src_dyn_cond_ids;
    for my $cond (@{$source->items}) {
        push @{$src_pid2src_dyn_cond_ids{$cond->adgroup_id}}, $cond->id; 
    }
    return ($src_id2dst_id, \%src_pid2src_dyn_cond_ids, $destination);
}

=head2 prepare_delete

Подготовка списка условий нацеливания к удалению.

=cut

sub prepare_delete {
    my ($self) = @_;

    for my $dyn_cond (@{$self->items}) {
        if ($dyn_cond->adgroup->status_moderate ne 'New') {
            $dyn_cond->do_bs_sync_banners(1);
            $dyn_cond->do_bs_sync_adgroup(1);
        }

        $dyn_cond->do_update_adgroup_last_change(1);
    }

    return $self;
}

=head2 delete

Удаление списка условий нацеливания из БД.

=cut

sub delete {
    my ($self) = @_;

    $self->prepare_delete();
    $self->prepare_logging('delete');
    Direct::Model::DynamicCondition::Manager->new(items => $self->items)->delete();
    $self->do_logging();

    return;
}

=head2 prepare_logging($action, %params)
=head2 do_logging

Методы для логирования событий (действий).

Параметры:
    $action -> выполненное действие: create/update
    %params:
        log_price__type -> тип логирования для цен

=cut

sub prepare_logging {
    my ($self, $action, %params) = @_;

    my $adgroup_id2campaign = get_hashes_hash_sql(PPC(pid => [map { $_->adgroup_id } @{$self->items}]), [
        'SELECT g.pid, g.cid, c.currency FROM phrases g JOIN campaigns c ON (g.cid = c.cid)',
        where => {'g.pid' => SHARD_IDS},
    ]);

    for my $dyn_cond (@{$self->items}) {
        if (
            $action eq 'create' ||
            $action eq 'copy' ||
            ($action eq 'update' && ($dyn_cond->is_price_changed || $dyn_cond->is_price_context_changed))
        ) {
            push @{$self->data->{log_price}}, {
                cid => $adgroup_id2campaign->{$dyn_cond->adgroup_id}->{cid},
                pid => $dyn_cond->adgroup_id,
                id => $dyn_cond->id,
                type => $params{log_price__type} // "dyn_condition_${action}",
                price => $dyn_cond->has_price ? $dyn_cond->price : 0,
                price_ctx => $dyn_cond->has_price_context ? $dyn_cond->price_context : 0,
                currency => $adgroup_id2campaign->{$dyn_cond->adgroup_id}->{currency},
            };
        }
    }

    return;
}

sub do_logging {
    my ($self) = @_;

    LogTools::log_price($self->data->{log_price}) if $self->data->{log_price};

    return;
}

1;
