use strict;
use warnings;
use utf8;

=head1 NAME

    Direct::Errors - DSL для описания ошибок

=head1 SYNOPSIS

    package ModuleErrors;

    use Direct::Errors;

    error 'BadLang' => (code => 101201, text => 'Incorrect language');
    error 'ReqField' => (code => 1001);
    warning 'NoPhraseId' => (code => 102305, text => 'Parameter "phraseid" is empty');

    package SomeModule;

    use ModuleErrors;

    # error_Name("text message", "description")
    # warning_Name("text message", "description")
    error_BadLang("Язык текста баннера не соответствует гео таргенингу", "Для таргенинга на украину ....")
    error_ReqField();
    my $defect = warning_NoPhraseId(); # text - 'Parameter "phraseid" is empty'

    warn $defect->code; # 102305
    warn $defect->name; # 'NoPhraseId'
    warn $defect->text;
    warn $defect->is_warning;
    warn $defect->is_error;
    warn $defect->type; # 'warning' | 'error'
    warn $defect->desciption;

    package AnotherModuleErrors;

    error 'ReqField' => (code => 1001);
    error 'ReqField' => (code => 1008); # не допустимо (одинаковые имена ошибок)
    warning 'ReqField' => (code => 1001); # допустимо (типы ошибок разные)

    error 'BadRequest' => (code => 1045);
    error 'EmptyGeo' => (code => 1045); # не допустимо (одинаковые коды ошибок при одинаковых типах)

    sub error_InvalidChars {}
    error 'InvalidChars' => (code => 1050); # не допустимо (функция с таким имененм в этом пространстве имен уже есть)

=head1 DESCRIPTION

    Модуль для написания других модулей содержащих описание ошибок и/или warnings.
    При импорте модуля с описанием ошибок в вызывающее пространство имен проверяется
    пересечения по имени(name), коду(code) и типу(type) ошибки.
    Проверка пересечения на повторение ошибок также выполняется внутри описания модуля.

    Запрещены ошибки содержащие одинаковые имена (name) и тип ошибки (type)
    Запрещены ошибки содержащие одинаковые коды (code) и тип ошибки (type)

    Числовые коды ошибок следует выбирать в диапазоне 1000 .. 9999

    В рамках одного модуля в случае пересечения ошибок по кодам будет exception.

    За тем чтобы ошибки в разных модулях не пересекались по кодам следит юнит
    тест unit_tests/Direct/errors_non_intersect.t

    Ошибки и варнинги могут пересекаться по кодам, и внутри одного диапазона,
    могут как находится в одном файле, так и в разных

=head1 FUNCTIONS

=head2 error "Name" => (code => $error_code, text => $text_message)

    Описывает ошибку с типом "error", c числовым кодом ошибки $error_code,
    текстовым описание  $text_message. Имя ошибки должно быть в CamelCase.

    Ошибка будет доступна в виде именованной функции error_Name($new_text_message, $description);
    Оба параметра функции не обязательные.
    $new_text_message - текстовое, человекочитаемое сообщение об ошибке (при его отсутствии будет использован исходный текст ошибки $text_message)
    $description - подробные сведения об ошибке

=head2 warning "Name" => (code => $warning_code, text => $text_message)

    Описывает ошибку с типом "warning", c числовым кодом $warning_code,
    текстовым описание  $text_message. Имя ошибки должно быть в CamelCase.

    Варнинг будет доступна в виде именованной функции warning_Name($new_text_message, $description);
    Оба параметра функции не обязательные.
    $new_text_message - текстовое, человекочитаемое сообщение о причине варнинга (при его отсутствии будет использован исходный текст $text_message)
    $description - подробные сведения об варнинге

=cut

package Direct::Errors;

use base qw/Exporter/;
our @EXPORT = qw/error warning iget_noop/;

use Carp;
use Yandex::I18n;
use Direct::Defect;
use TextTools qw/process_text_template/;

our $NO_HASH_TAG //= 0;

*iget_noop = \&Yandex::I18n::iget_noop;

sub error($%) {

    my ($text_code, %params) = @_;
    my $caller = caller(0);
    build_defect_function($caller, error => $text_code, %params);
}

sub warning($%) {

    my ($text_code, %params) = @_;
    my $caller = caller(0);
    build_defect_function($caller, warning => $text_code, %params);
}


# {package_name => {method_name => {code => '', name => '', method_name => '', type => '', module => ''}}}
our %DEFECT_METHODS;
# {package_name => {code => {type => {}}}
our %DEFECT_TYPES;

sub build_defect_function {

    my ($package, $type, $text_code, %defect_params) = @_;

    croak "no text_code" unless $text_code;
    croak "no code for $type $text_code" unless $defect_params{code};
    croak "text_code $text_code has underscore" if $text_code =~ /_/;

    my $defect_methods = $DEFECT_METHODS{$package} ||= {};
    my $defect_types = $DEFECT_TYPES{$package} ||= {};

    my $method_name = $type . '_' . $text_code;
    my $code = $defect_params{code};
    if (exists $defect_methods->{$method_name}) {
        croak "$type $text_code already defined in module $package";
    } elsif (exists $defect_types->{$code} && exists $defect_types->{$code}->{$type}) {
        my $another_defect = $defect_types->{$code}->{$type};
        croak "$type $text_code has the same code that $another_defect->{type} $another_defect->{name} in module $package";
    } elsif ($package->can($method_name)) {
        croak "function $method_name already defined in module $package";
    }

    $defect_types->{$code}->{$type} = {
        name => $text_code,
        method_name => $method_name,
        type => $type,
    };

    my %suffixes = defined $defect_params{suffixes} ? %{$defect_params{suffixes}} : ();
    $suffixes{''} = $defect_params{description}; # основное имя ошибки с описанием по умолчанию

    foreach my $suffix_key (keys %suffixes) {
        my $defect_description = $suffixes{$suffix_key};
        my $suffix = $suffix_key ? '_' . $suffix_key : '';
        my $full_method_name = $method_name . $suffix;

        $defect_methods->{$full_method_name} = {
            code => $code,
            name => $text_code . $suffix,
            method_name => $full_method_name,
            type => $type,
            module => $package
        };

        add_method_to_package($package, $full_method_name, sub {

            my $description = shift;
            my %params = @_;

            unless (1 || $NO_HASH_TAG) { # временно выпилено
                my @caller = caller;
                my $hash = _calc_hash($caller[2], $caller[0]);
                $description = $description ? "\#$hash: $description" : "\#$hash";
            }

            return new Direct::Defect(
                type => $type,
                name => $text_code,
                code => $code,
                text => Yandex::I18n::iget($defect_params{text}),
                (defined $description ? (
                    description => process_text_template($description, %params),
                ) : defined $defect_description ? (
                    description => process_text_template(Yandex::I18n::iget($defect_description), %params),
                ) : ()),
                suffix => $suffix_key || '',
            );
        });
    }
}

sub add_method_to_package {

    my ($package, $method_name, $sub) = @_;

    croak "method $method_name duplicated" if $package->can($method_name);
    no strict 'refs';
    *{"${package}::${method_name}"} = $sub;
}

sub import {

    my $module = shift;

    if ($module eq __PACKAGE__) {
        my $caller = caller(0);
        {
            no strict 'refs';
            push @{"${caller}::ISA"}, __PACKAGE__;
        }
        utf8->import;
        strict->import;
        warnings->import;
        __PACKAGE__->export_to_level(1, undef, @EXPORT);
    } else {
        $module->export_errors(1);
    }
}

sub list_defect_names {

    my $package = shift;
    return exists $DEFECT_METHODS{$package} ? keys %{$DEFECT_METHODS{$package}} : ();
}

sub list_defect_methods {

    my $package = shift;
    return exists $DEFECT_METHODS{$package} ? values %{$DEFECT_METHODS{$package}} : ();
}

sub export_errors {

    my ($module, $level) = @_;
    my $where_to_export = caller($level);


    my @defect_names = $module->list_defect_names;
    return unless @defect_names;

    {
        no strict 'refs';
        push @{"${module}::ISA"}, 'Exporter';
        my $export = "${module}::EXPORT";
        $$export = [] unless defined $$export;
        push @$export, @defect_names;
    }

    my $defect_methods = $DEFECT_METHODS{$where_to_export} ||= {};
    my $defect_types = $DEFECT_TYPES{$where_to_export} ||= {};
    my @defect_methods = $module->list_defect_methods;
    foreach my $method (@defect_methods) {
        my $method_name = $method->{method_name};

        if (exists $defect_methods->{$method_name}) {
            # здесь ловится совпадение буквенного кода ошибки без суффиксов, потому что такие обязательно есть
            my $source_module = $defect_methods->{$method_name}->{module};
            croak "package $where_to_export: $method->{type} $method->{name} from module $module already defined in module $source_module";
        } elsif (exists $defect_types->{$method->{code}} && exists $defect_types->{$method->{code}}->{$method->{type}}
        && (!(($defect_types->{$method->{code}}->{$method->{type}}->{module} eq $method->{module}) # для численного кода проверяем, не смотрим ли мы на псевдонимы ошибок
            && _similar_name($defect_types->{$method->{code}}->{$method->{type}}->{method_name}, $method_name)))) {
            my $another_error = $defect_types->{$method->{code}}->{$method->{type}};
            croak "package $where_to_export: $method->{type} $method->{name} from module $module has the same code ($method->{code}) as that $another_error->{type} $another_error->{name} in module $another_error->{module}";
        } elsif (defined &{"${where_to_export}::$method_name"}) {
            croak "function $method_name already defined in package $where_to_export";
        }
        $defect_types->{$method->{code}}->{$method->{type}} = $defect_methods->{$method_name} = $method;
    }

    $module->export_to_level($level + 1, $module, @defect_names);
}

sub _calc_hash {
    my $num = shift;
    my $package = shift;

    # Берем первый 2 символа полного имени пакета
    # и имя файла
    my @parts = split('::', $package);
    my $short_package = substr($parts[0], 0, 2) . $parts[-1];

    return substr( # берем первые 8 цифр
        join('',
            map { hex($_) } # кодируем в десятичное число
            unpack( "H*", # преобразуем в hex
                substr( # вырезаем соль
                    crypt("$num$short_package", "DI"), # получаем хэш
                    , 2
                )
            ) =~  /(..)/g # разбиваем посимвольно
        )
        , 0, 8
    );
}

=head2 _similar_name(name1, name2)

    Правда ли, что name1 и name2 — два имени одной и той же ошибки.
    Если у имени есть суффикс (после второго underscore), отрезаем его;
    повторяем со вторым, сравниваем как строки.

=cut

sub _similar_name {
    my ($left, $right) = @_;
    $left  =~ s/_(.*)_.*/_$1/;
    $right =~ s/_(.*)_.*/_$1/;
    return ($left eq $right);
}

1;
