package Direct::Errors::Messages::Client;

use strict;
use warnings;
use utf8;

=pod

    $Id$

=head1 NAME

    Direct::Errors::Messages::Client - ошибки по вине клиента на уровне движка

=head1 SYNOPSIS

    use Direct::Errors::Messages::Client;
    my $err_obj = error_SomeClient(
        'some error description text'
    );

=head1 DESCRIPTION

    Модуль с общими клиентскими ошибками, которые ловятся на уровне движка

    Формат описания ошибок см. в Direct::Errors;

=cut

use Direct::Errors;


# Аутентификационные (паспорт) ошибки
# коды 2000..2999

error AuthFail => ( # ошибка заимствована из API4
    code => 53,
    text => iget_noop("Ошибка авторизации")
);

error FinanceTokenInvalid => ( # ошибка заимствована из API4
    code => 350,
    text => iget_noop("Неверный токен для финансовых операций")
);

error NoMasterToken => ( # ошибка заимствована из API4
    code => 359,
    text => iget_noop("MasterToken для данного клиента не существует")
);

error PayTokenNotProvided => ( # ошибка заимствована из API4
    code => 360,
    text => iget_noop("Не предоставлен платежный токен")
);

error InvalidPaymentToken => ( # ошибка заимствована из API4
    code => 361,
    text => iget_noop("Неверный платежный токен")
);

error ExpiredPaymentToken => ( # ошибка заимствована из API4
    code => 362,
    text => iget_noop("Требуется обновить платежный токен")
);

error UnknownUser => ( # ошибка заимствована из API4
    code => 513,
    text => iget_noop("Ваш логин не подключен к Яндекс.Директу")
);


# Ошибки прав
# коды 3000..3499

error NoRights => ( # ошибка заимствована из API4
    code => 54,
    text => iget_noop("Нет прав"),
    suffixes => {
        CantWrite  => iget_noop("Нет прав на запись"),
        CantDelete => iget_noop("Нет прав на удаление"),
    }
);

error AccessToApiDenied => (
    code => 3000,
    text => iget_noop("Нет доступа к API"),
    suffixes => {
        AccountBlocked  => iget_noop("Аккаунт пользователя блокирован"),
    }
);

error AccessToApiDeniedForApp => (
    code => 3002,
    text => iget_noop("Нет доступа к API у приложения"),
    suffixes => {
        DC2 => iget_noop("Эта версия Коммандера больше не поддерживается. Используйте последнюю версию: https://yandex.ru/promo/commander/new"),
    }
);

# Неподдерживаемые поля, объекты и операции
# коды 3500..3999

error 'NotSupported' => (
    code => 3500,
    text => iget_noop("Не поддерживается"),
    suffixes => {
        Multiplier => iget_noop("Корректировка данного типа не поддерживается"),
        CampaignType => iget_noop("Тип кампании не поддерживается"),
        AdGroupType => iget_noop("Группа объявлений данного типа не поддерживается"),
        AdType => iget_noop("Объявление данного типа не поддерживается"),
    }
);


# Неверные параметры запроса
# коды 4000..4999

error BadParams => (
    code => 4000,
    text => iget_noop("Параметры запроса указаны неверно")
);

error WrongSelectionCriteria => (
    code => 4001,
    text => iget_noop("Неверно заданы параметры фильтрации SelectionCriteria"),
    suffixes => {
        LimitExceeded => iget_noop('Массив #param# не может содержать более #limit# элементов'),
        NoNeededParams => iget_noop('В структуре SelectionCriteria должен быть указан хотя бы один из параметров: #params#')
    }
);

error IncorrectPage => (
    code => 4002,
    text => iget_noop("Неверно указаны параметры страницы"),
    suffixes => {
        NonpositiveLimit => iget_noop('Значение параметра Limit должно быть больше 0'),
        LimitExceeded => iget_noop('Значение параметра Limit должно быть не больше #limit#'),
        OffsetExceeded => iget_noop('Значение параметра Offset должно быть не больше #limit#'),
        NegativeOffset   => iget_noop('Значение параметра Offset должно быть больше или равно 0')
    }
);

error RequiredAtLeastOneOfParameters => (
    code => 4003,
    text => iget_noop("Не передано ни одного из необходимых параметров")
);

error PossibleOnlyOneParameter => (
    code => 4004,
    text => iget_noop("Передан больше чем один параметр"),
    description => iget_noop("Только один параметр из #fields# может быть указан")
);

error MixedTypes => (
    code => 4005,
    text => iget_noop("В одном запросе необходимо использовать Id одного типа")
);

error OperationsIncompatible => (
    code => 4006,
    text => iget_noop("В одном запросе заданы несовместимые операции")
);

error DuplicateParameterValues => (
    code => 4007,
    text => iget_noop("Повторяющиеся значения параметра")
);


# Ошибки парсинга запроса
# коды 8000..8299

error OperationNotFound => ( # ошибка заимствована из API4
    code => 55,
);

error UnknownLocale => ( # ошибка заимствована из API4
    code => 511,
    text => iget_noop("Неизвестный язык")
);

error BadRequestMethod => ( # ошибка заимствована из API4
    code => 512,
    text => iget_noop("Неверный метод запроса")
);

error BadRequest => (
    code => 8000,
    text => iget_noop("Некорректный запрос"),
    suffixes => {
        ClientLogin  => iget_noop("В HTTP-заголовке Client-Login необходимо указать логин рекламодателя"),
    }
);


# Незавершенная регистрация
# коды 8600..8799

error IncompleteSignUp => ( # ошибка заимствована из API4
    code => 58,
    text => iget_noop('Незавершенная регистрация'),
    description => iget_noop('Необходимо заполнить для приложения заявку на доступ в интерфейсе Директа и дождаться её подтверждения')
);


# Ограничение запросов к API
# коды 9000..9299

error RateLimitExceeded => (
    code => 56,
    text => iget_noop('Превышен лимит запросов метода')
);

error ConcurrentLimitExceed => ( # ошибка заимствована из API4
    code => 57,
    text => iget_noop('Превышен лимит одновременных запросов метода')
);

error NotEnoughUnits => ( # ошибка заимствована из API4
    code => 152,
    text => iget_noop('Недостаточно баллов')
);

error ConnectionsLimitExceeded => ( # ошибка заимствована из API4
    code => 506,
    text => iget_noop('Превышено ограничение на количество соединений')
);

error OfflineTaskQueueLimitExceeded => (
    code => 9000,
    text => iget_noop('Превышено ограничение на размер очереди'),
);

# Ограничение количества объектов в операциях в одном запросе
# коды 9300..9599

error RequestLimitExceeded => (
    code => 9300,
    text => iget_noop("Превышено ограничение на количество объектов в одном запросе")
);

error TooBroadSelectionClause => (
    code => 9301,
    text => iget_noop("Слишком широкое условие отбора")
);


# Зависимость ставок от стратегий
# коды 9600..9799

error DoesNotMatchStrategy => (
    code => 9600,
    text => iget_noop("Поле не соответствует установленной стратегии"),
    suffixes => {
        BroadMatch => iget_noop('Настройки дополнительных релевантных фраз не соответствуют выбранной стратегии')
    }
);

error BidChangeNotAllowed => (
    code => 9601,
    text => iget_noop("Не допускается изменение ставки")
);


# Дублирующиеся объекты
# коды 9800..9999

error Duplicated => (
    code => 9800,
    text => iget_noop("Объект не может присутствовать в запросе более одного раза")
);

error AlreadyExists => (
    code => 9801,
    text => iget_noop("Объект с указанными характеристиками уже существует"),
);

error DuplicatedArrayItem => (
    code => 9802,
    text => iget_noop("Элемент не может присутствовать в списке более одного раза")
);


# Предупреждения
# коды 10000..11000

warning Duplicated => (
    code => 10000,
    text => iget_noop("Объект присутствует в запросе более одного раза")
);

warning VCardAddDuplicate => (
    code => 10100,
    text => iget_noop("Указанная визитка дублирует ранее созданную визитку")
);

warning SitelinksAddDuplicate => (
    code => 10120,
    text => iget_noop("Указанный набор быстрых ссылок дублирует ранее созданный набор")
);

warning KeywordAlreadyExists => (
    code => 10140,
    text => iget_noop('Ключевое слово уже существует'),
);

warning NewKeywordHasBeenCreated => (
    code => 10141,
    text => iget_noop('Новое ключевое слово создано в результате обновления'),
);

warning BidNotUseful => (
    code => 10160,
    text => iget_noop("Ставка не будет применена")
);

warning PriorityNotUseful => (
    code => 10161,
    text => iget_noop("Приоритет не будет применен")
);

warning DailyBudgetReset => (
    code => 10162,
    text => iget_noop("Дневной бюджет сброшен"),
    description => iget_noop('Дневной бюджет имеет смысл только для ручных стратегий')
);

warning SettingNotChanged => (
    code => 10163,
    text => iget_noop("Настройка не будет изменена"),
    suffixes => {
        RequireServicing => iget_noop('Кампания не будет отправлена на сервисируемость, так как клиент не является сервисируемым'),
    }
);

warning AlreadyExists => (
    code => 10164,
    text => iget_noop("Объект с указанными характеристиками уже существует"),
);

warning ParamNotUseful => (
    code => 10165,
    text => iget_noop("Параметр не будет применен"),
);

warning NewStrategyHasBeenCreated => (
    code => 10166,
    text => iget_noop('Новая стратегия создана в результате обновления'),
);

warning ObjectsAlreadyLinked => (
    code => 10170,
    text => iget_noop("Объекты уже связаны"),
);

warning ObjectsNotLinked => (
    code => 10171,
    text => iget_noop("Объекты не связаны"),
);

warning RestrictedRegions => (
    code => 10172,
    text => iget_noop("Показы запрещены в некоторых регионах"),
);

warning BroadMatchReset => (
    code => 10173,
    text => iget_noop('Дополнительные релевантные фразы отключены'),
    description => iget_noop('При выбранной стратегии дополнительные релевантные фразы не применимы')
);

1;

__END__
