package Direct::Feature;

use Settings;
use Direct::Modern;
use Yandex::Trace qw//;
use Yandex::HTTP qw//;
use Yandex::TVM2;
use JSON qw/encode_json decode_json/;

use LogTools qw/log_messages/;
use HttpTools;
use EnvTools;

use base qw/Exporter/;

our $LOG_REQUESTS //= 0;

our $NUM_ATTEMPTS //= 3;
our $SOFT_TIMEOUT //= 0.3;


=head2 get_clients_uids_features($ClientIDs, $uids)

Получает для всех переданных clientIds, uids все включенные для них фичи через ручку intapi-java
Пример ответа:
{
  "result": {
    "uids_features": {},
    "client_ids_features": {
      "4227": [
        "test_feature_id",
        "test_feature_id_2"
      ],
      "5883788": [
        "feature_without_percent"
      ]
    }
  },
  "success": true
}

=cut
sub get_clients_uids_features {
    my ($ClientIDs, $uids) = @_;
    my $request = {client_ids => $ClientIDs, uids => $uids};
    my $intapi_app_id = $Settings::TVM_READONLY_INTAPI_APP_ID // $Settings::TVM2_APP_ID{intapi};
    my $ticket = eval{ Yandex::TVM2::get_ticket($intapi_app_id) }
        or die "Cannot get ticket for $intapi_app_id: $@";
    my $url = Yandex::HTTP::make_url(
        ($Settings::DIRECT_READONLY_JAVA_INTAPI_URL // $Settings::DIRECT_JAVA_INTAPI_URL). 'feature_dev/access', ''
    );
    
    my $trace_id = Yandex::Trace::current_span_id();
    my $trace_header = defined $trace_id ? join(',', map {$_//0} $trace_id, Yandex::Trace::generate_traceid(), $trace_id, 0) : undef;
    
    my $profile = Yandex::Trace::new_profile('intapi_java:feature_dev.access');

    my $response = Yandex::HTTP::http_parallel_request(POST => { 1 => {
                url => $url,
                body => encode_json($request),
            }},
        timeout => 3,
        connect_timeout => 1,
        num_attempts => $NUM_ATTEMPTS,
        soft_timeout => $SOFT_TIMEOUT,
        headers => {
            'Content-Type' => 'application/json',
            'Accept' => '*/*',
            'X-Yandex-Trace' => $trace_header,
            'X-Ya-Service-Ticket' => $ticket,
        },
    )->{1};

    if ($LOG_REQUESTS) {
        log_messages("get_clients_uids_features", {req => $request, resp => $response});
    }

    my ($uids_features, $client_ids_features, $error);
    if ($response->{is_success}) {
        my $resp_decoded = decode_json($response->{content});
        if ($resp_decoded->{success}){
            ($uids_features, $client_ids_features) = @{$resp_decoded->{result}}{qw/uids_features client_ids_features/}
        }
        else {
            $error = $response->{content};
        }
    }
    else {
        $error = $response->{content} || ('Status: ' . $response->{headers}->{Status} . ' (' . $response->{headers}->{Reason} . ')');
    }
    
    return {
        per_uid_features => $uids_features,
        per_client_id_features =>  $client_ids_features,
        error => $error,
    }
}

=head2 abt_info($ClientID)

    Получить  информацию по ab-тестам и фичам

=cut
sub abt_info {
    my ($ClientID) = @_;
    my $res = abt_info_parallel([$ClientID]);
    return $res->{$ClientID};
}


=head2 abt_info_parallel([$ClientID1, ...])

    Получить информацию по ab-тестам и фичам для нескольких клиентов
    Вернёт хэш с ключами по ClientID

=cut
sub abt_info_parallel {
    my @client_ids = @{ $_[0] };
    return {} if !@client_ids;

    my $intapi_app_id = $Settings::TVM_READONLY_INTAPI_APP_ID // $Settings::TVM2_APP_ID{intapi};
    my $ticket = eval{ Yandex::TVM2::get_ticket($intapi_app_id) }
        or die "Cannot get ticket for $intapi_app_id: $@";

    my $url = Yandex::HTTP::make_url(
        ($Settings::DIRECT_READONLY_JAVA_INTAPI_URL // $Settings::DIRECT_JAVA_INTAPI_URL). 'feature_dev/abt_info', ''
    );

    my $trace_id = Yandex::Trace::current_span_id();
    my $trace_header = defined $trace_id ? join(',', map {$_//0} $trace_id, Yandex::Trace::generate_traceid(), $trace_id, 0) : undef;

    my $profile = Yandex::Trace::new_profile('intapi_java:feature_dev.abt_info');

    my $r = HttpTools::get_request();
    my %common_request_params = (
        ip         => $r ? http_remote_ip($r) : undef,
        yandexuid  => $r ? get_cookie($r, 'yandexuid') || 0 : 0,
        yexp       => $r ? get_cookie($r, 'yexp') : undef,
        host       => $r ? http_server_host($r) : undef,
        user_agent => $r ? http_get_header($r, "User-Agent") : undef,
        interface_lang => $r ? HttpTools::lang_auto_detect($r) : undef,
    );
    if ( !is_production() && $r ){
        $common_request_params{np_fixed_features} = get_cookie($r, 'np_fixed_features');
    }
    my %reqs = map { $_ => {client_id  => $_, %common_request_params} } @client_ids;

    my $resp_hash = Yandex::HTTP::http_parallel_request(
        POST => {
            map {
                $_ => {
                    url => $url,
                    body => encode_json($reqs{$_}),
                }
            } @client_ids
        },
        timeout => 3,
        connect_timeout => 1,
        num_attempts => $NUM_ATTEMPTS,
        soft_timeout => $SOFT_TIMEOUT,
        headers => {
            'Content-Type' => 'application/json',
            'Accept' => '*/*',
            'X-Yandex-Trace' => $trace_header,
            'X-Ya-Service-Ticket' => $ticket,
        },
    );

    if ($LOG_REQUESTS) {
        log_messages("abt_info", {req => \%reqs, resp => $resp_hash});
    }

    my %result;
    for my $client_id (keys %$resp_hash) {
        my $response = $resp_hash->{$client_id};
        if ($response->{is_success}) {
            my $resp_decoded = decode_json($response->{content});
            if ($resp_decoded->{success}) {
                $result{$client_id} = {
                    features => $resp_decoded->{result}{features},
                    boxes => $resp_decoded->{result}{boxes},
                    boxes_crypted => $resp_decoded->{result}{boxes_crypted},
                };
            }
            else {
                $result{$client_id} = {error => $response->{content}};
            }
        } else {
            $result{$client_id} = {
                error => $response->{content} || 'Status: ' . $response->{headers}->{Status} . ' (' . $response->{headers}->{Reason} . ')'
            };
        }
    }
    return \%result;
}

1;
