package Direct::Feeds;

use Direct::Modern;

use Mouse;
use List::MoreUtils qw/part uniq/;
use POSIX qw/strftime/;

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils qw/hash_cut/;

use Client qw/get_client_limits/;
use TextTools qw/get_num_array_by_str/;
use Property;

use Direct::Storage;

use Direct::Model::Feed;
use Direct::Model::Feed::Manager;
use Direct::Model::FeedHistoryItem;
use Direct::Model::Campaign;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::Feed]');
has 'total' => (is => 'ro', isa => 'Int');
has 'storage' => (is => 'rw', isa => 'Direct::Storage', lazy => 1, default => sub { Direct::Storage->new() });

# Параметры работы скрипта ppcFeedToBannerland.pl
# Задаются во внутреннем отчете feed_to_banner_land_settings
# количество одновременных запросов в BL
my $DEFAULT_CHUNK_SIZE = 1;
my $CHUNK_SIZE_PROP_NAME = 'bl_chunk_size';
# сколько записей выбирать скриптом за один раз
my $DEFAULT_SELECT_CHUNK_SIZE = 3;
my $SELECT_CHUNK_SIZE_PROP_NAME = 'bl_select_chunk_size';
# после скольки ошибок больше не пытаться скачать ссылку на фид
my $DEFAULT_MAX_ERRORS_COUNT = 1;
my $MAX_ERRORS_COUNT_PROP_NAME = 'bl_max_errors_count';
# переобходим фиды со статусом Error через указанное количество дней
my $DEFAULT_RECHECK_INTERVAL_ERROR = 1;
my $RECHECK_INTERVAL_ERROR_PROP_NAME = 'bl_recheck_interval_error';
my $DEFAULT_MAX_SLEEP_TIME = 5;
my $MAX_SLEEP_TIME_PROP_NAME = 'bl_max_sleep_time_seconds';


=head2 get_by

По заданному критерию возвращает список фидов (моделей Direct::Model::Feed).

    my $feeds = get_by($client_id, %opts);

где C<%opts>:
    id/feed_id      -> искать по точному совпадению с указанным иденитификатором(ами) фида
    active          -> Bool; фильтровать фиды по активности (активный - фид который хоть раз мы выгружали в BannerLand)
    filter          -> HashRef; произвольный фильтр в формате Yandex::DBTools
    limit, offset   -> параметры для пагинации
    total_count     -> при использовании limit/offset также возвращать общее количество элементов
    sort            -> { column_name => 'asc'|'desc' }
    with_campaigns  -> для каждого фида загружать также список кампаний, в которых он используется

=cut
sub get_by {
    my ($class, $client_id, %opts) = @_;

    my @select_columns = Direct::Model::Feed->get_db_columns_list('feeds');
    my %where = (
        %{$opts{filter} // {}},
        ClientID => $client_id,
    );

    my $need_totals = $opts{total_count};
    my $calc_found_rows = $need_totals ? " SQL_CALC_FOUND_ROWS " : "";

    if (my $feed_id = ($opts{id} // $opts{feed_id})) {
        $where{feed_id} = $feed_id;
    }

    if ($opts{active}) {
        $where{update_status} = 'Done';
        $where{offers_count__gt} = 0;
    }

    my $limit_str = "";
    if ($opts{limit}) {
        $limit_str .= " LIMIT @{[int($opts{limit})]} ";
        $limit_str .= " OFFSET @{[int($opts{offset})]} " if $opts{offset};
    }

    my $order_str = "ORDER BY `feed_id`";
    if ($opts{sort}) {
        my @order_clauses;
        while (my($field, $direction) = each %{$opts{sort}}) {
            my ($field_db_name) = Direct::Model::Feed->get_db_columns_list('feeds', [$field]);
            push @order_clauses, sql_quote_identifier($field_db_name)." ".($direction eq 'asc' ? 'ASC' : 'DESC');
        }
        $order_str = "ORDER BY " . join(",", @order_clauses);
    }

    my $rows = get_all_sql(PPC(ClientID => $client_id), [
        "SELECT $calc_found_rows",
        join(", ", map { sql_quote_identifier($_) } @select_columns),
        "FROM feeds",
        WHERE => \%where,
        $order_str,
        $limit_str
    ]);
    my $found_rows = $need_totals ? select_found_rows(PPC(ClientID => $client_id)) : undef;

    my $feeds = Direct::Model::Feed->from_db_hash_multi($rows);
    $class->_enrich_with_campaigns($feeds) if $opts{with_campaigns};

    return $class->new(items => $feeds, $need_totals ? (total => $found_rows) : ());
}

=head2 get_history

Получает из БД историю загрузок одного фида (в т.ч. обрезанную для пагинации в интерфейсе).
Именно по причине этой обрезки не грузим сразу в рамках get_by().

    my $history_items = Direct::Feeds->get_history($feed->client_id, $feed_id, %opts);
    my ($history_items, $total_count_before_filtering) = Direct::Feeds->get_history($feed->client_id, $feed_id, %opts);

где %opts:
    - offset, limit - параметры для пагинации (offset без limit нельзя)

=cut
sub get_history {
    my ($self, $client_id, $feed_id, %opts) = @_;

    my @select_columns = Direct::Model::FeedHistoryItem->get_db_columns_list('perf_feed_history');
    my %where = (feed_id => $feed_id);

    my $need_totals = wantarray;
    my $calc_found_rows = $need_totals ? " SQL_CALC_FOUND_ROWS " : "";

    my $limit_str = "";
    if ($opts{limit}) {
        $limit_str .= " LIMIT @{[int($opts{limit})]} ";
        $limit_str .= " OFFSET @{[int($opts{offset})]} " if $opts{offset};
    }

    my $rows = get_all_sql(PPC(ClientID => $client_id), [
        "SELECT $calc_found_rows",
        join(", ", map { sql_quote_identifier($_) } @select_columns),
        "FROM perf_feed_history",
        WHERE => \%where,
        "ORDER BY `id` DESC",
        $limit_str,
    ]);

    my $history_items = Direct::Model::FeedHistoryItem->from_db_hash_multi($rows);

    if ($need_totals) {
        return ($history_items, select_found_rows(PPC(ClientID => $client_id)));
    } else {
        return $history_items;
    }
}

=head2 _enrich_with_campaigns($feeds)

Дополнить фиды $feeds списком кампаний, где они зайдествованы.

=cut
sub _enrich_with_campaigns {
    my ($class, $feeds) = @_;

    for my $chunk (sharded_chunks(ClientID => $feeds, by => sub { $_->client_id })) {
        my ($shard, $shard_feeds) = ($chunk->{shard}, $chunk->{ClientID});

        $_->campaigns([]) for @$shard_feeds;
        my $feed_map = { map { $_->id => $_ } @$shard_feeds };
        my @camps_mentioned;
        push @camps_mentioned, @{get_all_sql(PPC(shard => $shard), [
            "select ap.feed_id, group_concat(distinct g.cid) AS cids",
            "from adgroups_performance ap",
            "join phrases g using(pid)",
            where => { 'ap.feed_id' => [ keys %$feed_map ] },
            "group by ap.feed_id",
        ])};
        push @camps_mentioned, @{get_all_sql(PPC(shard => $shard), [
            "select ad.feed_id, group_concat(distinct g.cid) AS cids",
            "from adgroups_dynamic ad",
            "join phrases g using(pid)",
            where => { 'ad.feed_id' => [ keys %$feed_map ] },
            "group by ad.feed_id",
        ])};

        my %targets;
        for my $camp (@camps_mentioned) {
            my $feed = $feed_map->{$camp->{feed_id}};
            push @{$targets{$_}}, $feed for @{get_num_array_by_str($camp->{cids})};
        }
        my $campaigns = Direct::Model::Campaign->from_db_hash_multi(
            get_all_sql(PPC(shard => $shard), ["select cid, name from campaigns", where => {cid => [keys %targets], statusEmpty => "No"}])
        );
        for my $camp (@$campaigns) {
            $_->add_campaign($camp) for @{$targets{$camp->id}};
        }
    }

    return;
}

=head2 _enrich_with_categories($feeds)

Дополнить фиды $feeds данными по доступному дереву категорий ($feed->category).

=cut
sub _enrich_with_categories {
    my ($self, $selected_categories) = @_;
    my $feeds = $self->items;

    for my $chunk (sharded_chunks(ClientID => $feeds, by => sub { $_->client_id })) {
        my ($shard, $shard_feeds) = ($chunk->{shard}, $chunk->{ClientID});

        my @selected_feed_ids = grep {+@{$selected_categories->{$_} // []}} map {$_->id} @$shard_feeds;
        my $where = {'_OR' => [
            '_AND' => {is_deleted => 0, feed_id => [map {$_->id} @$shard_feeds]}, 
            '_AND' => {is_deleted => 1,   '_OR' => [map {('_AND' => [feed_id => $_, category_id => $selected_categories->{$_}])} 
                                                    @selected_feed_ids]}
        ]};

        my $rows = get_all_sql(PPC(shard => $shard), [
            "SELECT id, feed_id, category_id, parent_category_id, is_deleted, name FROM perf_feed_categories",
             WHERE => $where,
            "ORDER BY feed_id, category_id"
        ]);
        my (%feed_categories, %feed_categories_idx, %feed_deleted_categories);
        for my $cat (@$rows) {
            my $category = hash_cut($cat, qw/category_id parent_category_id name is_deleted/);
            $category->{is_deleted} = $category->{is_deleted} ? 1 : 0; 
            if (!$cat->{is_deleted}) {
                push @{$feed_categories{ $cat->{feed_id} }}, $category;
                $feed_categories_idx{ $cat->{feed_id} }->{ $cat->{category_id} } = $#{$feed_categories{ $cat->{feed_id} }};
            } else {
                push @{$feed_deleted_categories{ $cat->{feed_id} }}, $category;
            }
        }

        for my $feed (@$shard_feeds) {
            $feed->categories($feed_categories{$feed->id} // []);
            my $cat_idx_by_id = $feed_categories_idx{$feed->id};

            #для каждой категории заполняем поле path
            for my $cat (@{$feed->categories}) {
                next if $cat->{path};
                my $next_cat = $cat;
                my %used_categories = ($next_cat->{category_id} => 1);
                my @path = ($cat_idx_by_id->{ $next_cat->{category_id} });

                my @parent_path;
                #идем вверх по дереву пока не найдем категорию с заполненным path или не дойдем до корня
                while (exists $cat_idx_by_id->{ $next_cat->{parent_category_id} }) {
                    my $idx = $cat_idx_by_id->{ $next_cat->{parent_category_id} };
                    $next_cat = $feed->categories->[$idx];
                    if ($next_cat->{path}) {
                        @parent_path = (@{ $next_cat->{path} }, $idx);
                        last;
                    #если есть цикл, заполняем для него path отдельно
                    } elsif ($used_categories{ $next_cat->{category_id} }) {
                        my $first = (grep {$path[$_] eq $idx} 0..$#path)[0];
                        my @cycle = splice(@path, 0, $first + 1);
                        foreach my $i (0..$#cycle) {
                            $feed->categories->[$cycle[$i]]->{path} = [@cycle[$i+1..$#cycle], @cycle[0..$i-1]];
                        } 
                        @parent_path = (@{ $next_cat->{path} }, $idx);
                        last;
                    }
                    $used_categories{ $next_cat->{category_id} }++;
                    unshift @path, $idx;
                }
                foreach my $idx (@path) {
                    $feed->categories->[$idx]->{path} = [@parent_path];
                    push @parent_path, $idx;
                }
            }
            if ($feed_deleted_categories{$feed->id}) {
                push @{$feed->categories}, @{$feed_deleted_categories{$feed->id}};
            }
            for my $cat (@{$feed->categories}) {
                $cat->{path} ||= [];
                $cat->{parent_category_id} = 0 if !@{$cat->{path}};
            }
        }
    };

    return $self;
}


=head2 get_with_categories

Возвращает фиды с категориями
(для заданных групп возвращает выбранные в них категории даже если эти категории удалены)

    my $feeds = get_with_categories($client_id, %opts);

где C<%opts>:
    active_or_used  -> либо фид активный либо он есть в одной из переданных групп
    adgroups        -> список групп 
    adgroup_ids     -> список id групп 
    + параметры для get_by

=cut
sub get_with_categories {
    my ($class, $client_id, %opts) = @_;
    my $params = hash_cut(\%opts, qw/active_or_used adgroups adgroup_ids/);

    if ($params->{active_or_used}) {
        $opts{filter}->{_OR} = [
             '_AND' => {update_status => 'Done', offers_count__gt => 0},
             feed_id => [grep {$_} map {$_->{feed_id}} @{$params->{adgroups} || []}]
        ];
    }
    my $selected_categories = $class->_get_feed_selected_categories($params);
    return Direct::Feeds->get_by($client_id, %opts)->_enrich_with_categories($selected_categories);
}


=head2 _get_feed_selected_categories

По списку групп находит выбранные категории

=cut
sub _get_feed_selected_categories {
    my ($class, $options) = @_;
    my ($adgroups, $adgroup_ids) = @$options{qw/adgroups adgroup_ids/};

    my $feed_conditions = {};
    if ($adgroup_ids) {
        my $groups = Direct::AdGroups2->get_by(adgroup_id => $adgroup_ids, adgroup_type => [qw/dynamic performance/])->items; 
        my %filters_class = (dynamic => 'Direct::DynamicConditions', 'performance' => 'Direct::PerformanceFilters');
        foreach my $type (keys %filters_class) {
            my $groups_part = [grep {$_->{adgroup_type} eq $type and $_->feed_id} @$groups];
            my $filters = $filters_class{$type}->get_by(adgroup_id => [map {$_->id} @$groups_part], with_additional => 1)->items_by('adgroup_id');
            for my $group (@$groups_part) {
                push @{ $feed_conditions->{ $group->feed_id } }, map {$_->to_template_hash} @{ $filters->{ $group->id } };
            }
        }
    }
    if ($adgroups) {
        foreach my $group (@$adgroups) {
            next unless $group->{feed_id};
            push @{ $feed_conditions->{ $group->{feed_id} } }, @{ $group->{dynamic_conditions} } if $group->{dynamic_conditions};
            push @{ $feed_conditions->{ $group->{feed_id} } }, @{ $group->{performance_filters} } if $group->{performance_filters};
        }
    }

    my $selected_categories = {};
    foreach my $feed_id (keys %$feed_conditions) {
        my @conditions = map {@{ $_->{condition} }} grep {$_->{condition}} @{ $feed_conditions->{$feed_id} };
        my @categories = map {@{ $_->{value} }} 
                         grep {$_->{field} and $_->{field} eq 'categoryId' and $_->{relation} and $_->{relation} eq '==' and $_->{value}}
                         @conditions;
        $selected_categories->{$feed_id} = [uniq @categories];
    }
    return $selected_categories;
}


=head2 items_by($key)

Возвращает объекты, сгруппированные по указанному ключу.

    my $feed_id_to_feed_model = Direct::Feeds->get_by(...)->items_by('id');

=cut
sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';
    croak "Only `id` key is supported" unless $key eq 'id';

    my $result = {};
    for my $feed (@{$self->items}) {
        $result->{$feed->id} = $feed;
    }

    return $result;
}

=head2 check_add_limit($id_type, $id)

Возвращает истиное значение, если клиенту нельзя добавить больше фидов (из-за ограничения на их количество).

    my $is_addition_forbidden = Direct::Feeds->check_add_limit(client_id => $some_ClientID);

=cut
sub check_add_limit {
    my ($class, $id_type, $id) = @_;

    croak "Only `client_id` type is supported" unless $id_type eq 'client_id';
    my $client_id = $id;

    my $feed_count = get_one_field_sql(PPC(ClientID => $client_id), [
        "select count(*) from feeds", where => {ClientID => SHARD_IDS}
    ]);
    my $limits = get_client_limits($client_id);

    return $feed_count >= $limits->{feed_count_limit};
}

=head2 can_use_feeds($login_rights)

Проверяем, может ли пользователь пользоватьс функционалам по работе с фидами.

=cut
sub can_use_feeds {
    my ($class, $login_rights) = @_;
    return 1 if $login_rights->{super_control};
    return;
}

=head2 prepare_create

Подготовка фидов к сохранению (создание).

=cut
sub prepare_create {
    my ($self) = @_;

    for my $feed (@{$self->items}) {
        if ($feed->source eq 'file') {
            my $file_meta = $self->storage->save('perf_feeds', \$feed->content, ClientID => $feed->client_id);
            $feed->url($file_meta->url);
            $feed->cached_file_hash($file_meta->filename);
            $feed->refresh_interval(0);
        } elsif ($feed->source eq 'url') {
            $feed->cached_file_hash(undef);
            $feed->refresh_interval($Settings::DEFAULT_FEED_REFRESH_INTERVAL) if !$feed->has_refresh_interval || !$feed->refresh_interval;
        }
        $feed->update_status('New') if !$feed->has_update_status;
        $feed->last_change(strftime('%Y-%m-%d %H:%M:%S', localtime())) if !$feed->has_last_change;
    }

    return $self;
}

=head2 prepare_update

Подготовка фидов к сохранению (обновление).

=cut
sub prepare_update {
    my ($self) = @_;

    for my $feed (@{$self->items}) {
        croak "Cannot change feed source" if $feed->is_source_changed;

        if ($feed->source eq 'file') {
            if ($feed->has_content) {
                # Пересохраним содержимое
                my $file_meta = $self->storage->save('perf_feeds', \$feed->content, ClientID => $feed->client_id);
                $feed->url($file_meta->url);
                $feed->cached_file_hash($file_meta->filename);
            }
        }

        if (
            $feed->is_url_changed || $feed->is_login_changed || $feed->is_encrypted_password_changed ||
            $feed->is_cached_file_hash_changed  || $feed->is_is_remove_utm_changed
        ) {
            $feed->update_status('New');
            $feed->do_bs_sync_banners(1);
        }

        $feed->last_change(strftime('%Y-%m-%d %H:%M:%S', localtime())) if $feed->is_changed;
    }

    return $self;
}

=head2 save

Сохраняет новый фид или изменения в старом: в БД, в MDS. В процессе может обновлять поля
(например, update_status), в соответствии с правилами бизнес-логики.

=cut
sub save {
    my ($self) = @_;

    my ($updates, $creates) = part { $_->has_id && $_->id ? 0 : 1 } @{$self->items};

    if ($updates) {
        $self->new(items => $updates, storage => $self->storage)->prepare_update();
        Direct::Model::Feed::Manager->new(items => $updates)->update();
    }

    if ($creates) {
        $self->new(items => $creates, storage => $self->storage)->prepare_create();
        Direct::Model::Feed::Manager->new(items => $creates)->create();
    }

    return;
}

=head2 delete_unused($client_id, $feed_ids)

Удаляет указанные фиды указанного клиента, которые нигде не используются.

    my $result = Direct::Feeds->delete_unused($ClientID, $feed_ids);

где

    $result = {
        $some_feed_id_1 => "deleted",
        $some_feed_id_2 => "not_found",
        $some_feed_id_3 => "used",
        ...
    };

=cut
sub delete_unused {
    my ($class, $client_id, $feed_ids) = @_;

    my $feed_usage = get_hashes_hash_sql(PPC(ClientID => $client_id), [
        "select f.feed_id, count(distinct ap.pid) as cnt, f.update_status",
        "from feeds f left join adgroups_performance ap using (feed_id)",
        where => { ClientID => SHARD_IDS, 'f.feed_id' => $feed_ids },
        "group by f.feed_id",
    ]);

    my %delete_in_state = map { $_ => 1 } qw/Done Error/;
    my @feed_ids_to_delete = map { $_->{feed_id} } grep { $_->{cnt} == 0 && $delete_in_state{$_->{update_status}} } values %$feed_usage;
    do_in_transaction {
        my $unused_feed_ids = get_one_column_sql(
            PPC(ClientID => $client_id), [
                "select f.feed_id from feeds f left join adgroups_performance ap using(feed_id) left join adgroups_dynamic ad using(feed_id)",
                where => {'ap.feed_id__is_null' => 1, 'ad.feed_id__is_null' => 1, 'f.feed_id' => \@feed_ids_to_delete},
                "for update"
            ]
        );
        do_delete_from_table(PPC(ClientID => $client_id), 'perf_feed_vendors', where => { feed_id => $unused_feed_ids });
        do_delete_from_table(PPC(ClientID => $client_id), 'perf_feed_categories', where => { feed_id => $unused_feed_ids });
        do_delete_from_table(PPC(ClientID => $client_id), 'perf_feed_history', where => { feed_id => $unused_feed_ids });
        do_delete_from_table(PPC(ClientID => $client_id), 'feeds', where => { feed_id => $unused_feed_ids });
    };
    my $leftovers = get_hash_sql(PPC(ClientID => $client_id), [
        "select feed_id, 1 from feeds", where => {feed_id => \@feed_ids_to_delete},
    ]);

    my $result = {};
    for my $feed_id (@$feed_ids) {
        my $usage_count = exists $feed_usage->{$feed_id} ? $feed_usage->{$feed_id}{cnt} : undef;
        my $state_prohibition = exists $feed_usage->{$feed_id} ? !$delete_in_state{$feed_usage->{$feed_id}{update_status}} : undef;
        if ($leftovers->{$feed_id} || $usage_count || $state_prohibition) {
            $result->{$feed_id} = "used";
        } elsif (not exists $feed_usage->{$feed_id}) {
            $result->{$feed_id} = "not_found";
        } else {
           $result->{$feed_id} = "deleted";
        }
    }

    return $result;
}

{
     my $bl_chunk_size_prop = Property->new($CHUNK_SIZE_PROP_NAME);

=head2 get_bl_chunk_size

    Возращает заданный параметр bl_chunk_size или значение по умолчанию

=cut
    sub get_bl_chunk_size {
        my $cache_time = shift;
        return eval { $bl_chunk_size_prop->get($cache_time); } || $DEFAULT_CHUNK_SIZE;
    }

=head2 set_bl_chunk_size

    Устанавливает заданный параметр bl_chunk_size

=cut
    sub set_bl_chunk_size {
        $bl_chunk_size_prop->set(shift);
    }
}

{
    my $select_chunk_size_prop = Property->new($SELECT_CHUNK_SIZE_PROP_NAME);

=head2 get_bl_select_chunk_size

    Возращает заданный параметр bl_select_chunk_size или значение по умолчанию

=cut
    sub get_bl_select_chunk_size {
        my $cache_time = shift;
        return eval { $select_chunk_size_prop->get($cache_time); } || $DEFAULT_SELECT_CHUNK_SIZE;
    }

=head2 set_bl_select_chunk_size

    Устанавливает заданный параметр bl_select_chunk_size

=cut
    sub set_bl_select_chunk_size {
        $select_chunk_size_prop->set(shift);
    }
}

{
    my $max_errors_count_prop = Property->new($MAX_ERRORS_COUNT_PROP_NAME);

=head2 get_bl_max_errors_count

    Возращает заданный параметр bl_max_errors_count или значение по умолчанию

=cut
    sub get_bl_max_errors_count {
        my $cache_time = shift;
        return eval { $max_errors_count_prop->get($cache_time); } || $DEFAULT_MAX_ERRORS_COUNT;
    }

=head2 set_bl_max_errors_count

    Устанавливает заданный параметр bl_max_errors_count

=cut
    sub set_bl_max_errors_count {
        $max_errors_count_prop->set(shift);
    }
}

{
    my $recheck_interval_error_prop = Property->new($RECHECK_INTERVAL_ERROR_PROP_NAME);

=head2 get_bl_recheck_interval_error

        Возращает заданный параметр bl_recheck_interval_error или значение по умолчанию

=cut
    sub get_bl_recheck_interval_error {
        my $cache_time = shift;
        return eval { $recheck_interval_error_prop->get($cache_time); } || $DEFAULT_RECHECK_INTERVAL_ERROR;
    }

=head2 set_bl_recheck_interval_error

        Устанавливает заданный параметр bl_recheck_interval_error

=cut
    sub set_bl_recheck_interval_error {
        $recheck_interval_error_prop->set(shift);
    }
}

{
    my $max_sleep_time_prop = Property->new($MAX_SLEEP_TIME_PROP_NAME);

=head2 get_max_sleep_time_seconds

        Возращает заданный параметр bl_max_sleep_time_seconds или значение по умолчанию

=cut
    sub get_max_sleep_time_seconds {
        my $cache_time = shift;
        return eval { $max_sleep_time_prop->get($cache_time); } || $DEFAULT_MAX_SLEEP_TIME;
    }

=head2 set_max_sleep_time_seconds

        Устанавливает заданный параметр bl_max_sleep_time_seconds

=cut
    sub set_max_sleep_time_seconds {
        $max_sleep_time_prop->set(shift);
    }
}

=head2 extract_domain

Определить главный домен по структуре bannerland data. Выбирается самый популярный, короткий, раньше по алфавиту. Пустой домен имеет наименьший приоритет.
Вход:
{
    domain => {
        aa => 40,
        aaa => 100,
        aaaa => 100,
        bbb => 100,
        ccc => 50
    }
}
Выход:
"aaa"

=cut

sub extract_domain {
    my $data = shift;
    if (!$data || !$data->{domain} || !%{$data->{domain}}) {
        return undef;
    }
    my @domains;
    for my $domain (keys %{$data->{domain}}) {
        if ($domain ne '') {
            push @domains, {domain => $domain, num => $data->{domain}->{$domain}};
        }
    }
    return '' unless @domains;

    @domains = sort {
        $b->{num} <=> $a->{num}
        || length($a->{domain}) <=> length($b->{domain})
        || $a->{domain} cmp $b->{domain}
    } @domains;
    return $domains[0]->{domain};
}

1;
