package Direct::Keywords;

use Mouse;
with qw/Direct::Role::Copyable/;

use Direct::Modern;

use Settings;

use Direct::Model::Keyword;
use Direct::Model::Keyword::Manager;
use Direct::BsData;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::HashUtils qw/hash_cut/;
use List::MoreUtils qw/uniq any each_array/;

use LogTools qw//;
use ShardingTools qw/choose_shard_param/;
use Models::PhraseTools qw/phrase_should_null_phrase_id/;
use ModerateChecks qw/check_moderate_phrase/;
use MailNotification qw/mass_mail_notification/;
use Primitives qw//;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::Keyword]');
has 'total' => (is => 'ro', isa => 'Int');
has 'data'  => (is => 'ro', isa => 'HashRef', init_arg => undef, lazy => 1, default => sub { +{}; });

around BUILDARGS => sub { my ($orig, $class) = (shift, shift); $class->$orig(@_ == 1 ? (items => $_[0]) : @_) };

=head2 manager_class
=cut

sub manager_class { 'Direct::Model::Keyword::Manager' }

=head2 WEB_FIELD_NAMES

Название полей в web интерфейсе для расшифровки результатов валидации.

=cut

sub WEB_FIELD_NAMES {(
    text                => {field => sprintf('"%s"', iget("Текст ключевой фразы"))},
    price               => {field => sprintf('"%s"', iget('Цена на поиске'))},
    price_context       => {field => sprintf('"%s"', iget('Цена на сети'))},
    autobudget_priority => {field => sprintf('"%s"', iget('Приоритет автобюджета'))},
)}

my %KEYS_MAP = (cid => 'campaign_id', gid => 'adgroup_id', pid => 'adgroup_id', id => 'keyword_id');

=head2 get_by($key, $vals, %options)

По заданному критерию возвращает instance с выбранными ключевыми фразами.

Параметры:
    $key      -> по какому ключу выбирать: campaign_id/adgroup_id/keyword_id
    $vals     -> scalar|arrayref; значение ключа
    %options:
        'shard_key'     -> (если $key равен 'keyword_id'); идентификатор для выбора номера шарда,
                            должен быть одним из @{$ShardingTools::DEFAULT_MAPPING_SHARD_KEYS}
        limit/offset    -> параметры для постраничной выборки
        filter          -> HashRef; дополнительный фильтр
        with_auction    -> Провести аукцион для фраз и получить цены

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;

    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};
    croak "only `campaign_id/adgroup_id/keyword_id` keys are supported" unless $key =~ /^(?:campaign|adgroup|keyword)_id$/;

    return $class->new(items => []) if !defined $vals || (ref($vals) eq 'ARRAY' && !@$vals);

    my @shard;
    my %where = %{$options{filter} // {}};
    if ($key eq 'adgroup_id') {
        @shard = (pid => $vals);
        $where{'kw.pid'} = SHARD_IDS;
    } elsif ($key eq 'campaign_id') {
        @shard = (cid => $vals);
        $where{'kw.cid'} = SHARD_IDS;
    } else {
        # keyword_id
        @shard = $options{shard} ? (shard => $options{shard}) : choose_shard_param(\%options);
        $where{'kw.id'} = $vals;
    }

    my (@select_columns, @from_tables);

    push @select_columns,
        Direct::Model::Keyword->get_db_columns(bids => 'kw', prefix => ''),
        Direct::Model::Keyword->get_db_columns(bids_href_params => 'kwhp', prefix => ''),
        Direct::Model::Keyword->get_db_columns(bids_phraseid_history => 'kwhist', prefix => '');

    push @from_tables,
        'bids kw',
        'LEFT JOIN bids_href_params kwhp ON (kwhp.cid = kw.cid AND kwhp.id = kw.id)',
        'LEFT JOIN bids_phraseid_history kwhist ON (kwhist.cid = kw.cid AND kwhist.id = kw.id)';

    my $calc_found_rows = $options{limit} && $options{total_count} ? 'SQL_CALC_FOUND_ROWS' : '';

    my $keyword_rows = get_all_sql(PPC(@shard), [
        sprintf("SELECT $calc_found_rows %s FROM %s", join(', ', @select_columns), join(' ', @from_tables)),
        where => \%where,
        'ORDER BY kw.id',
        $options{limit} ? (
            limit => $options{limit}, $options{offset} ? (offset => $options{offset}) : (),
        ) : (),
    ]);

    my $found_rows = $calc_found_rows ? select_found_rows(PPC(@shard)) : undef;
    my $self = $class->new(items => [], $calc_found_rows ? (total => $found_rows) : ());

    return $self unless @$keyword_rows;

    push @{$self->items}, @{Direct::Model::Keyword->from_db_hash_multi($keyword_rows)};

    if ($options{with_auction}) {
        Direct::BsData->enrich_with_bs_data($self->items);
    }

    return $self;
}

=head2 items_by($key)

Возвращает структуру с ключевыми фразами, вида:
    $key //eq 'id' => {$keyword1->id => $keyword1, $keyword2->id => $keyword2, ...};
    $key eq 'gid'  => {$adgroup_id1 => [$keyword1, $keyword2, ...], adgroup_id2 => [$keyword3, ...], ...};

=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';
    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};
    croak "by `id`/`adgroup_id`/`keyword_id` only supported" unless $key =~ /^(?:id|adgroup_id|keyword_id)$/;

    my %result;
    if ($key eq 'keyword_id') {
        $result{$_->id} = $_ for @{$self->items};
    } else {
        push @{$result{ $_->adgroup_id }}, $_ for @{$self->items};
    }

    return \%result;
}

=head2 prepare($keywords)

Подготовка ряда значений для набора ключевых фраз.
В этом месте, например, проверяется необходимость сброса CTR и перемодерации.

=cut

sub prepare {
    my ($class, $keywords) = @_;

    my %has_new_or_changed_phrases;

    for my $keyword (@$keywords) {
        if ($keyword->has_id) {
            next if !$keyword->is_changed;
            my $old_text = $keyword->has_old && $keyword->old->text;

            if ($keyword->has_adgroup_id) {
                $has_new_or_changed_phrases{$keyword->adgroup_id} ||= !$keyword->has_old 
                    || ModerateChecks::without_minuswords($keyword->old->normalized_text) 
                        ne
                       ModerateChecks::without_minuswords($keyword->normalized_text);
            }

            if (phrase_should_null_phrase_id({phrase => $keyword->text, phrase_old => $old_text})) {
                # Нужно сбросить CTR и перемодерировать
                $keyword->bs_phrase_id(0);
                $keyword->bs_history(undef);
                $keyword->status_moderate('New');
            } elsif (check_moderate_phrase($keyword->text, $old_text) == 1) {
                # Нужна только перемодерация
                $keyword->status_moderate('New');
            }
        } else {
            # FIXME shouldn't be in update, call prepare_create", but called from Smart2
            $keyword->status_moderate('New');
            $keyword->status_bs_synced('No');
            $keyword->do_moderate_template_banners(1);
        }
    }

    foreach my $keyword (@$keywords) {
        next unless $keyword->has_adgroup_id;
        next unless $has_new_or_changed_phrases{$keyword->adgroup_id};
        $keyword->do_moderate_template_banners(1);
    }

    return;
}

=head2 prepare_create

Подготовка списка ключевых фраз к созданию.

=cut

sub prepare_create {
    my ($self) = @_;

    for my $keyword (@{$self->items}) {
        $keyword->last_change('now');
        $keyword->status_moderate('New');
        $keyword->status_bs_synced('No');
        $keyword->warn_on_place_lost('Yes') if !$keyword->has_warn_on_place_lost;

        # Если фраза добавляется в группу не-черновик
        if ($keyword->adgroup->status_moderate ne 'New') {
            # Пересинхронизируем группу
            $keyword->do_bs_sync_adgroup(1);
            $keyword->do_schedule_forecast(1);

            # Перемодерируем группу
            $keyword->do_moderate_adgroup(1);
            $keyword->do_moderate_template_banners(1);
            $keyword->do_clear_banners_moderation_flags(1);

            # Заморозим автобюджетный алерт
            $keyword->do_freeze_autobudget_alert(1);
        }

        $keyword->do_update_adgroup_last_change(1);
        $keyword->do_clear_auto_price_queue(1);
    }

    return $self;
}

=head2 create

Создание списка ключевых фраз в БД.

=cut

sub create {
    my ($self) = @_;

    # Выдача идентификаторов
    my $ids = get_new_id_multi(phid => scalar(@{$self->items}));
    $_->id(shift @$ids) for @{$self->items};

    $self->prepare_create();
    $self->prepare_logging('create');
    Direct::Model::Keyword::Manager->new(items => $self->items)->create();
    $self->do_logging();

    return;
}

=head2 prepare_update

Подготовка списка ключевых фраз к обновлению.

=cut

sub prepare_update {
    my ($self) = @_;

    $self->prepare($self->items);

    for my $keyword (@{$self->items}) {
        # Статус синхронизации группы
        # Время последнего обновления группы
        if (any { $keyword->$_ } map { "is_${_}_changed" } qw/text is_suspended href_param1 href_param2/) {
            $keyword->do_bs_sync_adgroup(1) if $keyword->adgroup->status_moderate ne 'New';
            $keyword->do_update_adgroup_last_change(1);
        }

        # При необходимости (пере)модерации фразы перемодерируем группу не-черновик
        if ($keyword->status_moderate eq 'New' && $keyword->adgroup->status_moderate ne 'New') {
            $keyword->do_moderate_adgroup(1);
            $keyword->do_moderate_template_banners(1);
            $keyword->do_clear_banners_moderation_flags(1);
        }

        # При изменении цен/приоритета_автобюджета условия отправляются отдельным транспортом
        if (any { $keyword->$_ } map { "is_${_}_changed" } qw/price price_context autobudget_priority/) {
            $keyword->status_bs_synced('No');
        }

        # Время последнего обновления фразы
        if (any { $keyword->$_ } map { "is_${_}_changed" } qw/
            text is_suspended href_param1 href_param2 price price_context autobudget_priority
        /) {
            $keyword->last_change('now');
        } elsif ($keyword->is_changed) {
            # workaround for: ON UPDATE CURRENT_TIMESTAMP
            $keyword->last_change('keep');
        }

        # Пересчет прогноза показов
        # Заморозка автобюджетного алерта
        # Удаление кампании из очереди на установку авто-цен
        if (any { $keyword->$_ } map { "is_${_}_changed" } qw/text is_suspended price price_context autobudget_priority/) {
            if ($keyword->adgroup->status_moderate ne 'New') {
                $keyword->do_schedule_forecast(1);
                if (any { $keyword->$_ } map { "is_${_}_changed" } qw/text is_suspended autobudget_priority/) {
                    $keyword->do_freeze_autobudget_alert(1);
                }
            }
            $keyword->do_clear_auto_price_queue(1);
        }
    }

    return;
}

=head2 update(%options)

Обновление списка ключевых фраз в БД.

Параметры:
    %options:
        log_price__type -> тип логирования для цен

=cut

sub update {
    my ($self, %options) = @_;

    $self->prepare_update();
    $self->prepare_logging('update', %{hash_cut \%options, qw/log_price__type/});
    Direct::Model::Keyword::Manager->new(items => $self->items)->update();
    $self->do_logging();

    return;
}

=head2 prepare_delete

Подготовка списка ключевых фраз к удалению.

=cut

sub prepare_delete {
    my ($self) = @_;
    for my $keyword (@{$self->items}) {
        $keyword->do_update_adgroup_last_change(1);
        if ($keyword->adgroup->status_moderate ne 'New') {
            $keyword->do_bs_sync_adgroup(1);
            $keyword->do_schedule_forecast(1);
        }
    }

    return $self;
}

=head2 delete

Удаление списка перфоманс фильтров (помечаем как удаленные в БД).

=cut

sub delete {
    my ($self) = @_;
    $self->prepare_delete();
    $self->prepare_logging('delete');
    Direct::Model::Keyword::Manager->new(items => $self->items)->delete();
    $self->do_logging();

    return;
}

=head2 prepare_logging($action, %params)
=head2 do_logging

Методы для логирования событий (действий).

Параметры:
    $action -> выполненное действие: create/update/delete
    %params:
        log_price__type -> тип логирования для цен
        uid             -> uid пользователя, над которым выполняется операция (по умолчанию берется из %LogTools::context)

=cut

sub prepare_logging {
    my ($self, $action, %params) = @_;

    my %log_context = %LogTools::context;
    my $uid = $params{uid} || $log_context{uid};

    croak "undefined uid" if !defined $uid;

    my @adgroup_ids = uniq(map { $_->adgroup_id } @{$self->items});
    my $adgroup_id2campaign = get_hashes_hash_sql(PPC(pid => \@adgroup_ids), [q{
        SELECT g.pid, g.cid, IFNULL(c.currency, 'YND_FIXED') AS currency FROM phrases g JOIN campaigns c ON (g.cid = c.cid)
    }, where => {
        'g.pid' => SHARD_IDS,
    }]);

    my $main_bid_by_gid = Primitives::get_main_banner_ids_by_pids(\@adgroup_ids);

    for my $keyword (@{$self->items}) {
        my $main_banner_id = $main_bid_by_gid->{$keyword->adgroup_id};
        my %logprice_tmpl = (
            %{$adgroup_id2campaign->{$keyword->adgroup_id}}, # cid, pid, currency
            id => $keyword->id,
            price => $keyword->has_price ? $keyword->price : 0,
            price_ctx => $keyword->has_price_context ? $keyword->price_context : 0,
        );

        if ($action eq 'create') {
            push @{$self->data->{log_price}}, {%logprice_tmpl, type => $params{log_price__type} // 'insert1'};

            push @{$self->data->{notifications}}, {
                object      => 'banner',
                event_type  => 'b_word',
                object_id   => $main_banner_id,
                old_text    => '',
                new_text    => $keyword->text,
                uid         => $uid,
            } if $main_banner_id;
        }
        elsif ($action eq 'update') {
            if ($keyword->is_price_changed || $keyword->is_price_context_changed) {
                push @{$self->data->{log_price}}, {%logprice_tmpl, type => $params{log_price__type} // 'update1'};
            }

            push @{$self->data->{notifications}}, {
                object      => 'phrase',
                event_type  => 'ph_price',
                object_id   => $main_banner_id,
                old_text    => $keyword->old->price,
                new_text    => $keyword->price,
                uid         => $uid,
            } if $main_banner_id && $keyword->is_price_changed;

            push @{$self->data->{notifications}}, {
                object     => 'phrase',
                event_type => 'ph_change',
                object_id  => $main_banner_id,
                old_text   => $keyword->old->text,
                new_text   => $keyword->text,
                uid        => $uid,
            } if $main_banner_id && $keyword->is_text_changed;
        }
        elsif ($action eq 'delete') {
            push @{$self->data->{log_price}}, {%logprice_tmpl, type => $params{log_price__type} // 'delete1'};

            push @{$self->data->{notifications}}, {
                object      => 'phrase',
                event_type  => 'ph_change',
                object_id   => $main_banner_id,
                old_text    => $keyword->text,
                new_text    => '',
                uid         => $uid,
            } if $main_banner_id;
        }
        else {
            croak "Unknown action: $action";
        }
    }

    return;
}

sub do_logging {
    my ($self) = @_;

    LogTools::log_price($self->data->{log_price}) if $self->data->{log_price};
    mass_mail_notification($self->data->{notifications}) if $self->data->{notifications};

    return;
}

=head2 copy_extra
=cut

sub copy_extra {}

1;
