use strict;
use warnings;
use utf8;

package Direct::Market;

use base qw/Exporter/;
our @EXPORT = qw//;
our @EXPORT_OK = qw/
    fill_in_groups_rating
    get_domains_rating
/;

use Settings;
use Yandex::DBTools;


=head2 fill_in_groups_rating($groups)

    Добавляет к каждому баннеру из группы рейтинг домена на маркете.
    Функция изменяет данные группы in place, добавляя к баннерам поле market_rating.
    market_rating == -1 если данных по рейтингу нет
        
    Параметры:
        $groups - массив [] групп объявлений,
                    каждая группа хеш {banners => [{domain => ''}]}
                    у каждого баннера ожидается поле filter_domain - домен для которого запрашивается рейтинг маркета

=cut 

sub fill_in_groups_rating {

    my ($groups) = @_;
    
    my %domains;
    foreach (@$groups) {
        push @{$domains{$_->{domain}}}, $_ foreach grep { defined $_->{domain} } @{$_->{banners}};
    }
    
    my $market_ratings = get_hash_sql(PPCDICT, [
        'SELECT dd.domain, mr.rating
        FROM market_ratings mr
        JOIN domains_dict dd ON dd.domain_id = mr.domain_id',
        WHERE => {'dd.domain' => [grep {$_} keys %domains]}
    ]);
    
    while (my ($domain, $banners) = each %domains) {
        my $rating = $domain ? $market_ratings->{$domain} || -1 : -1;
        $_->{market_rating} = $rating foreach @$banners;
    }
}

=head2 get_domains_rating($domains)

    Получить рейтинг маркета на домены

    Параметры:
        $domains - массив [] доменов
    
    Результат:
        {} - хеш {domain => rating}, если рэйтинга на домент нет, то вернется {domain => -1}

=cut

sub get_domains_rating {

    my $domains = shift;

    my $market_ratings = get_hash_sql(PPCDICT, [
        'SELECT dd.domain, mr.rating
        FROM market_ratings mr
        JOIN domains_dict dd ON dd.domain_id = mr.domain_id',
        WHERE => {'dd.domain' => $domains}
    ]);
    
    foreach my $domain (@$domains) {
        next unless defined $domain;
        $market_ratings->{$domain} = -1 unless exists $market_ratings->{$domain};
    };
    $market_ratings->{_EMPTY_DOMAIN} = -1;

    return $market_ratings;
}


1;
