package Direct::Model::AdGroup::Manager;

use Direct::Modern;
use Mouse;

extends 'Yandex::ORM::Model::Manager::Base';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils qw/xminus/;
use Yandex::DateTime qw/now/;
use DateTime::Format::MySQL;
use Scalar::Util qw/blessed/;

use Primitives qw//;
use PrimitivesIds qw//;
use MinusWords qw/mass_save_minus_words/;
use HierarchicalMultipliers qw/save_hierarchical_multipliers/;

use Direct::Model::AdGroup;

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::AdGroup]',
);

# Соответствие типа группы типу кампании, где она может находится
my %ALLOWED_ADGROUP_TO_CAMPAIGN_TYPES = (
    base    => { text => 1 },
    dynamic => { dynamic => 1 },
    mobile_content => { mobile_content => 1 },
    performance => { performance => 1 },
    mcbanner => { mcbanner => 1 },
    cpm_banner => { cpm_banner => 1, cpm_deals => 1 },
    cpm_video => { cpm_banner => 1 },
);

# (!) Этот класс -- абстрактный
sub BUILD { croak "Cannot instantiate abstract class: ".__PACKAGE__ if blessed($_[0]) eq __PACKAGE__ }

=head2 create

Создание в БД записей для соответствующих объектов (групп объявлений).

=cut

sub create {
    my ($self) = @_;

    # Выполним дополнительную проверку: не даем сохранить группу в несоответствующую по типу кампанию
    {;
        my @campaign_ids = keys %{ +{map { $_->campaign_id => 1 } @{$self->items}} };
        my $campaign_id2type = get_hash_sql(PPC(cid => \@campaign_ids), ["SELECT cid, type FROM campaigns", where => {cid => SHARD_IDS}]);
        for my $adgroup (@{$self->items}) {
            $adgroup->geo(0) if !defined $adgroup->{geo};
            my $campaign_type = $campaign_id2type->{$adgroup->campaign_id};
            next if defined $campaign_type && $ALLOWED_ADGROUP_TO_CAMPAIGN_TYPES{$adgroup->adgroup_type}->{$campaign_type};
            croak sprintf(
                "adgroup #%d with type `%s` can be added only to campaign with type: %s",
                $adgroup->id, $adgroup->adgroup_type,
                join(', ', keys %{ $ALLOWED_ADGROUP_TO_CAMPAIGN_TYPES{$adgroup->adgroup_type} }),
            );
        }
    }

    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});
        $self->_create_in_shard($shard, $shard_items);
    }

    $_->reset_state() for @{$self->items};

    return;
}

sub _create_in_shard {
    my ($self, $shard, $adgroups) = @_;

    for my $adgroup (@$adgroups) {
        $adgroup->last_change(DateTime::Format::MySQL->format_datetime(now())) if !$adgroup->has_last_change;
    }

    my @columns = Direct::Model::AdGroup->get_db_columns_list('phrases');

    do_in_transaction {
        # Сохраним минус-слова
        $self->_save_adgroup_minus_words($shard, [grep { $_->has_minus_words } @$adgroups]);

        # Создадим группы
        $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'phrases', \@columns, $adgroups);

        # Сохраним коэффициенты на группы
        $self->_save_hierarchical_multipliers($shard, [grep { $_->has_hierarchical_multipliers } @$adgroups]);

        # Сохраним дополнительные параметры группы
        $self->_save_adgroup_params($shard, $adgroups, 'create');

        # Сохраним теги
        $self->_save_adgroup_tags($shard, [grep { $_->has_tags } @$adgroups]);
       
        # Сохраним теги для отправки в БК
        $self->_save_adgroup_bs_tags($shard, [grep { $_->has_adgroup_bs_tags } @$adgroups]);
    };

    return;
}

=head2 update

Обновление в БД записей для соответствующих объектов (групп объявлений).

=cut

sub update {
    my ($self) = @_;

    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});
        $self->_update_in_shard($shard, $shard_items);
    }

    $_->reset_state() for @{$self->items};

    return;
}

sub _update_in_shard {
    my ($self, $shard, $adgroups) = @_;

    for my $adgroup (@$adgroups) {
        $adgroup->geo(0) if !defined $adgroup->{geo};
        if ($adgroup->do_update_status_post_moderate_unless_rejected) {
            $adgroup->is_status_post_moderate_changed(1);
            $adgroup->set_db_column_value('phrases', 'statusPostModerate', sprintf(
                "IF(statusPostModerate = 'Rejected', 'Rejected', %s)", sql_quote($adgroup->status_post_moderate)
            ), dont_quote => 1);
        }

        if (defined $adgroup->do_update_last_change) {
            if ($adgroup->do_update_last_change) {
                $adgroup->last_change(DateTime::Format::MySQL->format_datetime(now()));
                $adgroup->set_db_column_value('phrases', 'LastChange', 'NOW()', dont_quote => 1);
            } else {
                $adgroup->set_db_column_value('phrases', 'LastChange', 'LastChange', dont_quote => 1);
            }
            $adgroup->is_last_change_changed(1);
        }
    }

    do_in_transaction {
        # Сохраним минус-слова
        $self->_save_adgroup_minus_words($shard, [grep { $_->is_minus_words_changed } @$adgroups]);

        # Обновим таблицу `phrases`
        $self->_update_one_table_in_db(PPC(shard => $shard), phrases => 'pid', $adgroups);

        # Сохраним коэффициенты на группы
        $self->_save_hierarchical_multipliers($shard, [
            grep { $_->has_hierarchical_multipliers && $_->is_hierarchical_multipliers_changed } @$adgroups
        ]);

        # Сохраним дополнительные параметры группы
        $self->_save_adgroup_params($shard, $adgroups, 'update');

        # Сохраним теги
        $self->_save_adgroup_tags($shard, [grep { $_->has_tags } @$adgroups]);

        # Обработаем флаги
        $self->_do_update_banners($shard, $adgroups);
        $self->_do_clear_banners_moderation_flags($shard, $adgroups);
        $self->_do_schedule_forecast($shard, $adgroups);
    };

    return;
}

sub _save_hierarchical_multipliers {
    my ($self, $shard, $adgroups) = @_;

    return unless @$adgroups;

    # Сохранение единого набора коэффициентов
    for my $adgroup (@$adgroups) {
        save_hierarchical_multipliers($adgroup->campaign_id, $adgroup->id, $adgroup->hierarchical_multipliers);
    }

    return;
}

sub _save_adgroup_minus_words {
    my ($self, $shard, $adgroups) = @_;

    return unless @$adgroups;

    my $minus_words_hash2id = {};
    my %client_minus_words;

    for my $adgroup (@$adgroups) {
        unless (@{$adgroup->minus_words}) {
            # Минус-слов нет
            $adgroup->_mw_id(undef);
        } else {
            push @{$client_minus_words{$adgroup->client_id}}, $adgroup->minus_words;
        }
    }

    # Получим идентификаторы для минус-слов
    for my $client_id (keys %client_minus_words) {
        $minus_words_hash2id->{$client_id} = mass_save_minus_words($client_minus_words{$client_id}, $client_id);
    }

    for my $adgroup (grep { @{$_->minus_words} } @$adgroups) {
        $adgroup->_mw_id($minus_words_hash2id->{$adgroup->client_id}->{$adgroup->_minus_words_hash});
    }

    return;
}

sub _save_adgroup_params {
    my ($self, $shard, $adgroups, $action) = @_;

    $adgroups = [grep { $_->has_href_params && length($_->href_params // '') } @$adgroups] if $action eq 'create';
    $adgroups = [grep { $_->is_href_params_changed } @$adgroups] if $action eq 'update';
    return if !@$adgroups;

    my @params_to_insert = map { [$_->id, $_->href_params] } @$adgroups;

    do_mass_insert_sql(PPC(shard => $shard), q{
        INSERT INTO group_params (pid, href_params)
        VALUES %s ON DUPLICATE KEY UPDATE
            href_params = VALUES(href_params)
    }, \@params_to_insert, {sleep => 1, max_row_for_insert => 5000});

    return;
}

sub _save_adgroup_bs_tags {
    my ($self, $shard, $adgroups) = @_;

    my @bs_tags_to_insert = map { [$_->id, $_->adgroup_bs_tags->page_group_tags, $_->adgroup_bs_tags->target_tags] } grep { $_->adgroup_bs_tags } @$adgroups;
    do_mass_insert_sql(PPC(shard => $shard), q{
            INSERT INTO adgroup_bs_tags (pid, page_group_tags_json, target_tags_json)
            VALUES %s ON DUPLICATE KEY UPDATE
                page_group_tags_json = VALUES(page_group_tags_json),
                target_tags_json = VALUES(target_tags_json)
        }, \@bs_tags_to_insert, {sleep => 1, max_row_for_insert => 5000}) if @bs_tags_to_insert;

    return;
}

sub _save_adgroup_tags {
    my ($self, $shard, $adgroups) = @_;

    return unless @$adgroups;

    my %campaign_ids;
    my @adgroup_ids = map { $campaign_ids{$_->campaign_id} = 1; $_->id } @$adgroups;

    my $tag_name2id_by_cid = {};
    my $_get_tag_name2id_by_cid = sub {
        my ($cids) = @_;
        my $tag_rows = get_all_sql(PPC(shard => $shard), [q{SELECT tag_id, cid, tag_name FROM tag_campaign_list}, where => {cid => $cids}]);
        $tag_name2id_by_cid->{$_->{cid}}->{lc $_->{tag_name}} = $_->{tag_id} for @$tag_rows;
    };

    # Использование lc в коде ниже, для регистронезависимого сравнения - плохо работает, например, для турецкого яыка
    # С переходом на perl-5.16, лучше заменить на fc (http://perldoc.perl.org/functions/fc.html)

    # Добавим новые теги на кампании
    {;
        $_get_tag_name2id_by_cid->([keys %campaign_ids]);

        my (@tags_to_insert, %new_tag_ids_by_cid);
        for my $adgroup (@$adgroups) {
            for my $tag (grep { !$tag_name2id_by_cid->{$adgroup->campaign_id}->{lc $_->tag_name} } @{$adgroup->tags}) {
                push @tags_to_insert, [$adgroup->campaign_id, $tag->tag_name];
                push @{$new_tag_ids_by_cid{$adgroup->campaign_id}}, undef;
            }
        }

        $new_tag_ids_by_cid{$_} = get_new_id_multi('tag_id', scalar(@{$new_tag_ids_by_cid{$_}}), cid => $_) for keys %new_tag_ids_by_cid;
        do_mass_insert_sql(PPC(shard => $shard),
            'INSERT IGNORE INTO tag_campaign_list (tag_id, cid, tag_name) VALUES %s',
            [map { [shift(@{$new_tag_ids_by_cid{$_->[0]}}), @$_] } @tags_to_insert],
        ) if @tags_to_insert;

        $_get_tag_name2id_by_cid->([keys %new_tag_ids_by_cid]);
    }

    for my $adgroup (@$adgroups) {
        $_->id($tag_name2id_by_cid->{$adgroup->campaign_id}->{lc $_->tag_name}) for @{$adgroup->tags};
    }

    # Актуализируем список тегов на группах в БД
    {;
        my %adgroup_old_tag_ids;
        my $adgroup_old_tag_rows = get_all_sql(PPC(shard => $shard), [q{SELECT pid, tag_id FROM tag_group}, where => {pid => \@adgroup_ids}]);
        push @{$adgroup_old_tag_ids{$_->{pid}}}, $_->{tag_id} for @$adgroup_old_tag_rows;

        my (@rows_to_insert, @to_delete_where);
        for my $adgroup (@$adgroups) {
            my $old_tag_ids = $adgroup_old_tag_ids{$adgroup->id} // [];
            my $new_tag_ids = [map { $tag_name2id_by_cid->{$adgroup->campaign_id}->{lc $_->tag_name} } @{$adgroup->tags}];

            push @rows_to_insert, map { [$adgroup->id, $_] } @{ xminus($new_tag_ids, $old_tag_ids) };

            my $tag_ids_to_delete = xminus($old_tag_ids, $new_tag_ids);
            push @to_delete_where, _AND => {pid => $adgroup->id, tag_id => $tag_ids_to_delete} if @$tag_ids_to_delete;
        }

        do_in_transaction {
            do_mass_insert_sql(PPC(shard => $shard), 'INSERT IGNORE INTO tag_group (pid, tag_id) VALUES %s', \@rows_to_insert) if @rows_to_insert;
            do_delete_from_table(PPC(shard => $shard), 'tag_group', where => { _OR => \@to_delete_where }) if @to_delete_where;
        };
    }

    return;
}

sub _do_update_banners {
    my ($self, $shard, $adgroups) = @_;

    my (%adgroup_ids, %case_values, @banner_ids_to_remoderate);

    for my $adgroup (@$adgroups) {
        # Значение geoflag возвращается через аксессор флага
        if (defined (my $geoflag = $adgroup->do_update_banners_geoflag)) {
            $adgroup_ids{$adgroup->id} = 1;
            $case_values{geoflag}->{$adgroup->id} = sql_quote($geoflag);
            $case_values{opts}->{$adgroup->id} = Yandex::DBTools::sql_set_mod('opts', {geoflag => $geoflag});
        }

        # Отправка в БК
        if ($adgroup->do_bs_sync_banners) {
            $adgroup_ids{$adgroup->id} = 1;
        }

        if ($adgroup->do_remoderate_banner_ids) {
            push @banner_ids_to_remoderate, @{$adgroup->do_remoderate_banner_ids};
        }
    }
    if (%adgroup_ids) {
        do_update_table(PPC(shard => $shard), 'banners', {
            statusBsSynced => 'No',
            (map { $_ => sql_case(pid => $case_values{$_}, default__dont_quote => $_, dont_quote_value => 1) } keys %case_values),
            LastChange__dont_quote => 'LastChange',
        }, where => {
            pid => [keys %adgroup_ids],
        }, dont_quote => [keys %case_values]);
    }

    if (@banner_ids_to_remoderate) {
        do_update_table(PPC(shard => $shard), 'banners', {
            statusBsSynced => 'No',
            statusModerate => 'Ready',
            statusPostModerate__dont_quote => "IF(statusPostModerate = 'Rejected', 'Rejected', 'No')",
            LastChange__dont_quote => 'NOW()',
        }, where => {
            bid => \@banner_ids_to_remoderate,
            statusModerate__ne => 'New',
        });
        do_delete_from_table(PPC(shard => $shard), 'banners_minus_geo',
            where => {
                bid => \@banner_ids_to_remoderate,
                type => 'current',
            },
        );
    }

    return;
}

sub _do_clear_banners_moderation_flags {
    my ($self, $shard, $adgroups) = @_;

    my @adgroup_ids = map { $_->id } grep { $_->do_clear_banners_moderation_flags } @$adgroups;
    return if !@adgroup_ids;

    if (my @bids = PrimitivesIds::get_bids(pid => \@adgroup_ids)) {
        # Удаление флажков постмодерации и автомодерации
        do_delete_from_table(PPC(shard => $shard), 'post_moderate', where => {bid => \@bids});
        do_delete_from_table(PPC(shard => $shard), 'auto_moderate', where => {bid => \@bids});
    }

    return;
}

sub _do_schedule_forecast {
    my ($self, $shard, $adgroups) = @_;

    my %campaign_ids = map { $_->campaign_id => 1 } grep { $_->do_schedule_forecast } @$adgroups;
    Primitives::schedule_forecast_multi([keys %campaign_ids]) if %campaign_ids;

    return;
}

__PACKAGE__->meta->make_immutable;

1;
