package Direct::Model::AdGroupDynamic::Manager;

use Direct::Modern;
use Mouse;

extends 'Direct::Model::AdGroup::Manager';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Primitives qw//;

use Direct::Model::AdGroupDynamic;

has '+items' => (isa => 'ArrayRef[Direct::Model::AdGroupDynamic]');

=head2 _create_in_shard($shard, $adgroups)

Внутренний метод.
Создание в БД в определённом шарде записей для соответствующих объектов (групп динамических объявлений).

=cut

sub _create_in_shard {
    my ($self, $shard, $adgroups) = @_;

    my @columns = Direct::Model::AdGroupDynamic->get_db_columns_list('adgroups_dynamic');

    # Сохраним домены
    $self->_save_dynamic_adgroup_main_domains($shard, $adgroups);

    do_in_transaction {
        # Создадим базовые группы
        $self->SUPER::_create_in_shard($shard, $adgroups);

        $self->_insert_to_secondary_table_in_db(PPC(shard => $shard), 'adgroups_dynamic', phrases => 'pid', \@columns, $adgroups);

        _lowlevel_check_adgroups__inside_transaction($adgroups, 'create');
    };

    return;
}

=head2 _update_in_shard($shard, $adgroups)

Внутренний метод.
Обновление в БД в определённом шарде записей для соответствующих объектов (групп динамических объявлений).

=cut

sub _update_in_shard {
    my ($self, $shard, $adgroups) = @_;

    # Сохраним домены
    $self->_save_dynamic_adgroup_main_domains($shard, [grep { $_->is_main_domain_changed } @$adgroups]);

    do_in_transaction {
        # Обновим базовые группы
        $self->SUPER::_update_in_shard($shard, $adgroups);

        $self->_update_one_table_in_db(PPC(shard => $shard), adgroups_dynamic => 'pid', $adgroups);

        _lowlevel_check_adgroups__inside_transaction($adgroups, 'update');
    };

    return;
}

sub _save_dynamic_adgroup_main_domains {
    my ($self, $shard, $adgroups) = @_;

    $adgroups = [grep { $_->main_domain } @$adgroups];
    return unless @$adgroups;

    my @main_domains = map { $_->main_domain } @$adgroups;
    my $domain2id = Primitives::get_domain2domain_id(\@main_domains);

    do_mass_insert_sql(PPC(shard => $shard),
        "INSERT IGNORE INTO domains (domain_id, domain, reverse_domain) VALUES %s",
        [map { [$domain2id->{$_}, Primitives::_normalize_domain($_), Primitives::reverse_domain($_)] } @main_domains],
    );

    for my $adgroup (@$adgroups) {
        $adgroup->_main_domain_id($domain2id->{ $adgroup->main_domain });
    }

    return;
}

sub _lowlevel_check_adgroups__inside_transaction {
    my ($adgroups, $action) = @_;
    $adgroups = [grep { $_->is_main_domain_changed || $_->is_feed_id_changed } @$adgroups] if $action eq 'update';
    return if !@$adgroups;
    my $invalid_gids = get_one_column_sql(PPC(pid => [map { $_->id } @$adgroups]), [
        "SELECT pid, main_domain_id, feed_id FROM adgroups_dynamic", WHERE => {pid => SHARD_IDS},
        "HAVING IF(main_domain_id > 0, 1, 0) + IF(feed_id > 0, 1, 0) != 1",
    ]);
    croak "Failed constraint `main_domain_id OR feed_id` for dynamic adgroups: @{[join(',', @$invalid_gids)]}" if @$invalid_gids;
}

__PACKAGE__->meta->make_immutable;

1;
