package Direct::Model::AdGroupMobileContent::Manager;

use Direct::Modern;
use Mouse;

extends 'Direct::Model::AdGroup::Manager';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;

use List::MoreUtils qw/zip/;

require Direct::Model::AdGroupMobileContent;

has '+items' => (isa => 'ArrayRef[Direct::Model::AdGroupMobileContent]');

=head2 _create_in_shard($shard, $adgroups)

Внутренний метод.
Создание в БД в определённом шарде записей для соответствующих объектов (групп РМП объявлений).

=cut

sub _create_in_shard {
    my ($self, $shard, $adgroups) = @_;

    my @columns = Direct::Model::AdGroupMobileContent->get_db_columns_list('adgroups_mobile_content');

    # Создадим "мобильный контент"
    $self->_create_mobile_content($shard, $adgroups);

    do_in_transaction {
        # Создадим базовые группы
        $self->SUPER::_create_in_shard($shard, $adgroups);

        $self->_insert_to_secondary_table_in_db(PPC(shard => $shard), 'adgroups_mobile_content', phrases => 'pid', \@columns, $adgroups);
    };

    return;
}

=head2 _update_in_shard($shard, $adgroups)

Внутренний метод.
Обновление в БД в определённом шарде записей для соответствующих объектов (групп РМП объявлений).

=cut

sub _update_in_shard {
    my ($self, $shard, $adgroups) = @_;

    do_in_transaction {
        # Обновим базовые группы
        $self->SUPER::_update_in_shard($shard, $adgroups);

        $self->_update_one_table_in_db(PPC(shard => $shard), adgroups_mobile_content => 'pid', $adgroups);
    };

    return;
}

sub _create_mobile_content {
    my ($self, $shard, $adgroups) = @_;

    my @ukey_db_columns = qw/ClientID store_content_id store_country os_type content_type/;

    # [Шаг 1] Прочитаем существующие записи в `mobile_content` и определим данные для вставки
    my $existent_mob_content_by_ukey = get_hash_sql(PPC(shard => $shard), [
        sprintf('SELECT CONCAT_WS(":", %s) AS ukey, mobile_content_id FROM mobile_content', join(', ', @ukey_db_columns)),
        where => {
            _OR => [map { (_AND => { zip @ukey_db_columns, @{$_->mobile_content->get_uniq_key} }) } @$adgroups]
        }
    ]);
    
    my @mob_content_objs_to_insert;
    for my $adgroup (@$adgroups) {
        my $ukey = join ':', @{$adgroup->mobile_content->get_uniq_key};

        if (my $id = $existent_mob_content_by_ukey->{$ukey}) {            
            $adgroup->mobile_content->id($id);
            $adgroup->mobile_content_id($id);
        } else {
            push @mob_content_objs_to_insert, $adgroup->mobile_content;
        }
    }

    return unless @mob_content_objs_to_insert;

    # [Шаг 2] Вставим новые записи
    my $new_mob_content_ids = get_new_id_multi('mobile_content_id', scalar(@mob_content_objs_to_insert));
    do_mass_insert_sql(PPC(shard => $shard), sprintf(
        q{INSERT IGNORE INTO mobile_content (%s) VALUES %%s},
        join(', ', map { sql_quote_identifier($_) } ('mobile_content_id', @ukey_db_columns))
    ), [map { my $obj = $_; [
        shift @$new_mob_content_ids,
        map { $obj->get_db_column_value('mobile_content', $_, default_is_ok => 1, strict => 1) } @ukey_db_columns
    ] } @mob_content_objs_to_insert]);

    # [Шаг 3] Выберем вставленные записи
    my $inserted_mob_content_by_ukey = get_hash_sql(PPC(shard => $shard), [
        sprintf('SELECT CONCAT_WS(":", %s) AS ukey, mobile_content_id FROM mobile_content', join(', ', @ukey_db_columns)),
        where => {
            _OR => [map { (_AND => { zip @ukey_db_columns, @{$_->get_uniq_key} }) } @mob_content_objs_to_insert]
        }
    ]);
    $_->id($inserted_mob_content_by_ukey->{ join(':', @{$_->get_uniq_key}) }) for @mob_content_objs_to_insert;

    # Заполним mobile_content_id для групп
    $_->mobile_content_id($_->mobile_content->id) for @$adgroups;

    return;
}

__PACKAGE__->meta->make_immutable;

1;
