package Direct::Model::AdditionsItemCallout::Manager;

use Direct::Modern;
use Mouse;
use List::MoreUtils qw/part/;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ListUtils qw/xuniq/;

use Settings;
use Direct::Model::AdditionsItemCallout;

extends 'Yandex::ORM::Model::Manager::Base';

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::AdditionsItemCallout]',
);

=head2 create

Сохранение уточнений

=cut

sub create {
    my ($self) = @_;
    my @columns = Direct::Model::AdditionsItemCallout->get_db_columns_list('additions_item_callouts');

    for my $chunk (sharded_chunks(ClientID => $self->items, by => sub { $_->client_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{ClientID});

        # проставляем hash на всех item-ах в объекте (в ->calc_hash устанавливается только у объектов без хеша)
        for my $callout (@$shard_items) {
            $callout->calc_hash();
        }

        my $uniq_shard_items = [xuniq {$_->callout_text} @$shard_items];

        # находим по хешам, уточнения которые уже есть в базе, их обновляем только
        # ((ClientID = 123 AND hash = 546 AND callout_text = "...") OR (ClientID = ...) OR ...)
        my $where = sql_condition(PPC(shard => $shard), {
            _OR => [map {(_AND => {
                                    ClientID => $_->client_id,
                                    hash => $_->hash,
                                    callout_text => $_->callout_text,
                                  }
                        )} @$uniq_shard_items
                   ]
        });
        my $old_callouts_data = get_all_sql(PPC(shard => $shard), [
            "select ClientID, additions_item_id, callout_text, is_deleted
             from additions_item_callouts
            ", where => $where
        ]);
        my $old_callouts_db = {
            map {("$_->{ClientID}/$_->{callout_text}" => $_)}
            @$old_callouts_data
        };

        my ($new_callouts, $old_callouts) = part { ! $old_callouts_db->{$_->client_id . "/" . $_->callout_text} ? 0 : 1 } @$uniq_shard_items;

        my $new_callouts_ids = {};
        if ($new_callouts) {
            my $additions_item_ids = get_new_id_multi('additions_item_id', scalar(@$new_callouts));
            for my $new_callout (@$new_callouts) {
                $new_callout->id(pop @$additions_item_ids);
                $new_callout->status_moderate("Ready");
                $new_callout->set_last_change();
                $new_callout->calc_hash();
                $new_callout->set_datetimes();
                $new_callout->mark_new_item();
                $new_callout->is_deleted(0);

                $new_callouts_ids->{$new_callout->client_id . "/" . $new_callout->callout_text} = $new_callout->id;
            }
            
            my $inserted = $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'additions_item_callouts', \@columns, $new_callouts, (ignore_duplicates => 1));
            my $new_callouts_count = scalar @$new_callouts;

            # если количество новых коллаутов и количество новых записей в таблице не совпадают - нужно еще раз сходить в базу, чтобы получить нужные id
            if ($inserted != $new_callouts_count) {
                my $callouts_data = get_all_sql(PPC(shard => $shard), [
                    "select ClientID, additions_item_id, callout_text
                    from additions_item_callouts
                    ", where => $where
                ]);
                my $callouts_db = {
                    map {("$_->{ClientID}/$_->{callout_text}" => $_)}
                    @$callouts_data
                };

                # обновляем id для коллаутов, которые не получилось вставить в таблицу
                for my $new_callout (@$new_callouts) {
                    my $callout_id = $callouts_db->{$new_callout->client_id . "/" . $new_callout->callout_text}->{additions_item_id};
                    if ($new_callout->id != $callout_id) {
                        $new_callout->id($callout_id);
                    }
                }
            }
        }

        # проставляем id на всех item-ах в объекте
        for my $callout (@$shard_items) {
            unless ($callout->has_id) {
                my $additions_item_id = $new_callouts_ids->{$callout->client_id . "/" . $callout->callout_text} ||
                                        $old_callouts_db->{$callout->client_id . "/" . $callout->callout_text}->{additions_item_id};
                $callout->id($additions_item_id);
            }
        }

        if ($old_callouts) {
            for my $old_callout (@$old_callouts) {
                if ($old_callouts_db->{$old_callout->client_id . "/" . $old_callout->callout_text}->{is_deleted}) {
                    $old_callout->is_deleted(0);
                    $old_callout->set_last_change();
                    $old_callout->mark_new_item();
                }
            }

            $self->_update_one_table_in_db(PPC(shard => $shard), 'additions_item_callouts', 'additions_item_id', $old_callouts);
        }
    }

    return;
}

=head2 prepare_update

=cut

sub prepare_update {
    my ($self) = @_;

    for my $chunk (sharded_chunks(ClientID => $self->items, by => sub {$_->client_id})) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{ClientID});
        for my $item (@$shard_items) {
            if ($item->is_item_changed) {
                $item->set_last_change();
            }
        }
    }

    return;
}

=head2 update

Обновление уточнений

=cut

sub update {
    my ($self) = @_;

    $self->prepare_update();

    for my $chunk (sharded_chunks(ClientID => $self->items, by => sub {$_->client_id})) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{ClientID});

        $shard_items = [grep {$_->is_item_changed} xuniq {$_->id} @$shard_items];
        $self->_update_one_table_in_db(PPC(shard => $shard), additions_item_callouts => 'additions_item_id', $shard_items);
    }

    return;
}

=head2 delete

Помечаем уточнения удаленными

=cut

sub delete {
    my ($self) = @_;

    for my $chunk (sharded_chunks(ClientID => $self->items, by => sub { $_->client_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{ClientID});
        for my $item (@$shard_items) {
            $item->set_last_change();
            $item->is_deleted(1);
        }
        $shard_items = [xuniq {$_->id} @$shard_items];
        $self->_update_one_table_in_db(PPC(shard => $shard), additions_item_callouts => 'additions_item_id', $shard_items);
    }

    return;
}

1;
