package Direct::Model::Banner;

use Direct::Modern;
use Mouse;

use BannerFlags qw//;
use Lang::Guess qw/analyze_text_lang_with_context/;
use List::Util qw/first/;
use TextTools qw/space_cleaner get_num_array_by_str/;

use Direct::Model::Banner::LanguageUtils;

extends 'Yandex::ORM::Model::Base';
with 'Direct::Model::Role::Update';

__PACKAGE__->_setup(
    default_table  => 'banners',

    fields => [
        id                        => { type => 'Id', column => 'bid', primary_key => 1 },
        adgroup_id                => { type => 'Id', column => 'pid' },
        campaign_id               => { type => 'Id', column => 'cid' },
        banner_type               => { type => 'Enum', values => [qw/text dynamic mobile_content performance image_ad mcbanner cpm_banner/] },
        title                     => { type => 'Maybe[Str]', },
        title_extension           => { type => 'Maybe[Str]', },
        body                      => { type => 'Maybe[Str]', },
        href                      => { type => 'Maybe[Str]', length => 1024 },
        domain_id                 => { type => 'Maybe[Int]' },
        domain                    => { type => 'Maybe[Str]', length => 100, trigger => \&_on_domain_changed },
        reverse_domain            => { type => 'Maybe[Str]', length => 100 },
        bs_banner_id              => { type => 'Id', column => 'BannerID', default => 0 },
        status_bs_synced          => { type => 'Enum', values => [qw/Yes No Sending/], column => 'statusBsSynced', default => 'No', volatile => 1 },
        status_archived           => { type => 'Enum', values => [qw/Yes No/], column => 'statusArch', default => 'No' },
        status_show               => { type => 'Enum', values => [qw/Yes No/], column => 'statusShow', default => 'Yes' },
        status_active             => { type => 'Enum', values => [qw/Yes No/], column => 'statusActive', default => 'No' },
        status_moderate           => { type => 'Enum', values => [qw/Yes No Sent Sending Ready New/], column => 'statusModerate', default => 'New', volatile => 1 },
        status_post_moderate      => { type => 'Enum', values => [qw/New Yes No Ready Sent Rejected/], column => 'statusPostModerate', default => 'No', volatile => 1 },
        vcard_id                  => { type => 'Maybe[Id]' },
        vcard_status_moderate     => { type => 'Enum', values => [qw/New Sent Sending Ready Yes No/], column => 'phoneflag', default => 'New' },
        language                  => { type => 'Enum', values => [qw/Yes No ru en uk de be kk tr uz unknown vie es pt cs pl/], column => 'language', default => 'unknown' },
        sitelinks_set_id          => { type => 'Maybe[Id]' },
        sitelinks_status_moderate => { type => 'Enum', values => [qw/New Sent Sending Ready Yes No/], column => 'statusSitelinksModerate', default => 'New' },
        last_change               => { type => 'Timestamp', column => 'LastChange' },

        # поле - копия флага из opts
        geoflag                   => { type => 'Bool', default => 0, trigger => \&_on_opts_flag_changed },

        # Image-specific fields
        image_hash                => { type => 'Maybe[Str]', table => 'banner_images' },

        # Display URL
        display_href              => {
            type => 'Maybe[Str]',
            table => 'banner_display_hrefs',
        },
        display_href_status_moderate => {
            type => 'Maybe[Enum]',
            values => [qw/Sent Sending Ready Yes No/],
            default => 'Ready',
            table => 'banner_display_hrefs',
            column => 'statusModerate',
            alias => 'display_href_status_moderate',
        },
        tl_id                     => { type => 'Maybe[Id]', table => 'banner_turbolandings' },
        turbolanding_status_moderate => {
            type => 'Maybe[Enum]',
            values => [qw/Sent Sending Ready Yes No/], default => 'Ready',
            table => 'banner_turbolandings',
            column => 'statusModerate',
            alias => 'turbolanding_status_moderate',
           
        },
        
        # 
        turbolanding_href_params              => {
            type   => 'Maybe[Str]',
            table  => 'banner_turbolanding_params',
            column => 'href_params'
        },
        # Hidden fields
        _opts                     => { type => 'Str', column => 'opts', default => '' },
        _device_type_legacy       => { type => 'Enum', values => [qw/desktop mobile/], column => 'type', default => 'desktop' },
        _flags                    => { type => 'Maybe[Str]', column => 'flags' },
        _minus_geo => {
            type => 'Maybe[Str]',
            table => 'banners_minus_geo',
            column => 'minus_geo',
        },

    ],

    # Additional attributes
    additional => [
        is_mobile       => { type => 'Bool', track_changes => 1, trigger => \&_on_is_mobile_changed },
        filter_domain   => { type => 'Maybe[Str]' },
        client_id       => { type => 'Id' },
        disclaimer      => { type => 'Maybe[Str]' },
        # эксперимент на баннер, внутренее, используется в экспорте в БК и в юнит-тесте
        experiment      => { type => 'Maybe[Str]' },

        disable_display_href      => { type => 'Bool', default => 0, trigger => \&_on_opts_flag_changed },

        rejection_reason_ids              => { type => 'ArrayRef[Int]',  default => sub { [] } },
        vcard_rejection_reason_ids        => { type => 'ArrayRef[Int]',  default => sub { [] } },
        sitelinks_rejection_reason_ids    => { type => 'ArrayRef[Int]',  default => sub { [] } },
        display_href_rejection_reason_ids => { type => 'ArrayRef[Int]',  default => sub { [] } },
        turbolanding_rejection_reason_ids => { type => 'ArrayRef[Int]',  default => sub { [] } },
        export_version                    => { type => 'Maybe[Str]' },
        vcard_export_version              => { type => 'Maybe[Str]' },
        sitelinks_export_version          => { type => 'Maybe[Str]' },
        display_href_export_version       => { type => 'Maybe[Str]' },
        turbolanding_export_version       => { type => 'Maybe[Str]' },
    ],

    relations => [
        image         => { type => 'Direct::Model::BannerImage' },
        vcard         => { type => 'Direct::Model::VCard', key => 'vcard_id' },
        sitelinks_set => { type => 'Direct::Model::SitelinksSet', key => 'sitelinks_set_id' },
        additions_callouts => { type => 'ArrayRef[Direct::Model::AdditionsItemCallout]' },
        turbolanding => { type => 'Direct::Model::TurboLanding::Banner', key => 'tl_id' },
        

        adgroup       => { type => 'Direct::Model::AdGroup', key => 'adgroup_id' },
        campaign      => { type => 'Direct::Model::Campaign', key => 'campaign_id' },
    ],

    state_flags => [qw/
        update_last_change
        check_redirect
        update_filter_domain
        update_aggregator_domain
        bs_sync_adgroup
        moderate_adgroup
        update_adgroup_last_change
        set_adgroup_has_phraseid_href
        moderate_campaign_if_no_active_banners
        push_flags_to_moderation
        clear_moderation_flags
        delete_vcard_from_moderation
        delete_sitelinks_set_from_moderation
        delete_display_href_from_moderation
        delete_image_from_moderation
        delete_video_addition_from_moderation
        moderate_turbolanding
        delete_turbolanding
        delete_turbolanding_from_moderation
        refresh_metrika_counters
        update_status_post_moderate_unless_rejected
        resume_adgroup_autobudget_show
        schedule_forecast
        pre_moderate    post_moderate    auto_moderate
        dissociate_vcard
        copy_ctr
        additions_change
        remove_media_video
        remove_creative_video
    /],
);

around BUILDARGS => sub {
    my ($orig, $class) = (shift, shift);
    my %args = @_ == 1 && ref($_[0]) eq 'HASH' ? %{$_[0]} : @_;

    if (defined $args{_opts}) {
        $args{geoflag} = scalar($args{_opts} =~ /\bgeoflag\b/);
        $args{disable_display_href} = scalar($args{_opts} =~ /\bno_display_href\b/);
    }

    $args{is_mobile} = $args{_device_type_legacy} eq 'mobile' if defined $args{_device_type_legacy};

    $class->$orig(%args);
};

sub is_title_supported { 1 }
sub is_title_extension_supported { 0 }
sub is_body_supported { 1 }
sub is_href_supported { 1 }
sub is_vcard_supported { 1 }
sub is_sitelinks_set_supported { 1 }
sub is_callouts_supported { 0 }
sub is_image_supported { 1 }
sub is_imagead_supported { 0 }

sub is_display_href_supported { 0 }
sub is_disable_display_href_supported { 0 }
sub is_turbolanding_supported { 0 }
sub is_permalink_supported { 0 }
sub is_pixels_supported { 0 }

sub is_creative_always_moderated { 0 }

sub _on_opts_flag_changed {
    my ($self, $new) = @_;

    return if !$self->_constructed && $self->_has_opts;

    my @flags;
    push @flags, 'geoflag'          if $self->has_geoflag && $self->geoflag;
    push @flags, 'no_display_href'  if $self->has_disable_display_href && $self->disable_display_href;

    $self->_opts(join ',' => @flags);
    return;
}


sub _on_is_mobile_changed {
    my ($self, $new) = @_;
    $self->_device_type_legacy($new ? 'mobile' : 'desktop') if $self->_constructed || !$self->_has_device_type_legacy;
}

sub _on_domain_changed {
    my ($self, $new) = @_;
    $self->reverse_domain(defined($new) ? scalar reverse($new) : undef) if $self->_constructed || !$self->has_reverse_domain;
}

sub has_phraseid_param {
    my ($self) = @_;
    return undef unless defined $self->href;
    return $self->href =~ /\{phrase\_?id|param127|retargeting_id\}/i;
}

sub has_coef_goal_context_id_param {
    my ($self) = @_;
    return undef unless defined $self->href;
    return $self->href =~ /\{coef_goal_context_id\}/i;
}

=head2 flags

Получение/установка флагов у баннера

Параметры:
    (age => 18, med_services => 1) - установка значения новых флагов,
            -1 - удалить флаг
    отсутвие параметров - получение существующего набора флагов

Результат:
    (flag => 1, flag_2 => value) - хеш с полным набором текущих флагов у баннера

=cut

sub flags {
    my ($self) = shift;

    my $hash_flags = BannerFlags::get_banner_flags_as_hash($self->_has_flags ? $self->_flags : '', all_flags => 1);
    return %$hash_flags unless @_;

    my $is_flags_changed = 0;
    my %new_flags = @_;
    while (my ($flag, $value) = each %new_flags) {
        ($hash_flags, my $is_changed) = BannerFlags::set_flag($hash_flags, $flag, $value, 1);
        $is_flags_changed ||= $is_changed;
    }

    $self->_flags(BannerFlags::serialize_banner_flags_hash($hash_flags));
    $self->_is_flags_changed($is_flags_changed);

    return %$hash_flags;
}

sub has_flags { shift->_has_flags(@_) }
sub is_flags_changed { shift->_is_flags_changed(@_) }

sub detect_lang {
    my $self = shift;

    if (Direct::Model::Banner::LanguageUtils::is_empty_language($self->language)) {
        return analyze_text_lang_with_context($self->client_id, join ' ', $self->title, $self->title_extension // (), $self->body) || 'ru';
    } else {
        return $self->language;
    }
}

sub minus_geo
{
    my $self = shift;
    return get_num_array_by_str($self->_minus_geo);
}

after status_moderate => sub {
    my ($self, $status) = @_;
    if ($status && $status =~ /Ready|Sending|Sent/) {
        $self->_clear_minus_geo();
    }
};

=head2 is_title_changed_significantly

Есть ли значительные изменения в заголовке

=cut

sub is_title_changed_significantly
{
    my ($self) = @_;
    my $prev = $self->has_old ? $self->old->title : '';
    my $new = $self->title;
    $prev = space_cleaner($prev);
    $new = space_cleaner($new);
    return $new eq $prev ? 0 : 1;
}

=head2 is_title_extension_changed_significantly

Есть ли значительные изменения во втором заголовке

=cut

sub is_title_extension_changed_significantly
{
    my ($self) = @_;
    my $prev = $self->has_old ? ($self->old->title_extension // '') : '';
    my $new = $self->title_extension // '';
    $prev = space_cleaner($prev);
    $new = space_cleaner($new);
    return $new eq $prev ? 0 : 1;
}

=head2 is_body_changed_significantly

=cut

sub is_body_changed_significantly
{
    my ($self) = @_;
    my $prev = $self->has_old ? $self->old->body : '';
    my $new = $self->body;
    $prev = space_cleaner($prev);
    $new = space_cleaner($new);
    return $new eq $prev ? 0 : 1;
}

=head2 is_text_changed_significantly

=cut

sub is_text_changed_significantly
{
    my $self = shift;
    return $self->is_title_changed_significantly || $self->is_title_extension_changed_significantly || $self->is_body_changed_significantly;
}

1;
