package Direct::Model::Banner::Manager;

use Direct::Modern;
use Mouse;

extends 'Yandex::ORM::Model::Manager::Base';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::DateTime qw/now/;
use Yandex::ListUtils qw/xuniq chunks/;
use DateTime::Format::MySQL;
use List::Util qw/first any/;
use List::MoreUtils qw/all uniq/;

use URLDomain qw//;
use AggregatorDomains qw//;
use Moderate::ReModeration;
use BannerImages qw//;
use Primitives qw//;
use Models::Banner qw//;
use BS::History qw//;

use Direct::Model::Banner;
use Direct::BannersAdditions;
use Direct::Model::BannerCreative::Manager;
use Direct::Model::TurboLanding::Banner::Manager;
use Direct::Model::Banner::LanguageUtils;

require Direct::Model::Role::OnlyBannersResources;

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::Banner]',
);

# Соответствие типа баннера типу группы, где он может находится
my %ALLOWED_BANNER_TO_ADGROUP_TYPES = (
    text    => { base => 1 },
    dynamic => { dynamic => 1 },
    mobile_content => { mobile_content => 1 },
    performance => { performance => 1 },
    image_ad => { base => 1, mobile_content => 1 },
    mcbanner => { mcbanner => 1 },
    cpm_banner => { cpm_banner => 1, cpm_video => 1 },
    cpc_video => { base => 1, mobile_content => 1 },
);

our $MAX_ROWS_PER_DELETE = 100;

=head2 create

Создание в БД записей для соответствующих объектов (объявлений).

=cut

sub create {
    my ($self) = @_;

    # Выполним дополнительную проверку: не даем сохранить баннер в несоответствующую по типу группу
    {;
        my @adgroup_ids = keys %{ +{map { $_->adgroup_id => 1 } @{$self->items}} };
        my $adgroup_id2type = get_hash_sql(PPC(pid => \@adgroup_ids), ["SELECT pid, adgroup_type FROM phrases", where => {pid => SHARD_IDS}]);
        for my $banner (@{$self->items}) {
            my $adgroup_type = $adgroup_id2type->{$banner->adgroup_id};
            next if defined $adgroup_type && $ALLOWED_BANNER_TO_ADGROUP_TYPES{$banner->banner_type}->{$adgroup_type};
            croak sprintf(
                "banner #%d with type `%s` can be added only to adgroup with type: %s",
                $banner->id, $banner->banner_type,
                join(', ', keys %{ $ALLOWED_BANNER_TO_ADGROUP_TYPES{$banner->banner_type} }),
            );
        }
    }

    for my $chunk (sharded_chunks(bid => $self->items, by => sub { $_->id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{bid});
        $self->_create_in_shard($shard, $shard_items);
    }

    $_->reset_state() for @{$self->items};

    return;
}

sub _create_in_shard {
    my ($self, $shard, $banners) = @_;

    for my $banner (@$banners) {
        $banner->last_change(DateTime::Format::MySQL->format_datetime(now())) if !$banner->has_last_change;
    }

    my @columns = Direct::Model::Banner->get_db_columns_list('banners');

    # Создадим записи в таблице `banners`
    $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'banners', \@columns, $banners);

    # Обработаем флаги
    $self->_do_check_redirect($shard, $banners);
    $self->_do_update_filter_domain($shard, $banners);
    $self->_do_update_aggregator_domain($shard, $banners);
    $self->_do_update_adgroups($shard, $banners);
    $self->_do_moderate_campaign_if_no_active_banners($shard, $banners);
    $self->_do_insert_moderation_data($shard, $banners);
    $self->_do_copy_ctr($shard, $banners);

    $self->_do_update_images($shard, [grep { $_->has_image_hash && $_->image_hash } @$banners]);
    $self->_do_update_display_hrefs($shard, $banners);
    $self->_do_update_banners_additions($shard, $banners);
    $self->_do_update_minus_geo($shard, $banners);

    $self->_do_save_turbolandings($shard, $banners);
    $self->_do_refresh_metrika_counters($shard, $banners);
    $self->_do_update_permalinks($shard, $banners);

    # Добавим баннеры в очередь на заполнение языка
    Direct::Model::Banner::LanguageUtils::add_banners_to_fill_language_queue( [ map {$_->id} @$banners ] );

    return;
}

=head2 update(%options)

Обновление в БД записей для соответствующих объектов (объявлений).

=cut

sub update {
    my ($self, %options) = @_;

    for my $chunk (sharded_chunks(bid => $self->items, by => sub { $_->id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{bid});
        $self->_update_in_shard($shard, $shard_items, %options);
    }

    $_->reset_state() for @{$self->items};

    return;
}

sub _update_in_shard {
    my ($self, $shard, $banners, %options) = @_;

    my $banners_with_cleared_language = [];

    for my $banner (@$banners) {
        # Любое изменение языка баннера в перле - это всегда сброс
        if ($banner->is_language_changed) {
            push(@$banners_with_cleared_language, $banner->id);
        }

        if ($banner->do_update_status_post_moderate_unless_rejected) {
            $banner->is_status_post_moderate_changed(1);
            $banner->set_db_column_value('banners', 'statusPostModerate', sprintf(
                "IF(statusPostModerate = 'Rejected', 'Rejected', %s)", sql_quote($banner->status_post_moderate)
            ), dont_quote => 1);
        }

        if (defined $banner->do_update_last_change) {
            if ($banner->do_update_last_change) {
                $banner->last_change(DateTime::Format::MySQL->format_datetime(now()));
                $banner->set_db_column_value('banners', 'LastChange', 'NOW()', dont_quote => 1);
            } else {
                $banner->set_db_column_value('banners', 'LastChange', 'LastChange', dont_quote => 1);
            }
            $banner->is_last_change_changed(1);
        }

        # Тип баннера не можем быть изменён
        croak "banner #".$banner->id.": field `banner_type` cannot be changed" if $banner->is_banner_type_changed;
    }

    # Обновим таблицу `banners`
    
    $self->_update_one_table_in_db(PPC(shard => $shard), banners => 'bid', $banners, where => $options{where});

    $self->_do_update_adgroups($shard, $banners);

    # Добавим баннеры со сброшенными языками в очередь на заполнение языка
    Direct::Model::Banner::LanguageUtils::add_banners_to_fill_language_queue($banners_with_cleared_language);

    # Если баннеры с ролью OnlyBannersResources - остальные изменения в БД не делаем
    return if all {$_->does(Direct::Model::Role::OnlyBannersResources->meta)} @{$banners};

    # Обработаем флаги
    $self->_do_check_redirect($shard, $banners);
    $self->_do_update_filter_domain($shard, $banners);
    $self->_do_update_aggregator_domain($shard, $banners);
    $self->_do_moderate_campaign_if_no_active_banners($shard, $banners);
    $self->_do_insert_moderation_data($shard, $banners);
    $self->_do_clear_moderation_data($shard, $banners);
    $self->_do_schedule_forecast($shard, $banners);
    $self->_do_dissociate_vcards($shard, $banners);

    $self->_do_update_images($shard, $banners);
    $self->_do_update_display_hrefs($shard, $banners);
    $self->_do_update_banners_additions($shard, $banners);
    $self->_do_update_minus_geo($shard, $banners);

    $self->_do_save_turbolandings($shard, $banners);
    $self->_do_refresh_metrika_counters($shard, $banners);
    $self->_do_update_permalinks($shard, $banners);

    return;
}

=head2 delete

Удаление из БД записей для соответствующих объектов (объявлений).

=cut

sub delete {
    my ($self) = @_;

    for my $chunk (sharded_chunks(bid => $self->items, by => sub { $_->id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{bid});
        $self->_delete_in_shard($shard, $shard_items);
    }

    $_->reset_state() for @{$self->items};

    return;
}

sub _delete_in_shard {
    my ($self, $shard, $banners) = @_;

    # ВНИМАНИЕ!
    # Данный метод подходит только для использования внутри перфоманса.
    # Для остальных типов баннеров необходима реализация удаления из модерации/редиректов и прочего.

    my @banner_ids = map { $_->id } @$banners;

    do_delete_from_table(PPC(shard => $shard), 'banners', where => {bid => \@banner_ids});
    do_mass_insert_sql(PPC(shard => $shard),
        "INSERT IGNORE INTO deleted_banners (bid, deleteTime) VALUES %s",
        [map { [$_->id, 'NOW()'] } @$banners],
        {sleep => 1, max_row_for_insert => 1000, dont_quote => 1},
    );
    delete_shard(bid => \@banner_ids);

    # Обработаем флаги
    $self->_do_update_adgroups($shard, $banners);

    $self->_do_refresh_metrika_counters($shard, $banners);

    return;
}

sub _do_check_redirect {
    my ($self, $shard, $banners) = @_;

    my @data_to_insert = map { [$_->id, 'banner'] } grep { $_->do_check_redirect } @$banners;
    do_mass_insert_sql(PPC(shard => $shard), "REPLACE INTO redirect_check_queue (object_id, object_type) VALUES %s", \@data_to_insert, {}) if @data_to_insert;

    return;
}

sub _do_update_filter_domain {
    my ($self, $shard, $banners) = @_;

    URLDomain::update_filter_domain($_->id, $_->domain) for grep { $_->do_update_filter_domain } @$banners;

    return;
}

sub _do_update_aggregator_domain {
    my ($self, $shard, $banners) = @_;

    my %data_for_update = map { $_->id => $_->href } grep { $_->do_update_aggregator_domain } @$banners;
    AggregatorDomains::update_aggregator_domains(\%data_for_update);

    return;
}

sub _do_update_adgroups {
    my ($self, $shard, $banners) = @_;

    my (%adgroup_ids, %case_values, @params_to_insert);
    for my $banner (grep {
        $_->do_update_adgroup_last_change ||
        $_->do_moderate_adgroup ||
        $_->do_bs_sync_adgroup ||
        $_->do_set_adgroup_has_phraseid_href ||
        $_->do_resume_adgroup_autobudget_show
    } @$banners) {
        $adgroup_ids{$banner->adgroup_id} = 1;

        # Принудительное изменение/сохранение времени модификации группы
        $case_values{LastChange}->{$banner->adgroup_id} = $banner->do_update_adgroup_last_change ? 'NOW()' : 'LastChange';

        $case_values{statusBsSynced}->{$banner->adgroup_id} = sql_quote('No') if $banner->do_bs_sync_adgroup || $banner->do_set_adgroup_has_phraseid_href;
        $case_values{statusAutobudgetShow}->{$banner->adgroup_id} = sql_quote('Yes') if $banner->do_resume_adgroup_autobudget_show;

        if ($banner->do_moderate_adgroup) {
            $case_values{statusModerate}->{$banner->adgroup_id} = sql_quote('Ready');
            $case_values{statusPostModerate}->{$banner->adgroup_id} = "IF(statusPostModerate = 'Rejected', 'Rejected', 'No')";
        }

        push @params_to_insert, [$banner->adgroup_id, 1] if $banner->do_set_adgroup_has_phraseid_href;
    }

    do_update_table(PPC(shard => $shard), 'phrases', {
        (map { $_ => sql_case(pid => $case_values{$_}, default__dont_quote => $_, dont_quote_value => 1) } keys %case_values)
    }, where => {pid => [keys %adgroup_ids]}, dont_quote => [keys %case_values]) if %adgroup_ids;

    do_mass_insert_sql(PPC(shard => $shard), q{
        INSERT INTO group_params (pid, has_phraseid_href)
        VALUES %s ON DUPLICATE KEY UPDATE
            has_phraseid_href = IF(VALUES(has_phraseid_href) > 0, VALUES(has_phraseid_href), has_phraseid_href)
    }, \@params_to_insert, {sleep => 1, max_row_for_insert => 5000}) if @params_to_insert;

    return;
}

sub _do_moderate_campaign_if_no_active_banners {
    my ($self, $shard, $banners) = @_;

    $banners = [grep { $_->do_moderate_campaign_if_no_active_banners } @$banners];

    my $cids_has_active_banners = {map { ($_->campaign_id => undef) } @$banners};
    if (my @cids = keys %$cids_has_active_banners) {
        my $type_by_cid = Campaign::Types::get_camp_type_multi(cid => \@cids);
        $cids_has_active_banners = Models::Banner::mass_has_camps_active_banners(\@cids, $type_by_cid);

        my @cids_to_moderate = grep { !$cids_has_active_banners->{$_} } @cids;
        do_update_table(PPC(shard => $shard), 'campaigns', {statusModerate => 'Ready'}, where => {statusModerate => [qw/New No/], cid => \@cids_to_moderate});
    }

    return;
}

sub _do_insert_moderation_data {
    my ($self, $shard, $banners) = @_;

    my (%pre_moderate, %post_moderate, %auto_moderate, %push_flags);
    for my $banner (@$banners) {
        $pre_moderate{$banner->id} = 1 if $banner->do_pre_moderate;
        $post_moderate{$banner->id} = 1 if $banner->do_post_moderate;
        $auto_moderate{$banner->id} = 1 if $banner->do_auto_moderate;

        $push_flags{$banner->id} = $banner if $banner->do_push_flags_to_moderation;
    }

    if (%pre_moderate) {
        Moderate::ReModeration->set_pre_moderation_flag([keys %pre_moderate]);
    }

    if (%auto_moderate) {
        Moderate::ReModeration->set_auto_moderation_flag([keys %auto_moderate]);
    }

    do_mass_insert_sql(PPC(shard => $shard), 'INSERT IGNORE post_moderate(bid) VALUES %s', [map { [$_] } keys %post_moderate]) if %post_moderate;
}

sub _do_clear_moderation_data {
    my ($self, $shard, $banners) = @_;

    if (my @bids = map { $_->id } grep { $_->do_clear_moderation_flags } @$banners) {
        # Удаление флажков постмодерации и автомодерации
        do_delete_from_table(PPC(shard => $shard), 'post_moderate', where => {bid => \@bids});
        do_delete_from_table(PPC(shard => $shard), 'auto_moderate', where => {bid => \@bids});

        # Удаление данных из mod_edit о том, что баннер редактировался модератором
        do_delete_from_table(PPC(shard => $shard), 'mod_edit', where => {type => 'banner', id => \@bids});
    }

    # Удаление непромодерированных визиток/сайтлинк-сетов из модерации + их версий
    my (%vcard_bids_by_cid, %sitelinks_set_bids_by_cid, %display_href_bids_by_cid, %image_bids_by_cid, %video_additions_by_cid, %turbolandings_by_cid);
    for my $banner (@$banners) {
        my $cid = $banner->campaign_id;
        push @{ $vcard_bids_by_cid{$cid}         }, $banner->id if $banner->do_delete_vcard_from_moderation;
        push @{ $sitelinks_set_bids_by_cid{$cid} }, $banner->id if $banner->do_delete_sitelinks_set_from_moderation;
        push @{ $display_href_bids_by_cid{$cid}  }, $banner->id if $banner->do_delete_display_href_from_moderation;
        push @{ $image_bids_by_cid{$cid}         }, $banner->id if $banner->do_delete_image_from_moderation;
        push @{ $video_additions_by_cid{$cid}    }, $banner->id if $banner->do_delete_video_addition_from_moderation;
        push @{ $turbolandings_by_cid{$cid}      }, $banner->id if $banner->do_delete_turbolanding_from_moderation;
    }

    if (%vcard_bids_by_cid) {
        my @vcard_bids = map { @$_ } values %vcard_bids_by_cid;
        do_delete_from_table(PPC(shard => $shard), 'mod_object_version', where => {obj_type => 'contactinfo', obj_id => \@vcard_bids});
    }
    if (%sitelinks_set_bids_by_cid) {
        my @sitelinks_set_bids = map { @$_ } values %sitelinks_set_bids_by_cid;
        do_delete_from_table(PPC(shard => $shard), 'mod_object_version', where => {obj_type => 'sitelinks_set', obj_id => \@sitelinks_set_bids});
    }
    if (%display_href_bids_by_cid) {
        my @display_href_bids = map { @$_ } values %display_href_bids_by_cid;
        do_delete_from_table(PPC(shard => $shard), 'mod_object_version', where => {obj_type => 'display_href', obj_id => \@display_href_bids});
    }
    if (%image_bids_by_cid) {
        my @image_bids = map { @$_ } values %image_bids_by_cid;
        do_delete_from_table(PPC(shard => $shard), 'mod_object_version', where => {obj_type => 'image', obj_id => \@image_bids});
    }
    if (%video_additions_by_cid) {
        my @video_bids = map { @$_ } values %video_additions_by_cid;
        do_delete_from_table(PPC(shard => $shard), 'mod_object_version', where => {obj_type => 'video_addition', obj_id => \@video_bids});
    }
    if (%turbolandings_by_cid) {
        my @turbolanding_bids = map { @$_ } values %turbolandings_by_cid;
        do_delete_from_table(PPC(shard => $shard), 'mod_object_version', where => {obj_type => 'turbolanding', obj_id => \@turbolanding_bids});
    }

    return;
}

sub _do_update_images {
    my ($self, $shard, $banners) = @_;

    # Сохраним картинки на баннер (добавленные или измененные)
    my (@images_to_assign, @image_bids_to_remove);
    for my $banner (@$banners) {
        next if !$banner->has_image_hash;

        if (defined $banner->image_hash && ($banner->is_image_hash_changed || $banner->image->is_changed)) {
            my $image_to_assign = {
                bid => $banner->id,
                cid => $banner->campaign_id,
                image_hash => $banner->image_hash,
            };
            $image_to_assign->{statusModerate} = $banner->image->status_moderate  if $banner->image->has_status_moderate;
            push @images_to_assign, $image_to_assign;
        }

        push @image_bids_to_remove, $banner->id if !defined $banner->image_hash && $banner->is_image_hash_changed;
    }

    BannerImages::save_banner_images(\@images_to_assign) if @images_to_assign;
    BannerImages::mass_banner_remove_image(\@image_bids_to_remove) if @image_bids_to_remove;

    return;
}

sub _do_update_display_hrefs {
    my ($self, $shard, $banners) = @_;

    my (@hrefs_to_insert, @bids_to_remove, @banners_to_update);
    for my $banner (@$banners) {
        my $old_has_display_href = $banner->has_old && $banner->old->has_display_href && $banner->old->display_href;
        my $new_has_display_href = $banner->has_display_href && $banner->display_href;
        my $new_deletes_display_href = $banner->has_display_href && !$banner->display_href;
        if (!$old_has_display_href && $new_has_display_href) {
            push @hrefs_to_insert, [
                $banner->id,
                $banner->display_href,
                $banner->has_display_href_status_moderate && $banner->display_href_status_moderate || 'Ready',
            ];
        } elsif ($new_deletes_display_href) {
            push @bids_to_remove, $banner->id;
        } elsif ($old_has_display_href && $new_has_display_href) {
            push @banners_to_update, $banner;
        }
    }

    if (@hrefs_to_insert) {
        do_mass_insert_sql(PPC(shard => $shard),
            'INSERT INTO banner_display_hrefs (bid, display_href, statusModerate) VALUES %s
            ON DUPLICATE KEY UPDATE
                display_href = VALUES(display_href),
                statusModerate = VALUES(statusModerate)',
            \@hrefs_to_insert,
        );
    }

    if (@bids_to_remove) {
        do_delete_from_table(PPC(shard => $shard), 'banner_display_hrefs',
            where => { bid => \@bids_to_remove},
        );
    }

    if (@banners_to_update) {
        $self->_update_one_table_in_db(PPC(shard => $shard), banner_display_hrefs => 'bid', $banners);
    }

    return;
}

sub _do_schedule_forecast {
    my ($self, $shard, $banners) = @_;

    my %campaign_ids = map { $_->campaign_id => 1 } grep { $_->do_schedule_forecast } @$banners;
    Primitives::schedule_forecast_multi([keys %campaign_ids]) if %campaign_ids;

    return;
}

sub _do_dissociate_vcards {
    my ($self, $shard, $banners) = @_;

    my @vcard_ids_to_dissociate = map { $_->do_dissociate_vcard } grep { $_->do_dissociate_vcard } @$banners;
    if (@vcard_ids_to_dissociate) {
        do_update_table(PPC(shard => $shard), 'vcards', {
            LastChange__dont_quote => 'LastChange',
            last_dissociation__dont_quote => 'NOW()',
        }, where => {vcard_id => \@vcard_ids_to_dissociate});
    }

    return;
}

sub _do_copy_ctr {
    my ($self, $shard, $banners) = @_;

    my $kw_id2data = {};
    for my $banner (@$banners) {
        next unless $banner->do_copy_ctr && $banner->do_copy_ctr->{keywords};
        for my $kw_id (keys %{$banner->do_copy_ctr->{keywords}}) {
            $kw_id2data->{$kw_id}->{phraseIdHistory} //= $banner->do_copy_ctr->{keywords}->{$kw_id};
            push @{$kw_id2data->{$kw_id}->{banners}}, $banner;
        }
    }

    my @insert_phraseid_hist;
    while (my ($kw_id, $data) = each %$kw_id2data) {
        my $history = {BS::History::parse_bids_history($data->{phraseIdHistory})};

        # TODO: По хорошему здесь надо выбирать баннер с лучшим CTR. Но пока решили не усложнять.
        my $bs_banner_id = first { $_ } map { @$_ } values %{$history->{banners}};
        $history->{banners} = {map { $_->id => $bs_banner_id } @{$data->{banners}}};

        my $cid = $data->{banners}->[0]->campaign_id;
        if (my $serialized_history = BS::History::serialize_bids_history($history)) {
            push @insert_phraseid_hist, [$cid, $kw_id, $serialized_history];
        }
    }

    do_mass_insert_sql(PPC(shard => $shard),
        "INSERT IGNORE INTO `bids_phraseid_history` (`cid`, `id`, `phraseIdHistory`) VALUES %s",
        \@insert_phraseid_hist,
    );

    return;
}

=head2 _do_update_banners_additions

  Привязка/отвязка дополнений на баннер

=cut

sub _do_update_banners_additions {
    my ($self, $shard, $banners) = @_;

    my $items_callouts = [];

    for my $banner (@$banners) {
        next unless $banner->do_additions_change;
        if ($banner->has_additions_callouts) {
            push @$items_callouts, @{$banner->additions_callouts};
        }
    }

    my $id_by_text = {};
    if (@$items_callouts) {
        $items_callouts = [xuniq {$_->callout_text} @$items_callouts];
        my $banners_additions = Direct::BannersAdditions->new(items_callouts => $items_callouts);
        $banners_additions->save();

        $id_by_text = {
            map {$_->callout_text => $_->id}
            grep {$_->has_id}
            @$items_callouts
        };
    }

    my $link_data = {};

    for my $banner (@$banners) {
        next unless $banner->do_additions_change;

        # проставляем id у дополнений, которые повторяются на разных баннерах, а сохраняли мы только уникальные
        for my $item (@{$banner->additions_callouts}) {
            if (! $item->has_id && $id_by_text->{$item->callout_text}) {
                $item->id($id_by_text->{$item->callout_text});
            }
        }

        if ($banner->has_additions_callouts) {
            $link_data->{$banner->id} = [map {$_->id} @{$banner->additions_callouts}];
        } else {
            # на баннере нет дополнений, но дополнения менялись, отвязываем
            $link_data->{$banner->id} = [];
        }
    }

    if (%$link_data) {
        Direct::BannersAdditions::link_to_banners($link_data, "callout")
    }

    return;
}

sub _do_update_minus_geo
{
    my ($self, $shard, $banners) = @_;
    my (@geo_to_insert, @geo_to_remove);
    for my $banner (@$banners) {
        if ($banner->_has_minus_geo && $banner->_minus_geo) {
            push @geo_to_insert, [ $banner->id, $banner->_minus_geo ];
        }
        else {
            push @geo_to_remove, $banner->id;
        }
    }

    if (@geo_to_insert) {
        do_mass_insert_sql(PPC(shard => $shard), 'REPLACE INTO banners_minus_geo (bid, minus_geo) VALUES %s',
            \@geo_to_insert
        );
    }
    if (@geo_to_remove) {
        do_delete_from_table(PPC(shard => $shard), 'banners_minus_geo',
            where => {
                bid => \@geo_to_remove,
                type => 'current',
            },
        );
    }
}

sub _do_update_video_resources {
    my ($self, $shard, $banners) = @_;
    
    my (@video_resources_to_insert, @video_resources_to_remove);
    for my $banner (@$banners) {
        if ($banner->do_remove_media_video) {
            push @video_resources_to_remove, $banner->id;
        }
        if ($banner->has_old
            && $banner->old->has_creative && $banner->old->creative->has_creative && $banner->old->creative->creative->resource_type eq 'media' 
            && !$banner->has_creative)
        {
            # старый креатив удален
            push @video_resources_to_remove, $banner->id;
            next;
        }
        next unless $banner->has_creative && $banner->creative->has_creative;
        my $video = $banner->creative->creative;
        die "invalid video addition" unless ref $video eq 'Direct::Model::VideoAddition';
        if ($video->resource_type eq 'media') {
            my $prev_video_id =
                $banner->has_old && $banner->old->has_creative && $banner->old->creative->has_creative
                    && $banner->old->creative->creative->resource_type eq 'media'
                ? $banner->old->creative->creative->id
                : 0;
            if ($video->id) {
                if ($video->id != $prev_video_id) {
                    push @video_resources_to_insert, [ $banner->id, $video->_used_resources ];
                }
            }
            else {
                push @video_resources_to_remove, $banner->id;
            }
        }
        else {
            if ($banner->has_old && $banner->old->has_creative && $banner->old->creative->has_creative
                && $banner->old->creative->creative->resource_type eq 'media')
            {
                # старое дополнение заменили на новое - удаляем старое
                push @video_resources_to_remove, $banner->id;
            }
        }
    }

    if (@video_resources_to_insert) {
        my $ids = get_new_id_multi('resource_id', scalar @video_resources_to_insert);
        do_mass_insert_sql(PPC(shard => $shard),
            'INSERT INTO banner_resources (resource_id, bid, used_resources) VALUES %s
            ON DUPLICATE KEY UPDATE
                used_resources = VALUES(used_resources)',
            [ map { [shift(@$ids), @$_] } @video_resources_to_insert ],
        );
    }

    if (@video_resources_to_remove) {
        do_delete_from_table(PPC(shard => $shard), 'banner_resources',
            where => { bid => \@video_resources_to_remove },
        );
    }

}

sub _save_banner_creatives
{
    my ($self, $shard, $banners) = @_;
    
    my (@creatives_to_delete, @creatives_to_create, @creatives_to_update);
    for my $banner (@$banners) {
        if ($banner->do_remove_creative_video) {
            push @creatives_to_delete, $banner->old->creative;
        }
        if ($banner->has_old && $banner->old->has_creative && !$banner->has_creative ) {
            # креатив удалился
            push @creatives_to_delete, $banner->old->creative;
        }
        next unless $banner->has_creative && $banner->creative->creative->resource_type eq 'creative';
        if (!$banner->creative->has_banner_id || $banner->creative->banner_id == 0) {
            $banner->creative->banner_id($banner->id);
            $banner->creative->adgroup_id($banner->adgroup_id);
            $banner->creative->campaign_id($banner->campaign_id);
        }
        $banner->creative->creative_id($banner->creative->creative->id);
        if ($banner->creative->has_id && $banner->creative->id > 0) {
            push @creatives_to_update, $banner->creative;
        }
        else {
            push @creatives_to_create, $banner->creative;
        }
    }

    Direct::Model::BannerCreative::Manager->new(items => \@creatives_to_create)->create();
    Direct::Model::BannerCreative::Manager->new(items => \@creatives_to_update)->update();
    Direct::Model::BannerCreative::Manager->new(items => \@creatives_to_delete)->delete();
}

sub _do_moderate_creatives
{
    my ($self, $shard, $banners) = @_;

    for my $banner (@$banners) {
        next unless $banner->has_creative && $banner->creative->creative->resource_type eq 'creative';
        next if $banner->is_creative_always_moderated;
        if ($banner->do_moderate_creative) {
            $banner->creative->status_moderate('Ready');
        }
    }
}

sub _do_save_turbolandings {
        my ($self, $shard, $banners) = @_;

        my (@to_delete, @to_create, @to_update);
        my $href_params = {
                to_delete => [], to_create => [], to_update => {}
        };
        my $has_banners_with_turbolanding_support;

        for my $banner (@$banners) {
                next unless $banner->is_turbolanding_supported();
                $has_banners_with_turbolanding_support ||= 1;

                $self->_define_kind_of_change_turbolanding_href_params ($banner, $href_params) if $banner->is_turbolanding_href_params_changed();

                $self->_set_status_moderate($banner => $banner->turbolanding) if ($banner->has_turbolanding && $banner->do_moderate_turbolanding);

                if ($banner->do_delete_turbolanding) {
                    $banner->do_refresh_metrika_counters(1);
                    push @to_delete, $banner->turbolanding;
                }
                elsif ($banner->has_old) {
                    if ($banner->old->has_turbolanding) {
                        if ($banner->turbolanding->id != $banner->old->turbolanding->id) {
                            #Если для баннера задан турболендинг с другим id - удаляем старую привязку и добавляем новую
                            push @to_delete, $banner->old->turbolanding;
                            $self->_add_turbolanding(\@to_create => ($banner, $banner->turbolanding));
                        }
                        elsif ($banner->do_moderate_turbolanding) {
                            push @to_update, $banner->turbolanding;
                        }
                    }
                    elsif($banner->has_turbolanding) {
                        $self->_add_turbolanding(\@to_create => ($banner, $banner->turbolanding));
                    }
                }
                elsif ($banner->has_turbolanding) {
                    $self->_add_turbolanding(\@to_create => ($banner, $banner->turbolanding));
                }
        }
        
        if ($has_banners_with_turbolanding_support) {
            Direct::Model::TurboLanding::Banner::Manager->new(items => \@to_delete)->delete() if \@to_delete;
            Direct::Model::TurboLanding::Banner::Manager->new(items => \@to_create)->create() if \@to_create;
            Direct::Model::TurboLanding::Banner::Manager->new(items => \@to_update)->update() if \@to_update;

            $self->_save_turbolanding_href_params($shard, $href_params);
        }

        return;
}

sub _define_kind_of_change_turbolanding_href_params {
        my ($self, $banner, $lists) = @_;
        my ($to_update, $to_delete, $to_create) = @$lists{qw/to_update to_delete to_create/};

        my $new_value = $banner->has_turbolanding_href_params ? $banner->turbolanding_href_params : '';
        my $old_value = $banner->has_old && $banner->old->has_turbolanding_href_params ?  $banner->old->turbolanding_href_params : '';

        if (!defined $new_value ||  $new_value eq '') {
                push @$to_delete, $banner->id if defined $old_value && $old_value gt '';
        }
        elsif(!defined $old_value || $old_value eq '') {
                push @$to_create, [$banner->id, $new_value];
        }
        else {
                $to_update->{$banner->id} = {href_params => $new_value} if $new_value ne $old_value;
        }

        return;
}

sub _set_status_moderate {
        my ($self, $banner, $item) = @_;

        my $status_moderate = $banner->status_moderate() eq 'New' ? 'New' : 'Ready';
        $item->status_moderate($status_moderate);

        return;
}

sub _add_turbolanding {
    my ($self, $list, $banner, $landing) = @_;

    $landing->bid($banner->id);

    $self->_set_status_moderate($banner => $landing);
    push  @$list, $landing;

    $banner->do_refresh_metrika_counters(1);

    return;
}

sub _do_refresh_metrika_counters {
        my ($self, $shard, $banners) = @_;

        my @affected_banners = grep {$_->do_refresh_metrika_counters} @$banners;
        my $turbolandings_by_bid;
        my $cid_by_bid;
        my $turbolanding_counters;

        my $has_banners_with_turbolanding_support = 0;
        foreach my $banner (@affected_banners) {
                next unless $banner->is_turbolanding_supported();
                $has_banners_with_turbolanding_support ||= 1;

                my $cid = $banner->campaign_id;
                $cid_by_bid->{$banner->id} = $cid;

                $turbolandings_by_bid->{$banner->id}->{$banner->turbolanding->id} = 1
                        if ($banner->has_turbolanding && !$banner->do_delete_turbolanding);

                _for_each_sitelink_with_turbolanding ( $banner,
                        sub { my ($bnr, $sl) = @_;
                             $turbolandings_by_bid->{$bnr->id}->{$sl->tl_id} = 1; }
                );

                my $tl_diff = _get_banner_turbolandings_diff($banner);
                $turbolanding_counters->{$cid} //= {} if (@{$tl_diff->{added}} || @{$tl_diff->{deleted}});
                $turbolanding_counters->{$cid}->{$_}++ foreach @{$tl_diff->{added}};
                $turbolanding_counters->{$cid}->{$_}-- foreach @{$tl_diff->{deleted}};
        }

        if ($has_banners_with_turbolanding_support) {
            my $counters_and_goals_by_tl_id = Direct::TurboLandings::get_metrika_counters_and_goals_by_tl_id(
                PPC(shard => $shard),
                [uniq map {keys %$_} grep {ref $_} values %$turbolanding_counters]
            );

            my $desired_counters_by_bid = $self->_get_counters($shard, $turbolandings_by_bid, $counters_and_goals_by_tl_id);

            $self->_refresh_counters($shard, $desired_counters_by_bid, $cid_by_bid);
            $self->_refresh_goals($shard, $turbolanding_counters, $counters_and_goals_by_tl_id);
        }

        return;
}

sub _get_banner_turbolandings_diff {
        my ($banner) = @_;

        my (%added, %deleted);

        if (!$banner->has_old) {
                #Если баннер новый - соберем все его турболендинги в "добавленные"
                if ($banner->has_turbolanding && !$banner->do_delete_turbolanding){
                        $added{$banner->turbolanding->id} //= 1;
                }
                _for_each_sitelink_with_turbolanding ( $banner,
                        sub { my ($bnr, $sl) = @_;
                                $added{$sl->tl_id} = 1;
                        }
                );
         }
        else {
                my (%alive, %linked_turbolandings);
                #Если баннер не новый и у него уже был турболендинг - занесем его прилинкованные.
                #Если после обработки турболендингов баннера и сайтлинков прилинкованный турболендинг не появится в %alive - он будет удален
                if ($banner->old->has_turbolanding) {
                    $linked_turbolandings{$banner->old->turbolanding->id} = 1;
                }

                if ($banner->has_turbolanding && !$banner->do_delete_turbolanding) {
                    my $tl_id = $banner->turbolanding->id;
                    $alive{$tl_id} = 1;
                    #Если текущий турболендинг баннера отсутствует в предыдущей версии - занесем его в "добавленные" и,
                    #чтобы не посчитать дважды при обработке сайтлинков - в прилинкованные
                    unless(exists $linked_turbolandings{$tl_id}){
                        $added{$tl_id} //= 1;
                        $linked_turbolandings{$tl_id} //= 1
                    }
                }

                #Соберем в прилинкованные турболендинги сайтлинков старой версии баннера
                _for_each_sitelink_with_turbolanding ( $banner->old,
                        sub { my ($bnr, $sl) = @_;
                             $linked_turbolandings{$sl->tl_id} = 1; }
                );

                #Турболендинги сайтлинков текущей версии баннера занесем в %alive
                _for_each_sitelink_with_turbolanding ( $banner,
                        sub { my ($bnr, $sl) = @_;
                                $alive{$sl->tl_id} = 1;
                                unless ( exists $linked_turbolandings{$sl->tl_id} ){
                                        # и в добавленные, если их не было в старой версии баннера
                                        $added{$sl->tl_id} = 1;
                                }
                        }
                );
                #Турболендинги, присутствующие в старой версии баннера, но отсутствующие в текущей занесем в удаляемые
                $deleted{$_} = 1 foreach grep {!exists $alive{$_}} keys %linked_turbolandings;
        }

        return {
                added   => [keys %added],
                deleted => [keys %deleted],
        }
}

sub _for_each_sitelink_with_turbolanding ($&){
        my ($banner, $code) = @_;

        return unless $banner->has_sitelinks_set;
        foreach my $sitelink (@{$banner->sitelinks_set->links}){
                next unless $sitelink->has_tl_id && $sitelink->tl_id;
                $code->($banner, $sitelink)
        }

        return;
}

sub _get_counters {
        my ($self, $shard, $turbolandings_by_bid, $counters_and_goals_by_tl_id) = @_;

        my $counters_by_bid = {};
        foreach my $bid (keys %$turbolandings_by_bid) {
                foreach my $tl_id (keys %{$turbolandings_by_bid->{$bid}}){
                        my $counters_and_goals = $counters_and_goals_by_tl_id->{$tl_id};
                        next unless $counters_and_goals;
                        $counters_by_bid->{$bid} //= {};
                        $counters_by_bid->{$bid}->{$_} = 1 foreach @{$counters_and_goals->{counters}};
                }
        }

        return $counters_by_bid;
}

sub _refresh_counters {
        my ($self, $shard, $desired_counters_by_bid, $cid_by_bid) = @_;

        do_in_transaction {
                my $existing_counters = get_all_sql(PPC(shard => $shard),
                        [q/SELECT bid, metrika_counter FROM camp_turbolanding_metrika_counters/,
                                WHERE => {bid => [keys %$cid_by_bid]}
                        ]);
                my $to_delete;
                foreach my $row (@$existing_counters){
                        if (exists $desired_counters_by_bid->{$row->{bid}}->{$row->{metrika_counter}}){
                                #Если в новой конфигурации существующий счетчик есть - модификаци БД не требуется
                                delete $desired_counters_by_bid->{$row->{bid}}->{$row->{metrika_counter}};
                                next;
                        }
                        #Если счетчика нет в новой конфигурации - удаляем его из БД
                        push @{$to_delete->{$row->{bid}}}, $row->{metrika_counter};
                }

                if (keys %$to_delete) {
                        foreach my $chunk (chunks([keys %$to_delete], $MAX_ROWS_PER_DELETE)) {
                                my $conditions = [map {
                                                ('_AND' => { bid => $_, metrika_counter => $to_delete->{$_}})
                                        } @$chunk];
                                do_delete_from_table(PPC(shard => $shard), 'camp_turbolanding_metrika_counters',
                                        where => {'_OR' => $conditions}
                                );
                        }
                }

                if (keys %$desired_counters_by_bid) {
                        my @to_insert;
                        #Оставшиеся в новой конфигурации счетчики добавляем в camp_turbolanding_metrika_counters
                        foreach my $bid (keys %$desired_counters_by_bid){
                                push @to_insert, map {[$cid_by_bid->{$bid}, $bid, $_]} keys %{$desired_counters_by_bid->{$bid}};
                        }
                        do_mass_insert_sql(PPC(shard => $shard),
                                'INSERT INTO camp_turbolanding_metrika_counters (cid, bid, metrika_counter) VALUES %s',
                                \@to_insert
                        );
                }
        };

        return;
}

sub _refresh_goals {
        my ($self, $shard, $turbolanding_counters, $counters_and_goals_by_tl_id) = @_;

        my @affected_turbolandings = uniq map {keys %$_} values %$turbolanding_counters;
        return unless @affected_turbolandings;

        my %goals_by_cid;
        my @goals_select_condition;
        foreach my $cid (keys %$turbolanding_counters){
                foreach my $tl_id (keys %{$turbolanding_counters->{$cid}}){
                        my $counters_and_goals = $counters_and_goals_by_tl_id->{$tl_id};
                        next unless $counters_and_goals;
                        my $usages_count = $turbolanding_counters->{$cid}->{$tl_id};

                        $goals_by_cid{$cid} //= {} if @{$counters_and_goals->{goals}};
                        foreach my $goal (@{$counters_and_goals->{goals}}) {
                                $goals_by_cid{$cid}->{$goal} += $usages_count;
                        }
                }
                #удалим из %{$goals_by_cid->{$cid}} те цели, у которых прирост счетчика использования нулевой
                foreach my $tl_id (keys %{$goals_by_cid{$cid}}) {
                        delete $goals_by_cid{$cid}->{$tl_id} if $goals_by_cid{$cid}->{$tl_id} == 0;
                }
                next unless keys $goals_by_cid{$cid};
                push @goals_select_condition, _AND => [cid => $cid, goal_id => [keys %{$goals_by_cid{$cid}}]];
        }

        return if !@goals_select_condition;

        do_in_transaction {
                my $known_goals = get_all_sql(PPC(shard => $shard), [
                                q/SELECT cid, goal_id, goal_role, links_count FROM camp_metrika_goals/,
                                WHERE => {_OR => \@goals_select_condition},
                        ]
                );

                #Для целей, присутствующих в БД обновим счетчики и, если нужно, goal_role. Цели отсутствующие в camp_metrika_goals - добавим
                my (%goals_for_update, @goals_for_insert);
                foreach my $row (@$known_goals){
                        my $cid = $row->{cid};
                        my $goal_id = $row->{goal_id};
                        my $usages_count = delete $goals_by_cid{$cid}->{$row->{goal_id}};
                        next unless $usages_count;
                        $goals_for_update{$cid} //= {};
                        $goals_for_update{$cid}->{$goal_id} = {links_count => $row->{links_count} + $usages_count};
                        $goals_for_update{$cid}->{$goal_id}->{goal_role} = join( ',', $row->{goal_role}, 'combined')
                                unless $row->{goal_role} =~ /\Wcombined\W/;
                }

                foreach my $cid (keys %goals_for_update) {
                        do_mass_update_sql(PPC(shard => $shard), 'camp_metrika_goals', 'goal_id', $goals_for_update{$cid},
                                where => {cid => $cid},
                        );
                }

                foreach my $cid (keys %goals_by_cid) {
                        foreach my $goal_id (keys %{$goals_by_cid{$cid}}) {
                                # links_count unsigned, запись с links_count в БК не выгружается, поэтому не будем их и добавлять
                                next if $goals_by_cid{$cid}->{$goal_id} < 1;
                                push @goals_for_insert, [
                                        $cid,
                                        $goal_id,
                                        $goals_by_cid{$cid}->{$goal_id},
                                        'combined',
                                ]
                        }
                }
                do_mass_insert_sql(PPC(shard => $shard),
                        q/INSERT IGNORE INTO camp_metrika_goals (cid, goal_id, links_count, goal_role) VALUES %s/, \@goals_for_insert
                );
        };
    return;
}

sub _save_turbolanding_href_params {
        my ($self, $shard, $lists) = @_;
        my ($to_update, $to_delete, $to_create) = @$lists{qw/to_update to_delete to_create/};

        if (@$to_delete) {
            do_delete_from_table(PPC(shard => $shard), 'banner_turbolanding_params',
                where => { bid => $to_delete });
        }

        if (@$to_create) {
            do_mass_insert_sql(PPC(shard => $shard),
                q/INSERT INTO banner_turbolanding_params (bid, href_params) VALUES %s
                        ON DUPLICATE KEY UPDATE
                        href_params = VALUES(href_params)
                /, $to_create );
        }

        if (keys %$to_update) {
                do_mass_update_sql(PPC(shard => $shard), 'banner_turbolanding_params', 'bid', $to_update);
        }

        return;
}

=head3 _do_update_permalinks($shard, $banners)

    Добавляет/обновляет/удаляет привязки организаций Справочника к баннерам.

=cut
sub _do_update_permalinks {
    my ($self, $shard, $banners) = @_;

    my (@to_delete, @to_create_or_update);

    for my $banner (@$banners) {
        next unless $banner->is_permalink_supported();

        if ($banner->has_old) {
            # Есть предыдущая версия баннера
            if ($banner->old->has_permalink && $banner->old->permalink) {
                if ($banner->has_permalink && $banner->permalink) {
                    if ($banner->permalink != $banner->old->permalink) {
                        push @to_create_or_update, $banner;
                    }
                } else {
                    push @to_delete, $banner;
                }
            } elsif ($banner->has_permalink && $banner->permalink) {
                # У старого баннера не было пермалинка
                push @to_create_or_update, $banner;
            }
        } elsif ($banner->has_permalink && $banner->permalink) {
            # Новый баннер
            push @to_create_or_update, $banner;
        }
    }

    my $bids_to_delete_permalinks = [ map {$_->id} @to_delete ];
    if (@$bids_to_delete_permalinks) {
        do_delete_from_table(PPC(shard => $shard), 'banner_permalinks',
            where => {
                bid                   => $bids_to_delete_permalinks,
                permalink_assign_type => 'manual',
            });

        do_delete_from_table(PPC(shard => $shard), 'banner_phones',
            where => {
                bid => $bids_to_delete_permalinks
            });
    }

    my $permalinks_to_create_or_update = [ map {[ $_->id, $_->permalink, 'manual' ]} @to_create_or_update ];
    if (@$permalinks_to_create_or_update) {
        do_mass_insert_sql(PPC(shard => $shard),
            q/INSERT INTO banner_permalinks (bid, permalink, permalink_assign_type) VALUES %s
                ON DUPLICATE KEY UPDATE
                permalink = VALUES(permalink)
        /, $permalinks_to_create_or_update
        );

        # Добавляем организации и сразу ставим статус published, т.к. до этого мы все ходили в ручку Справочника
        my $client_id = $to_create_or_update[0]->client_id;
        my $organizations = [
            map { [$client_id, $_, 'published'] } uniq map { $_->permalink } @to_create_or_update
        ];
        do_mass_insert_sql(PPC(shard => $shard),
            q/INSERT IGNORE INTO organizations (ClientID, permalink_id, status_publish) VALUES %s/,
            $organizations
        );
    }
}

1;
