package Direct::Model::BannerPerformance::Manager;

use Direct::Modern;
use Mouse;

extends 'Direct::Model::Banner::Manager';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use List::MoreUtils qw/part uniq/;
use GeoTools qw//;
use Direct::Model::BannerPerformance;
use JavaIntapi::GenerateObjectIds;


has '+items' => (isa => 'ArrayRef[Direct::Model::BannerPerformance]');

=head2 _create_in_shard($shard, $banners)

Внутренний метод.
Создание в БД в определённом шарде записей для соответствующих объектов (перфоманс объявлений).

=cut

sub _create_in_shard {
    my ($self, $shard, $banners) = @_;

    my @columns = Direct::Model::BannerPerformance->get_db_columns_list('banners_performance');

    my $new_ids = JavaIntapi::GenerateObjectIds->new(object_type => 'banner_creative',
            count => scalar @$banners)->call();
    $_->banner_creative_id(shift @$new_ids) for @$banners;

    do_in_transaction {
        # Создадим базовые баннеры
        $self->SUPER::_create_in_shard($shard, $banners);

        $self->_insert_to_secondary_table_in_db(PPC(shard => $shard), 'banners_performance', banners => ['bid', 'pid', 'cid'], \@columns, $banners);

        # Обработка флагов
        $self->_do_set_geo_for_new_creatives($shard, $banners);
        $self->_do_moderate_creatives($shard, $banners);
    };

    return;
}

=head2 _update_in_shard($shard, $banners, %options)

Внутренний метод.
Обновление в БД в определённом шарде записей для соответствующих объектов (перфоманс объявлений).

=cut

sub _update_in_shard {
    my ($self, $shard, $banners, %options) = @_;

    do_in_transaction {
        # Обновим базовые баннеры
        $self->SUPER::_update_in_shard($shard, $banners, %options);

        # Не делаем обновление при смене креатива, т.к. может быть операция переставления двух значений местами:
        #   (pid0, bid1, |creative_id1|) <-> (pid0, bid2, |creative_id2|)

        my ($banners_to_recreate, $banners_to_update) = part { $_->is_creative_id_changed ? 0 : 1 } @$banners;

        if ($banners_to_recreate) {
            my @columns = Direct::Model::BannerPerformance->get_db_columns_list('banners_performance');
            do_delete_from_table(PPC(shard => $shard), 'banners_performance', where => {bid => [map { $_->id } @$banners_to_recreate]});
            $self->_insert_to_secondary_table_in_db(PPC(shard => $shard), 'banners_performance', banners => ['bid', 'pid', 'cid'], \@columns, $banners_to_recreate);
        }

        if ($banners_to_update) {
            $self->_update_one_table_in_db(PPC(shard => $shard), banners_performance => 'bid', $banners_to_update);
        }

        # Обработка флагов
        $self->_do_set_geo_for_new_creatives($shard, $banners);
        $self->_do_moderate_creatives($shard, $banners);
    };

    return;
}

=head2 _delete_in_shard($shard, $banners)

Внутренний метод.
Удаление из БД в определённом шарде записей для соответствующих объектов (перфоманс объявлений).

=cut

sub _delete_in_shard {
    my ($self, $shard, $banners) = @_;

    my @banner_ids = map { $_->id } @$banners;

    do_in_transaction {
        do_delete_from_table(PPC(shard => $shard), 'banners_performance', where => {bid => \@banner_ids});

        # Удаление оставшихся частей
        $self->SUPER::_delete_in_shard($shard, $banners);
    };
    $self->_do_set_geo_for_new_creatives($shard, $banners);
    return;
}

=head2 get_joined_geo
    На вход принимает hashref списка пар id_креатива => translocal_opt.

    Возвращает объединенный по всем группам, которым принадлежит креатив список георегионов
    Результат - hashref:
    {
      id_креатива1 => 'id_региона1,id_региона2,id_региона3 ...',
      id_креатива2 => 'id_региона4,id_региона3,id_региона7 ...',
      ...
    }
=cut

sub get_joined_geo {
    my ($class, $translocal_options_by_creative_id) = @_;

    my @creative_ids = keys %$translocal_options_by_creative_id;
    my $joined_geo = get_hash_sql(PPC(creative_id => \@creative_ids), [
        'SELECT creative_id, GROUP_CONCAT( geo SEPARATOR ":") as joined_geo FROM banners_performance bp JOIN phrases ag USING(pid)',
        WHERE => {
            creative_id => SHARD_IDS
        },
        'GROUP BY' => 'bp.creative_id'
    ]);

    foreach my $creative_id (@creative_ids){
        unless (exists $joined_geo->{$creative_id}){
            $joined_geo->{$creative_id} = undef;
            next;
        }
        my $translocal_opt = $translocal_options_by_creative_id->{$creative_id};
        my $geo_union = GeoTools::get_targetings_union([split( /:/, $joined_geo->{$creative_id})], $translocal_opt);
        $joined_geo->{$creative_id} = GeoTools::moderation_countries($geo_union, $translocal_opt);
    }

    return $joined_geo;
}



sub _do_set_geo_for_new_creatives {
    my ($self, $shard, $banners) = @_;

    my %creative_params;
    for (@$banners) {
        next if  $creative_params{$_->creative_id} || !$_->do_set_geo_for_new_creative;
        $creative_params{$_->creative_id} = $_->do_set_geo_for_new_creative->{translocal_opt};
    }

    return if !%creative_params;

    # Вычислим и обновим sum_geo в БД

    my $case_values = $self->get_joined_geo(\%creative_params);

    do_update_table(PPC(shard => $shard), 'perf_creatives', {
        sum_geo__dont_quote => sql_case(creative_id => $case_values, default__dont_quote => 'sum_geo'),
    }, where => {
        creative_id => [keys %creative_params],
        statusModerate => ['New', 'Error'],
    });

    return;
}

sub _do_moderate_creatives {
    my ($self, $shard, $banners) = @_;

    my @creative_ids = keys %{+{ map { $_->creative_id => 1 } grep { $_->do_moderate_creative } @$banners }};
    return if !@creative_ids;

    do_update_table(PPC(shard => $shard), 'perf_creatives', {
        statusModerate => 'Ready',
        moderate_try_count => 0,
        moderate_send_time__dont_quote => 'NOW() + INTERVAL 10 MINUTE',
    }, where => {
        creative_id => \@creative_ids,
        statusModerate => ['New', 'Error'],
    });

    my @bids = map { $_->id } grep { $_->do_moderate_creative } @$banners;

    do_delete_from_table(PPC(shard => $shard), 'banners_minus_geo', where => { bid => \@bids, type => 'current' });

    return;
}

sub _do_check_redirect {
    my ($self, $shard, $banners) = @_;
    return;
}

sub _do_update_filter_domain {
    my ($self, $shard, $banners) = @_;
    return;
}

sub _do_update_adgroups {
    my ($self, $shard, $banners) = @_;

    # Незадействованные флаги: set_adgroup_has_phraseid_href, resume_adgroup_autobudget_show

    my (%adgroup_ids, %case_values, %update_adgroups_performance);
    for my $banner (grep {
        $_->do_update_adgroup_last_change ||
        $_->do_moderate_adgroup ||
        $_->do_bs_sync_adgroup
    } @$banners) {
        $adgroup_ids{$banner->adgroup_id} = 1;

        # Принудительное изменение/сохранение времени модификации группы
        $case_values{LastChange}->{$banner->adgroup_id} = $banner->do_update_adgroup_last_change ? 'NOW()' : 'LastChange';

        $case_values{statusBsSynced}->{$banner->adgroup_id} = sql_quote('No') if $banner->do_bs_sync_adgroup;

        if ($banner->do_moderate_adgroup) {
            $case_values{statusModerate}->{$banner->adgroup_id} = sql_quote('Yes');
            $case_values{statusPostModerate}->{$banner->adgroup_id} = sql_quote('Yes');
            $case_values{statusBsSynced}->{$banner->adgroup_id} = sql_quote('No'); # because we accepted the adgroup
        }
    }

    for my $banner (grep { $_->do_set_adgroup_bl_status } @$banners) {
        # отдельным циклом, иначе в предыдущем цикле можем заполнить только %adgroup_ids и LastChange
        my $status = $banner->do_set_adgroup_bl_status;
        $update_adgroups_performance{$banner->adgroup_id}->{statusBlGenerated} = $status;
    }

    do_update_table(PPC(shard => $shard), 'phrases', {
        (map { $_ => sql_case(pid => $case_values{$_}, default__dont_quote => $_, dont_quote_value => 1) } keys %case_values)
    }, where => {pid => [keys %adgroup_ids]}, dont_quote => [keys %case_values]) if %adgroup_ids;

    if (%update_adgroups_performance) {
        do_mass_update_sql(PPC(shard => $shard), 'adgroups_performance', pid => \%update_adgroups_performance);
    }

    return;
}

sub _do_moderate_campaign_if_no_active_banners {
    my ($self, $shard, $banners) = @_;

    $banners = [grep { $_->do_moderate_campaign_if_no_active_banners } @$banners];
    return if !@$banners;

    do_sql(PPC(shard => $shard), ["
        UPDATE campaigns c JOIN camp_options co ON (co.cid = c.cid)
        SET c.statusModerate = 'Ready', co.statusPostModerate = 'Accepted', c.statusBsSynced = 'No'
    ", WHERE => {
        'c.cid' => [uniq(map { $_->campaign_id } @$banners)],
        'c.statusModerate' => 'New',
    }]);

    return;
}

sub _do_insert_moderation_data {
    my ($self, $shard, $banners) = @_;
    return;
}

sub _do_clear_moderation_data {
    my ($self, $shard, $banners) = @_;
    return;
}

sub _do_update_images {
    my ($self, $shard, $banners) = @_;
    return;
}

sub _do_schedule_forecast {
    my ($self, $shard, $banners) = @_;
    return;
}

__PACKAGE__->meta->make_immutable;

1;
