package Direct::Model::BidRelevanceMatch::Helper;

=head1 NAME

=encoding utf8

Direct::Model::BidRelevanceMatch::Helper - функции доступные извне, вынесенные во избежание циклических зависимостей

=cut

use Direct::Modern;
use Currencies qw/currency_price_rounding/;
use Tools qw/calc_percentile/;
use base qw/Exporter/;
our @EXPORT = qw(
    $CONDITION_XLS_PREFIX_RELEVANCE_MATCH
    $CONDITION_XLS_RELEVANCE_MATCH
    is_relevance_match
);

# маркер фразы в xls содержащий автотаргетинг (+ см. is_relevance_match)
our $CONDITION_XLS_PREFIX_RELEVANCE_MATCH = "'---autotargeting";

# маркер автотаргетинга в сообщениях об ошибках при обработке xls
our $CONDITION_XLS_RELEVANCE_MATCH = "autotargeting: ";

=head2 is_relevance_match

    Проверка строки из XLS файла, что она описывает автотаргетинг

=cut

sub is_relevance_match($) {
    return shift =~ /^'?---autotargeting/i;
}

=head2 calc_price

    Расчет ставки для автотаргетинга на основе других ставок (например, фразы).
    От ставок (на поиске) на фразы берем 30ый перцентиль и округяем его вверх до шага торгов.

    $relevance_match_price = Direct::Model::BidRelevanceMatch::Helper::calc_price_by_keywords([ map {$_->{price}} @{$keywords} ], $camp->{currency});

=cut


sub calc_price {
    my ($prices, $currency, $percentile) = @_;
    return currency_price_rounding(calc_percentile($prices, $percentile // 0.3), $currency, up => 1);
}

=head2 calc_average_price
    Расчет средневзвешенной ставки для автотаргетинга на основе других ставок.
    В качестве весовых коэффициэнтов используется количество кликов.
    
    $relevance_match_price = calc_average_price([ {price => price1, clicks => clicks1}, { price => ...}, ... ], $currency)

=cut

sub calc_average_price {
    my ($prices_with_clicks, $currency, $context_price_coef) = @_;
    
    $context_price_coef = ($context_price_coef // 100) / 100;
    my ($price_sum, $clicks_sum);
    foreach my $pc (@$prices_with_clicks){
        $price_sum +=  ($pc->{price} // 0) * (($pc->{clicks} // 0) + 1);
        $clicks_sum += ($pc->{clicks} // 0) + 1;
    }
    
    return currency_price_rounding(
            $price_sum * $context_price_coef / $clicks_sum,
            $currency, up => 1);
}

=head2 extract_ctx_prices
    Из массива типовых hashref фраз с полями статистики готовит структуру $prices_with_click для calc_average_price.
    Фразы без ставок игнорируются.
    Если заданы $min_price и $max_price в результирующем массиве будут только записи со ставками не выходящими за заданные пределы.

    Возвращает структуру вида
    [ {price => price1, clicks => clicks1}, { price => ...}, ... ]
=cut

sub extract_ctx_prices {
    my ($phrases, $min_price, $max_price) = @_;
    
    my @ctx_prices;
    foreach my $ph (@$phrases){
        my $rec;
        if ($ph->{price_context}) {
            $rec = {price => $ph->{price_context}, clicks => $ph->{ctx_clicks}}
        }
        elsif ($ph->{price}) {
            $rec = {price => $ph->{price}, clicks => $ph->{clicks}}
        }
        else { next }
        
        next if defined $min_price && $rec->{price} < $min_price;
        next if defined $max_price && $rec->{price} > $max_price;

        push @ctx_prices, $rec;
    }
    
    return \@ctx_prices;
}

1;
