package Direct::Model::BidRelevanceMatch::Manager;

use Direct::Modern;
use Mouse;

extends 'Yandex::ORM::Model::Manager::Base';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use List::MoreUtils qw/part uniq/;

use Direct::Model::BidRelevanceMatch;

extends 'Direct::Model::Bid::Manager';

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::BidRelevanceMatch]');

=head2 create

    Создание в БД записей для беcфразного таргетинга

=cut

sub create {
    my ($self) = @_;

    my @bids_columns = Direct::Model::BidRelevanceMatch->get_db_columns_list('bids_base');

    # Обработаем объекты по шардам
    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});

        do_in_transaction {
                my $exists_bids = get_hash_sql(
                    PPC(shard => $shard),
                    [
                        q/SELECT g.pid, IFNULL(bid.bid_id, 0) FROM phrases g LEFT JOIN bids_base bid ON (g.pid = bid.pid AND bid.bid_type IN ('relevance_match','relevance_match_search'))/,
                        WHERE => [ 'g.pid' => [ map { $_->adgroup_id } @$shard_items ] ],
                        q/LOCK IN SHARE MODE/,
                    ]
                );
                my ($new_bids, $old_bids) = part { !!$exists_bids->{$_->adgroup_id} } @$shard_items;
                if ($new_bids) {
                    my $ids = get_new_id_multi(phid => scalar(@$new_bids));
                    $_->id(shift @$ids) for @{$new_bids};
                    $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'bids_base', \@bids_columns, $new_bids);
                    $self->_insert_to_secondary_table_in_db(
                        PPC(shard => $shard),
                        'bids_href_params',
                        'bids_base' => [ qw/cid id/ ],
                        [ Direct::Model::BidRelevanceMatch->get_db_columns_list('bids_href_params') ],
                        [ grep { ($_->has_href_param1 && defined $_->href_param1) || ($_->has_href_param2 && defined $_->href_param2) } @$shard_items ],
                        { id => 'bid_id' },
                    );
                }
                if ($old_bids) {
                    foreach my $old_bid (@$old_bids) {
                        $old_bid->id($exists_bids->{$old_bid->adgroup_id});
                        $old_bid->is_bid_type_changed(1);
                        $old_bid->is_suspended(0) unless $old_bid->has_is_suspended();
                        $old_bid->is_deleted(0);
                    }
                    $self->_update_one_table_in_db(PPC(shard => $shard), bids_base => 'bid_id', $old_bids);
                    $self->_update_href_params($shard, $shard_items);
                }

                # Обработаем флаги
                $self->_do_update_adgroups($shard, $shard_items);
            };
    }

    $_->reset_state() for @{$self->items};

    return;
}

=head2 update

    Обновление в БД записей для беcфразного таргетинга

=cut

sub update {
    my ($self) = @_;

    # Обработаем объекты по шардам
    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});

        do_in_transaction {
                $self->_update_one_table_in_db(PPC(shard => $shard), bids_base => 'bid_id', $shard_items);
                $self->_update_href_params($shard, $shard_items);

                # Обработаем флаги
                $self->_do_update_adgroups($shard, $shard_items);
            };
    };

    $_->reset_state() for @{$self->items};

    return;
}

sub _update_href_params {
    my ($self, $shard, $items) = @_;

    my @bids_href_params = grep { $_->has_href_param1 && $_->has_href_param2 && ($_->is_href_param1_changed || $_->is_href_param2_changed) } @$items;
    if (@bids_href_params) {
        do_mass_insert_sql(
            PPC(shard => $shard),
            q/
            INSERT INTO `bids_href_params` (`cid`, `id`, `param1`, `param2`)
            VALUES %s ON DUPLICATE KEY UPDATE
                param1 = VALUES(param1),
                param2 = VALUES(param2)
            /,
            [ map { [ $_->campaign_id, $_->id, $_->href_param1, $_->href_param2 ] } @bids_href_params ],
            { sleep => 1, max_row_for_insert => 5000 }
        );
    }

    return;
}

__PACKAGE__->meta->make_immutable;

1;
