package Direct::Model::BillingAggregate::Manager;

use Direct::Modern;
use Mouse;

extends 'Yandex::ORM::Model::Manager::Base';

use Settings;

use Yandex::DBTools;

use BalanceQueue;
use Campaign;
use DirectRedis;
use Direct::Model::BillingAggregate;
use RedisLock;

# в секундах
my $REDIS_LOCK_TTL = 5 * 60;

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::BillingAggregate]',
);

has 'operator_uid' => (
    is => 'ro',
    isa => 'Id',
    required => 1,
);

sub create {
    my ($self) = @_;

    for my $item (@{$self->items}) {
        $self->_create_one($item);
        $item->reset_state();
    }
}

sub _create_one {
    my ($self, $item) = @_;

    # fastpath
    if ($self->_find_existing_aggregate($item)) {
        return;
    }
    my $lock_guard = $self->_get_wallet_lock($item);
    if ($self->_find_existing_aggregate($item)) {
        return;
    }

    do_in_transaction sub {
        $self->_create_camp_record($item);
        $self->_add_to_balance_queue($item);
    };

    return;
}

sub _find_existing_aggregate {
    my ($self, $item) = @_;

    my $agg_id = get_one_field_sql(PPC(ClientID => $item->client_id), ["
        SELECT cid
        FROM campaigns",
        WHERE => [
            type => 'billing_aggregate',
            ClientID => $item->client_id,
            AgencyID => $item->agency_id,
            ProductID => $item->product_id,
        ],
        LIMIT => 1,
    ]);
    return $agg_id;
}

# берём лок на создание биллингового агрегата.
# Если случайно сделать под одним общим счётом два агрегата с одинаковым ProductID, будет не очень хорошо
sub _get_wallet_lock {
    my ($self, $item) = @_;

    my $lock_name = "billing_aggregate_create_".$item->wallet_id."_".$item->product_id;
    my $redis = DirectRedis::get_redis();
    my ($guard, $locked, $failed) = RedisLock::lock_multi_guard($redis, [$lock_name], $REDIS_LOCK_TTL);
    if (@$failed) {
        die "failed to get redis lock $lock_name";
    }
    return $guard;
}

sub _create_camp_record {
    my ($self, $item) = @_;

    my %camp_options = (
        client_chief_uid => $item->user_id,
        ClientID => $item->client_id,
        currency => $item->currency,
        client_fio => $item->client_fio,
        client_email => $item->email,
        statusEmpty => $item->status_empty,
        statusModerate => $item->status_moderate,
        statusPostModerate => $item->status_post_moderate,
        name => $item->campaign_name,
        type => $item->campaign_type,
        agency_uid => $item->agency_user_id,
        manager_uid => $item->manager_user_id,
        product_id => $item->product_id,
    );
    my $agg_id = Campaign::create_empty_camp(%camp_options);
    do_sql(PPC(ClientID => $item->client_id), ["
        UPDATE campaigns
        SET wallet_cid = ?",
        WHERE => [
            cid => $agg_id,
        ]
    ], $item->wallet_id);

    $item->id($agg_id);
}

sub _add_to_balance_queue {
    my ($self, $item) = @_;

    BalanceQueue::add_to_balance_info_queue($self->operator_uid, cid => [$item->id], BalanceQueue::PRIORITY_CAMP_ON_NEW_BILLING_AGGREGATE);
}

1;
