package Direct::Model::Campaign;

use Direct::Modern;

use Mouse;
use Mouse::Util::TypeConstraints;
use JSON;

use Yandex::ORM::Helpers qw/is_json_eq_without_quotes/;

use Client;
use HashingTools qw/url_hash_utf8/;
use Yandex::HashUtils qw/hash_cut/;
use Direct::Validation::Domains;
use Direct::Strategy::Tools qw/ strategy_from_strategy_hash /;
use MinusWordsTools;
use TimeTarget;

=head2 @ATTRIBUTION_MODELS

    Список доступных моделей атрибуции

=cut

our @ATTRIBUTION_MODELS = qw/
    last_click
    first_click
    last_significant_click
    last_yandex_direct_click
    first_click_cross_device
    last_significant_click_cross_device
    last_yandex_direct_click_cross_device
/;

my @SMS_TIME_PARTS = qw/
    sms_time_from_hours
    sms_time_from_minutes
    sms_time_to_hours
    sms_time_to_minutes
/;

=head2 @PLACEMENT_TYPES

    Список доступных видов размещения

=cut

our @PLACEMENT_TYPES = qw/
    search_page
    adv_gallery
/;

extends 'Yandex::ORM::Model::Base';
with 'Direct::Model::Role::Update';

# NB: многие атрибуты, которые в БД хранятся как set, в модели объявлены так же set-ами. Объявление их строками дало бы возможность
# конструировать объекты с произвольными значениями таких атрибутов, но это усложнило бы модель + и в web-интерфейсе, и в API
# существует предвалидация данных (в API - wsdl), которая проверить переданные значения.

__PACKAGE__->_setup(
    default_table  => 'campaigns',

    fields => [
        id                      => { type => 'Id', column => 'cid', primary_key => 1 },
        campaign_type           => { type => 'Enum', values => [qw/text mcb geo dynamic mobile_content performance mcbanner cpm_banner cpm_deals
                                    billing_aggregate internal_autobudget internal_distrib internal_free cpm_yndx_frontpage content_promotion cpm_price/], column => 'type' }, # wallet

        product_id              => { type => 'Maybe[Id]', column => 'ProductID' },
        client_id               => { type => 'Id', column => 'ClientID' },
        user_id                 => { type => 'Id', column => 'uid' },
        manager_user_id         => { type => 'Maybe[Id]', column => 'ManagerUID' },
        agency_id               => { type => 'Id', column => 'AgencyID', default => 0 },
        agency_user_id          => { type => 'Maybe[Id]', column => 'AgencyUID' },
        wallet_id               => { type => 'Id', column => 'wallet_cid' },
        campaign_name           => { type => 'Maybe[Str]', length => 255, column => 'name' },
        start_date              => { type => 'Maybe[Str]', default => '0000-00-00', column => 'start_time', },
        finish_date             => { type => 'Maybe[Str]', default => '0000-00-00', column => 'finish_time', },
        bs_order_id             => { type => 'Int', column => 'OrderID', default => 0 },
        currency                => { type => 'Enum', values => [qw/RUB UAH KZT USD EUR YND_FIXED CHF TRY BYN GBP/] },
        converted               => { type => 'Enum', values => [qw/Yes No/], column => 'currencyConverted', default => 'No' },
        sum                     => { type => 'Num', default => 0 },
        sum_to_pay              => { type => 'Num', default => 0 },
        sum_last                => { type => 'Num', default => 0 },
        status_archived         => { type => 'Enum', values => [qw/Yes No/], column => 'archived', default => 'No' },
        status_moderate         => { type => 'Enum', values => [qw/Yes No Sent Ready New/], column => 'statusModerate', default => 'New', volatile => 1 },
        status_post_moderate    => { type => 'Enum', values => [qw/New Yes No Accepted/], column => 'statusPostModerate', default => 'New', table => 'camp_options', volatile => 1 },
        status_bs_synced        => { type => 'Enum', values => [qw/Yes No Sending/], column => 'statusBsSynced', default => 'No', volatile => 1 },
        status_empty            => { type => 'Enum', values => [qw/Yes No/], column => 'statusEmpty', default => 'Yes' },
        status_show             => { type => 'Enum', values => [qw/Yes No/], column => 'statusShow', default => 'Yes' },
        status_active           => { type => 'Enum', values => [qw/Yes No/], column => 'statusActive', default => 'No' },
        stop_time               => { type => 'Str', default => '0000-00-00', table => 'camp_options', column => 'stopTime' },
        email                   => { type => 'Maybe[Str]', length => 255, table => 'camp_options' },
        client_fio              => { type => 'Maybe[Str]', length => 255, table => 'camp_options', column => 'FIO' },
        time_target             => { type => 'Maybe[Str]', column => 'timeTarget', default => undef },
        timezone_id             => { type => 'Int', },
        geo                     => { type => 'Maybe[Str]', },
        placement_types         => { type => 'Set', values => \@PLACEMENT_TYPES, table => 'camp_options', default => '' },
        broad_match_flag        => { type => 'Enum', values => [qw/Yes No/], default => 'No', table => 'camp_options', },
        broad_match_limit       => { type => 'Int', default => 0, table => 'camp_options',  },
        broad_match_goal_id     => { type => 'Maybe[Id]', table => 'camp_options', },
        device_target           => { type => 'Str', length => 100, table => 'camp_options', column => 'device_targeting', default => '' },
        money_warning_threshold => { type => 'Num', table => 'camp_options', column => 'money_warning_value', default => 20 },
        position_check_interval => { type => 'Str', table => 'camp_options', column => 'warnPlaceInterval', default => '60' }, # values => [qw/ 15 30 60 /],
        attribution_model       => { type => 'Maybe[Enum]', values => \@ATTRIBUTION_MODELS, column => 'attribution_model' },
        ab_segment_stat_ret_cond_id => { type => 'Maybe[Id]', column => 'ab_segment_stat_ret_cond_id' },
        ab_segment_ret_cond_id => { type => 'Maybe[Id]', column => 'ab_segment_ret_cond_id' },

        content_lang            => { type => 'Maybe[Str]', length => 2, table => 'camp_options' },

        status_autobudget_forecast => { type => 'Enum', values => [qw/New Valid Wrong/], default => 'New', column => 'statusAutobudgetForecast' },
        autobudget_forecast_date => { type => 'Maybe[Str]', default => undef, column => 'autobudgetForecastDate' },

        last_change             => { type => 'Timestamp', column => 'LastChange' },
        context_limit           => { type => 'Int', column => 'ContextLimit', default => 0 },
        context_price_coef      => { type => 'Int', column => 'ContextPriceCoef' },
        opts                    => { type => 'Set', values => ['', qw/
                no_title_substitute enable_cpc_hold no_extended_geotargeting hide_permalink_info is_virtual has_turbo_smarts is_auto_video_allowed
                is_alone_trafaret_allowed is_touch has_turbo_app require_filtration_by_dont_show_domains has_turbo_app is_simplified_strategy_view_enabled
                is_universal is_order_phrase_length_precedence_enabled is_new_ios_version_enabled is_skadnetwork_enabled is_allowed_on_adult_content
                is_brand_lift_hidden s2s_tracking_enabled recommendations_management_enabled price_recommendations_management_enabled
                use_current_region use_regular_region
            /] },
        fair_auction            => { type => 'Enum', column => 'fairAuction', values => [qw/Yes No/], default => 'No', table => 'camp_options' },
        brand_survey_id         => { type => 'Maybe[Str]', table => 'camp_options' },

        # Strategy fields
        strategy_name           => { type => 'Maybe[Enum]', values => \@Direct::Strategy::Tools::STRATEGY_NAMES },
        _strategy_data          => { type => 'Maybe[Str]', column => 'strategy_data' },
        strategy_id              => { type => 'Maybe[Id]', column => 'strategy_id' },

        # планируется оставить только одну платформу, поэтому enum, а не set
        platform               => {
            type => 'Maybe[Enum]', values => [qw/ search context both /],
            default => sub {shift->default_platform},
        },

        # Legacy strategy fields
        _strategy_name          => { type => 'Maybe[Enum]', column => 'strategy', values => ['', qw/different_places/], table => 'camp_options' },
        _autobudget             => { column => 'autobudget', type => 'Enum', values => [qw/Yes No/] },
        _autobudget_date        => { type => 'Maybe[Str]', column => 'autobudget_date' },

        # Hidden fields
        _disabled_domains          => { type => 'Maybe[Str]', column => 'DontShow' },
        _disabled_ssp              => { type => 'Maybe[Str]', column => 'disabled_ssp' },
        _disabled_video_placements => { type => 'Maybe[Str]', column => 'disabled_video_placements' },
        _disabled_ips              => { type => 'Maybe[Str]', column => 'disabledIps' },
        _competitors_domains       => { type => 'Maybe[Str]', table => 'camp_options', column => 'competitors_domains', },
        _sms_time                  => { type => 'Str', table => 'camp_options', column => 'sms_time', default => '00:00:00:00' },
        _mw_id                     => { type => 'Maybe[Id]', table => 'camp_options', column => 'mw_id' },
        _minus_words               => { type => 'Maybe[Str]', table => 'camp_options', column => 'minus_words' },
        _meaningful_goals          => { type => 'Maybe[Str]', table => 'camp_options', column => 'meaningful_goals',
            custom_eq => \&_is_meaningful_goals_equal},
    ],

    # Additional attributes
    additional => [
        _strategy              => { type => 'Direct::Strategy' },
        is_different_places    => { type => 'Bool', trigger => \&_on_is_different_places_changed },
        _metrika_counters      => { type => 'Maybe[Str]', track_changes => 1 },

        sms_time_from_hours    => { type => 'Str', trigger => sub { _on_sms_time_changed( shift, 'sms_time_from_hours', @_ ) } },
        sms_time_from_minutes  => { type => 'Str', trigger => sub { _on_sms_time_changed( shift, 'sms_time_from_minutes', @_ ) } },
        sms_time_to_hours      => { type => 'Str', trigger => sub { _on_sms_time_changed( shift, 'sms_time_to_hours', @_ ) } },
        sms_time_to_minutes    => { type => 'Str', trigger => sub { _on_sms_time_changed( shift, 'sms_time_to_minutes', @_ ) } },

        minus_words            => { type => 'ArrayRef[Str]', trigger => \&_on_minus_words_changed},
        _minus_words_hash      => { type => 'Str', track_changes => 1 },
        meaningful_goals       => { type => 'ArrayRef[HashRef]', trigger => \&_on_meaningful_goals_changed},

        hierarchical_multipliers       => { type => 'HashRef', trigger => \&_on_hierarchical_multipliers_changed },
        _hierarchical_multipliers_hash => { type => 'Str', track_changes => 1 },
    ],

    relations => [
        adgroups            => { type => 'ArrayRef[Direct::Model::AdGroup]' },
        user                => { type => 'Direct::Model::User' },
        metrika_counters    => { type => 'ArrayRef[Int]', trigger => \&_on_metrika_counters_changed },
        disabled_domains    => { type => 'ArrayRef[Str]', trigger => \&_on_disabled_domains_changed },
        disabled_ips        => { type => 'ArrayRef[Str]', trigger => \&_on_disabled_ips_changed },
        competitors_domains => { type => 'ArrayRef[Str]', trigger => \&_on_competitors_domains_changed },
        disabled_video_placements => { type => 'ArrayRef[Str]', trigger => \&_on_disabled_video_placements_changed }
    ],

    state_flags => [qw/
        update_last_change
        enqueue_for_delete
        save_metrika_goals
        resync_bids_retargeting
    /],
);


=head2 available_creative_types

Список типов креативов, доступных в кампании

=cut

sub available_creative_types { [] }


around BUILDARGS => sub {
    my ($orig, $class) = (shift, shift);

    my %args = ( @_ == 1 && ref($_[0]) eq 'HASH' ) ? %{$_[0]} : @_;

    if (exists $args{_disabled_domains} || exists $args{_disabled_ssp}) {
        my $ssps = from_json($args{_disabled_ssp} || '[]');
        my @domains = grep {$_} map {s/^\s*(.*?)\s*$/$1/r} split(/,/ => $args{_disabled_domains} // '');
        $args{disabled_domains} = Direct::Validation::Domains::merge_disabled_platforms({ssps => $ssps, rest => \@domains});
    }

    if (exists $args{_disabled_video_placements}) {
        $args{disabled_video_placements} = from_json($args{_disabled_video_placements} || '[]');
    }

    if (exists $args{_strategy_name}) {
        $args{is_different_places} = ($args{campaign_type} ne 'performance') && (($args{_strategy_name} // '') ne '');
    }

    if (defined $args{_strategy_data}) {
        my $hash = $args{_strategy_data};
        my $strategy = strategy_from_strategy_hash(from_json $hash);
        $args{_strategy} = $strategy;
        $args{strategy_name} ||= $strategy->name;
    }

    if ( exists $args{_disabled_ips} ) {
        # список IP-адресов, разделенных запятой или пробелами
        $args{disabled_ips} = [ grep { $_ } split( /[\s,]+/ => ( $args{_disabled_ips} // '' ) ) ];
    }

    if ( exists $args{_competitors_domains} ) {
        $args{competitors_domains} = [ grep { $_ } split( /[\s,]+/ => ( $args{_competitors_domains} // '' ) ) ];
    }

    # NB: удобно для API5 Campaigns.add
    if ( exists $args{_metrika_counters} ) {
        $args{metrika_counters} = [ split( /[\s,]+/ => ( $args{_metrika_counters} // '' ) ) ];
    }

    if ( exists $args{_sms_time} ) {
        my $serialized = delete( $args{_sms_time} );

        my @sms_time = split( ':' => $serialized );
        croak qq/Can't deserialize sms_time: $serialized/ if @sms_time != scalar( @SMS_TIME_PARTS );

        # hash slice
        @args{ @SMS_TIME_PARTS } = @sms_time;
    }

    if (exists $args{_minus_words}) {
        $args{minus_words} = MinusWordsTools::minus_words_str2array($args{_minus_words});
    }

    if (exists $args{_meaningful_goals}) {
        $args{meaningful_goals} = _deserialize_meaningful_goals($args{_meaningful_goals});
    }

    $class->$orig(%args);
};


sub _deserialize_meaningful_goals {
    my $goals_json = shift;

    my $goals = from_json($goals_json || '[]');

    state $meaningful_goal_value_serialize_as_string = Property->new('meaningful_goal_value_serialize_as_string');
    my $meaningful_goal_value_serialize_as_string_enabled = $meaningful_goal_value_serialize_as_string->get(10) || 0;

    if ($meaningful_goal_value_serialize_as_string_enabled != 0) {
        foreach (@$goals) {
            $_->{value} = $_->{value} + 0;
            $_->{is_metrika_source_of_value} = $_->{is_metrika_source_of_value} + 0;
        }
    }
    return $goals;
}


sub _serialize_meaningful_goals {
    my $goals = shift;
    my %opt = @_;
    state $json = JSON->new->canonical(1)->utf8(1);
    return undef  if !$goals || !@$goals;
    my @goals;
    if (defined $opt{is_allowed_to_use_value_from_metrika} && $opt{is_allowed_to_use_value_from_metrika}) {
        @goals = map {hash_cut $_, qw/goal_id value is_metrika_source_of_value/} @$goals;
    } else {
        @goals = map {hash_cut $_, qw/goal_id value/} @$goals;
    }
    state $meaningful_goal_value_serialize_as_string = Property->new('meaningful_goal_value_serialize_as_string');
    my $meaningful_goal_value_serialize_as_string_enabled = $meaningful_goal_value_serialize_as_string->get(10) || 0;
    if ($meaningful_goal_value_serialize_as_string_enabled != 0) {
        foreach (@$goals) {
            $_->{value} = "$_->{value}";
        }
    }
    return $json->encode(\@goals);
}


=head2 detect_platform

Определяет код площадки.

Параметры:
  old
  is_search_stop
  is_net_stop

=cut

sub detect_platform {
    my $self = shift;
    my %opt = @_;

    state $platforms = [qw/ _invalid_ search context both /];
    my $is_search =
        exists $opt{is_search_stop} ? !$opt{is_search_stop} :
        $opt{old} ? !$self->_detect_is_search_stop($opt{old}) :
        1;
    my $is_context =
        exists $opt{is_net_stop} ? !$opt{is_net_stop} :
        $opt{old} ? !$self->_detect_is_net_stop($opt{old}) :
        1;

    return $platforms->[(0 + $is_search) + 2 * (0 + $is_context)];
}

sub _detect_is_search_stop {
    my $self = shift;
    my $platform = shift || $self->platform;
    return ($platform // '') eq 'context';
}

sub _detect_is_net_stop {
    my $self = shift;
    my $platform = shift || $self->platform;
    return $platform eq 'search';
}


sub default_platform {
    my $self = shift;
    my ($type) = @_;

    $type ||= $self->campaign_type;

    state $platform_by_type = {
        performance        => 'context',
        mcb                => 'search',
        mcbanner           => 'search',
        content_promotion  => 'search',
        dynamic            => 'search', # ??? context_limited
        cpm_banner         => 'context',
        cpm_deals          => 'context',
        cpm_yndx_frontpage => 'context',
        content_promotion  => 'search',
        cpm_price          => 'context',
    };

    my $platform = $platform_by_type->{$type} || 'both';
    return $platform;
}


sub _on_metrika_counters_changed {
    my ( $self, $new ) = @_;

    $self->_metrika_counters( join ',' => @$new ) if $self->_constructed || ! $self->_has_metrika_counters;
}

sub is_metrika_counters_changed { shift->_is_metrika_counters_changed( @_ ) }


sub _on_disabled_domains_changed {
    my ( $self, $new ) = @_;

    if ($self->_constructed || !$self->_has_disabled_domains || !$self->_has_disabled_ssp) {
        my $client_limits = get_client_limits($self->client_id);
        my $split = Direct::Validation::Domains::split_disabled_platforms(
            $new,
            disable_any_domains_allowed => 1, #на уровне модели не надо валидировать домен по черному списку доменов
            blacklist_size_limit =>$client_limits->{general_blacklist_size_limit}
        );
        $self->_disabled_domains(join ',' => sort @{$split->{rest}});
        $self->_disabled_ssp(to_json([sort @{$split->{ssps}}]));
    }
    return;
}

sub is_disabled_domains_changed {
    my $self = shift;
    return $self->_is_disabled_domains_changed(@_) || $self->_is_disabled_ssp_changed(@_);
}

sub _on_is_different_places_changed {
    my $self = shift;
    return if $self->campaign_type eq 'performance';
    return if $self->campaign_type eq 'mcbanner';
    return if $self->campaign_type eq 'cpm_banner';
    return if $self->campaign_type eq 'cpm_deals';
    return if $self->campaign_type eq 'cpm_yndx_frontpage';
    return if $self->campaign_type eq 'content_promotion';
    return if $self->campaign_type eq 'cpm_price';

    my ($new) = @_;
    my $old = $self->_has_strategy_name && ($self->_strategy_name || '') eq 'different_places';
    if (!!$new != !!$old) {
        $self->_strategy_name($new ? 'different_places' : '');
    }
    return;
}


sub is_is_different_places_changed {
    my $self = shift;
    return ''  if $self->campaign_type eq 'performance';
    return ''  if $self->campaign_type eq 'mcbanner';
    return ''  if $self->campaign_type eq 'cpm_banner';
    return ''  if $self->campaign_type eq 'cpm_deals';
    return ''  if $self->campaign_type eq 'cpm_yndx_frontpage';
    return ''  if $self->campaign_type eq 'content_promotion';
    return ''  if $self->campaign_type eq 'cpm_price';

    return $self->_is_strategy_name_changed(@_);
}


sub _on_disabled_ips_changed {
    my ( $self, $new ) = @_;

    $self->_disabled_ips( join ',' => @$new ) if $self->_constructed || ! $self->_has_disabled_ips;
}

sub is_disabled_ips_changed { shift->_is_disabled_ips_changed( @_ ) }

sub _on_competitors_domains_changed {
    my ( $self, $new ) = @_;

    $self->_competitors_domains( join ',' => @$new ) if $self->_constructed || !$self->_has_competitors_domains;
}

sub is_competitors_domains_changed { shift->_is_competitors_domains_changed( @_ ) }

sub _on_disabled_video_placements_changed {
    my ( $self, $new ) = @_;

    if ($self->_constructed || !$self->_has_disabled_video_placements) {
        $self->_disabled_video_placements(to_json( [ sort @$new ] ));
    }
    return;
}

sub is_disabled_video_placements_changed { shift->_is_disabled_video_placements_changed( @_ ) }

sub _on_hierarchical_multipliers_changed {
    my ( $self, $new ) = @_;

    $self->_hierarchical_multipliers_hash(
        url_hash_utf8(
            JSON->new->utf8(0)->canonical->encode(
                $self->hierarchical_multipliers
            )
        )
    ) if $self->_constructed || !$self->has_hierarchical_multipliers;
}

sub _on_minus_words_changed {
    my ( $self, $new ) = @_;

    if ($self->has_minus_words) {
        $self->_minus_words(MinusWordsTools::minus_words_array2str($self->minus_words));
        $self->_minus_words_hash(MinusWordsTools::minus_words_utf8_hashcode($new));
    }
}

sub is_minus_words_changed { shift->_is_minus_words_hash_changed(@_) }

sub _on_meaningful_goals_changed {
    my ( $self, $new ) = @_;

    $self->_meaningful_goals(_serialize_meaningful_goals($new));
}

sub is_meaningful_goals_changed { shift->_is_meaningful_goals_changed(@_) }

sub is_hierarchical_multipliers_changed { shift->_is_hierarchical_multipliers_hash_changed( @_ ) }

sub _on_sms_time_changed {
    my ( $self, $field, $new ) = @_;

    return if !$self->_constructed && $self->_has_sms_time;

    my %parts = map {
        my $predicate = "has_$_";
        ( $_ ne $field && $self->$predicate ) ? ( $_ => $self->$_ ) : ();
    } @SMS_TIME_PARTS;

    $parts{ $field } = $new;

    # сериализуем только если есть все требуемые значения
    if ( keys( %parts ) == scalar( @SMS_TIME_PARTS ) ) {
        # срез хэша обеспечит порядок значений
        $self->_sms_time( join( ':' => @parts{ @SMS_TIME_PARTS } ) );
    }
}

=head2 supported_type

    Тип кампании, описанный в модели
    Необходимо переопределить в подклассе

=cut

sub supported_type {
    croak "Not implemented in base class";
}

=head2 to_hash

    Хэш с данными для клиентского интерфейса.

=cut
sub to_hash {
    my ($self) = @_;
    my $hash = $self->SUPER::to_hash;
    if (exists $hash->{id} && defined $hash->{id}) {
        $hash->{id} = int($hash->{id});
    }
    return $hash;
}


=head2 is_media_camp

Баян?

=cut

sub is_media_camp {
    my $self = shift;
    return $self->campaign_type eq 'mcb';
}

=head2 get_strategy

Возвращает текущую стратегию кампании

=cut

sub get_strategy {
    my $self = shift;

    return undef if !$self->_has_strategy;
    return $self->_strategy;
}



=head2 set_strategy

Устанавливает новую стратегию на кампанию.
Если новая совпадает с текущей - ничего не делает.

Опции:
    has_edit_avg_cpm_without_restart_enabled - включена фича редактирования параметров стратегии без рестарта

=cut

sub set_strategy {
    my ($self, $strategy, %opt) = @_;

    my $strategy_has_changes = !$self->_has_strategy || $strategy->is_differs_from($self->_strategy);

    if ($opt{is_attribution_model_changed} || $strategy_has_changes) {
        # если в кампании изменилась цель или модель аттрибуции сбарсываем время рестарта стратегии
        if ($opt{has_conversion_strategy_learning_status_enabled}) {
            $strategy->process_conversion_strategy_last_bidder_restart_time($self->_strategy,
                $opt{is_attribution_model_changed});
        }
        if (!$strategy_has_changes) {
            $self->_strategy_data($strategy->get_strategy_json());
        }
    }

    if ($strategy_has_changes) {
        # проверяем событие "смена параметров стратегии без рестарта"
        if ($opt{has_edit_avg_cpm_without_restart_enabled}) {
            $strategy->process_strategy_update_time_fields($self->_strategy);
        }
        $self->strategy_name($strategy->name);
        $self->_strategy_data($strategy->get_strategy_json());

        ## strategy-storage
        ## пока оставляем, для поддержки autobudget, autobudget_date
        ## и старого поведения camp_options.strategy
        $self->reset_strategy_fields();
        $strategy->set_camp_values($self);
        ##

        $self->status_bs_synced('No');

        if ($strategy->is_autobudget && !$self->is_media_camp) {
            $self->status_autobudget_forecast('New');
            $self->autobudget_forecast_date(undef);

            my $time_target = $self->has_time_target ? $self->time_target : undef;
            $self->time_target(TimeTarget::clear_extended_timetarget($time_target));

            $self->do_resync_bids_retargeting(1);
        }
    }

    return;
}


=head2 reset_strategy_fields

Сброс полей, ответственных за описание стратегии

=cut

sub reset_strategy_fields {
    my $self = shift;

    $self->_strategy_name(undef); # ???
    $self->_autobudget('No');
    $self->_autobudget_date(undef);

    return;
}


=head2 get_strategy_app_hash($strategy)

Возвращаем параметры стратегии вместе с правилами её применения

=cut

sub get_strategy_app_hash {
    my $self = shift;
    my ($strategy, %O) = @_;

    $strategy = $self->get_strategy  if !$strategy;
    my $name = $self->_strategy_name || '';

    my $is_net_stop = $self->_detect_is_net_stop;
    my $is_search_stop = $self->_detect_is_search_stop;
    $name ||= 'different_places'  if $self->is_different_places || $is_search_stop;

    my $strategy_hash = $strategy->get_strategy_hash();

    my %app_hash = (
        name => $name,
        #strategy_name => $strategy->name,
        #platform => $self->platform,
        is_autobudget => $strategy->_as_num($strategy->is_autobudget),
        is_search_stop => $strategy->_as_num($is_search_stop),
        is_net_stop => $strategy->_as_num($is_net_stop),
    );

    if (!$is_search_stop) {
        $app_hash{search} = $strategy_hash;
        $app_hash{net} = { name => ($is_net_stop ? 'stop' : $self->is_different_places ? 'maximum_coverage' : 'default') },
    }
    else {
        $app_hash{search} = { name => 'stop' },
        $app_hash{net} = $self->is_different_places && !$strategy->is_autobudget
            ? {name => 'maximum_coverage'}
            : $strategy_hash;
    }

    return \%app_hash;
}


=head2 get_flat_strategy_app_hash

Возвращаем параметры стратегии в виде плоского хеша

=cut

sub get_flat_strategy_app_hash {
    my $self = shift;
    my ($strategy) = @_;

    return {
        %{ $strategy->get_flat_strategy_hash() },
        platform => $self->platform,
        _search_strategy_name => $self->_detect_is_search_stop ? 'stop' : $strategy->name,
        _net_strategy_name => $self->_detect_is_net_stop ? 'stop' : $strategy->name,
    };
}

# сравнение кц, без учета кавычек
sub _is_meaningful_goals_equal {
    my ($old, $new) = @_;
    return is_json_eq_without_quotes($old, $new);
}


1;
