package Direct::Model::Campaign::Manager;

use Direct::Modern;

use Mouse;

# use Campaigns qw/queue_camp_operation/;

use Direct::Model::Campaign;
use MetrikaCounters;
use MinusWords qw/mass_save_minus_words/;
use HierarchicalMultipliers;

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::DateTime qw/now/;
use Yandex::HashUtils;

extends 'Yandex::ORM::Model::Manager::Base';

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::Campaign]'
);

=head2 create

    Создание в БД записей для соответствующих объектов (кампаний).

=cut

sub create {
    my ( $self ) = @_;

    for my $chunk ( sharded_chunks( cid => $self->items, by => sub { $_->id } ) ) {
        my ( $shard, $shard_items ) = ( $chunk->{shard}, $chunk->{cid} );
        do_in_transaction {
            $self->_create_in_shard( $shard, $shard_items );
        };
    }

    $_->reset_state() for @{ $self->items };

    return;
}

sub _create_in_shard {
    my ( $self, $shard, $campaigns ) = @_;

    my @campaign_columns = Direct::Model::Campaign->get_db_columns_list('campaigns');
    $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'campaigns', \@campaign_columns, $campaigns);

    #важно сохранять минус слова до создания в camp_options
    $self->_do_save_minus_words($shard, $campaigns);

    my @camp_options_columns = Direct::Model::Campaign->get_db_columns_list('camp_options');
    $self->_insert_to_secondary_table_in_db(PPC(shard => $shard), 'camp_options', campaigns => 'cid', \@camp_options_columns, $campaigns);

    # Сохраним счётчики метрики
    $self->_do_insert_metrika_counters( $shard, $campaigns );

    # Сохраним коэффициенты
    $self->_save_hierarchical_multipliers( $shard, [ grep { $_->has_hierarchical_multipliers } @$campaigns ] );

    return;
}

sub _do_insert_metrika_counters {
    my ( $self, $shard, $campaigns ) = @_;

    return unless @$campaigns;

    my @data;
    for my $camp ( @$campaigns ) {
        next if ! $camp->has_metrika_counters;
        for my $counter_id ( @{ $camp->metrika_counters } ) {
            push @data, [ $camp->id, $counter_id ];
        }
    }

    if ( @data ) {
        do_mass_insert_sql( PPC(shard => $shard), 'INSERT IGNORE metrika_counters(cid, metrika_counter) VALUES %s', \@data );
    }
}

=head2 update

    Обновление в БД записей для соответствующих объектов (кампаний).

=cut

sub update {
    my ($self, %options) = @_;

    for my $chunk ( sharded_chunks( cid => $self->items, by => sub { $_->id } ) ) {
        my ( $shard, $shard_items ) = ( $chunk->{shard}, $chunk->{cid} );
        do_in_transaction {
            $self->_update_in_shard( $shard, $shard_items, %options );
        };
    }

    return;
}

sub _update_in_shard {
    my ( $self, $shard, $campaigns, %options ) = @_;
    
    for my $campaign (@$campaigns) {
        if (defined $campaign->do_update_last_change) {
            if ($campaign->do_update_last_change) {
                $campaign->last_change(DateTime::Format::MySQL->format_datetime(now()));
                $campaign->set_db_column_value('campaigns', 'LastChange', 'NOW()', dont_quote => 1);
            } else {
                $campaign->set_db_column_value('campaigns', 'LastChange', 'LastChange', dont_quote => 1);
            }
            $campaign->is_last_change_changed(1);
        }
    }
    $self->_update_one_table_in_db( PPC(shard => $shard), campaigns => 'cid', $campaigns, where => $options{where} );

    #важно сохранять минус слова до обновления в camp_options
    $self->_do_save_minus_words($shard, [
            grep { $_->has_minus_words && $_->is_minus_words_changed } @$campaigns
        ]);

    $self->_update_one_table_in_db( PPC(shard => $shard), camp_options => 'cid', $campaigns ); # TODO: тут должно использоваться что-то вроде $options{where}?
    $self->_do_save_metrika_goals($shard, $campaigns);
    $self->_do_resync_bids_retargeting($shard, $campaigns);

    # Сохраним счётчики метрики
    $self->_do_update_metrika_counters( $shard, $campaigns );

    # Сохраним коэффициенты на группы
    $self->_save_hierarchical_multipliers( $shard, [
        grep { $_->has_hierarchical_multipliers && $_->is_hierarchical_multipliers_changed } @$campaigns
    ]);

    $self->_do_enqueue_for_delete( $shard, $campaigns );

    return;
}

# TODO :
#   * реализовать при переделке Direct::Campaigns на новых моделях:
#       - нужен флаг для скидывания statusBsSynced на баннерах при изменении списка счётчиков
#       - нужно удалять меченые телефоны (мы получаем их по номеру счетчика) при удалении счетчика

sub _do_update_metrika_counters {
    my ( $self, $shard, $campaigns ) = @_;

    return unless @$campaigns;

    my ( @cids, @add );
    for my $camp ( @$campaigns ) {

        next if ! $camp->has_metrika_counters || ! $camp->is_metrika_counters_changed;

        for my $counter_id ( @{ $camp->metrika_counters // [] } ) {
            push @cids, $camp->id;
            push @add, [ $camp->id, $counter_id ];
        }
    }

    if ( @add ) {
        do_delete_from_table( PPC(shard => $shard), 'metrika_counters', where => { cid => \@cids } );
        do_mass_insert_sql( PPC(shard => $shard), 'INSERT IGNORE metrika_counters(cid, metrika_counter) VALUES %s', \@add );
    }
}

sub _save_hierarchical_multipliers {
    my ( $self, $shard, $campaigns ) = @_;

    return unless @$campaigns;

    # Сохранение единого набора коэффициентов
    for my $camp ( @$campaigns ) {
        my %params_data = (campaign_id => $camp->id, campaign_type => $camp->campaign_type);
        my $multipliers = hash_cut(
            $camp->hierarchical_multipliers,
            HierarchicalMultipliers::get_types_allowed_on_camp()
        );
        hash_merge \%params_data, $multipliers;

        my $result = JavaIntapi::UpdateBidModifiers->new(%params_data)->call();
    }

    return;
}

sub _do_enqueue_for_delete {
    my ($self, $shard, $campaigns) = @_;

    my @cids_to_delete = map { $_->id } grep { $_->do_enqueue_for_delete } @$campaigns;

    return unless @cids_to_delete;

    # queue_camp_operation('del', \@cids_to_delete);

    do_mass_insert_sql(
        PPC(shard => $shard),
        "INSERT IGNORE INTO camp_operations_queue(cid, operation) VALUES %s",
        [map { [$_, 'del'] } @cids_to_delete]
    );

    return;
}

sub _do_save_minus_words {
    my ($self, $shard, $campaigns) = @_;
    return unless @$campaigns;

    my $minus_words_hash2id = {};
    my %client_minus_words;

    for my $campaign (@$campaigns) {
        if (!@{$campaign->minus_words}) {
            # Минус-слов нет
            $campaign->_mw_id(undef);
        } else {
            push @{$client_minus_words{$campaign->client_id}}, $campaign->minus_words;
        }
    }

    # Получим идентификаторы для минус-слов
    for my $client_id (keys %client_minus_words) {
        $minus_words_hash2id->{$client_id} = mass_save_minus_words($client_minus_words{$client_id}, $client_id);
    }

    for my $campaign (grep { @{$_->minus_words} } @$campaigns) {
        $campaign->_mw_id($minus_words_hash2id->{$campaign->client_id}->{$campaign->_minus_words_hash});
    }
}

sub _do_save_metrika_goals {
    my ($self, $shard, $campaigns) = @_;

    my @counters = map { $_->do_save_metrika_goals ? @{$_->metrika_counters} : ()  } @$campaigns;
    return unless @counters;

    my $counter_goals = MetrikaCounters::get_counters_goals(\@counters);
    my (@goals, @camp_goals);
    for my $camp (@$campaigns) {
        next unless $camp->do_save_metrika_goals;
        for my $counter (@{$camp->metrika_counters}) {
            croak "can't get metrika goals for counter $counter" unless defined $counter_goals->{$counter};
            for my $goal (@{$counter_goals->{$counter}}) {
                push @goals, [$goal->goal_id, $goal->goal_name, $goal->counter_status, $goal->goal_status, $goal->goal_type];
                push @camp_goals, [$camp->id, $goal->goal_id, $goal->goals_count, $goal->context_goals_count];
            }
        }
    }

    do_mass_insert_sql(
        PPCDICT,
        "INSERT IGNORE INTO metrika_goals(goal_id, name, counter_status, goal_status, goal_type)
            VALUES %s ON DUPLICATE KEY UPDATE
                name = VALUES(name),
                counter_status = VALUES(counter_status),
                goal_status = VALUES(goal_status),
                goal_type = VALUES(goal_type),
                logtime = NOW()",
        \@goals
    );
    do_mass_insert_sql(
        PPC(shard => $shard),
        "INSERT IGNORE INTO camp_metrika_goals(cid, goal_id, goals_count, context_goals_count)
            VALUES %s ON DUPLICATE KEY UPDATE
                goals_count = VALUES(goals_count),
                context_goals_count = VALUES(context_goals_count),
                stat_date = NOW()",
        \@camp_goals
    );
}


sub _do_resync_bids_retargeting {
    my $self = shift;
    my ($shard, $campaigns) = @_;

    my @cids = map {$_->id} grep {$_->do_resync_bids_retargeting} @$campaigns;
    return if !@cids;

    # todo: model?
    do_sql(PPC(shard => $shard), [
            "update bids_retargeting br
            join phrases p using(pid)
            set br.statusBsSynced = 'No', br.autobudgetPriority = 3",
            where => {'p.cid' => \@cids},
        ]);

    return;
}

1;
