package Direct::Model::CanvasHtml5Creative;

use Direct::Modern;

use List::MoreUtils qw/any/;
use Mouse;
use Mouse::Util::TypeConstraints;
use JSON;

use Direct::Model::CanvasCreativeScales;

extends 'Direct::Model::Creative';

# см. VALID_IMAGE_MIME_TYPES в
# https://github.yandex-team.ru/Canvas/canvas/blob/develop/pictorial_backend/html5/model.pytorial_backend/html5/model.py
my @IMAGE_TYPES = qw/gif jpg png/;

enum('source_media_type_enum', [@IMAGE_TYPES]);

has '+creative_type'       => (isa => enum([qw/html5_creative/]), lazy => 0, builder => sub { 'html5_creative' });
has '+template_id'         => (isa => 'Undef');
has '+template_name'       => (isa => 'Undef', lazy => 0, builder => sub { undef });
has '+layout_id'           => (isa => 'Maybe[Int]');
has '+layout'              => (isa => 'Undef', lazy => 0, builder => sub { undef });
has '+preview_scale'       => (lazy => 0, builder => sub { 0.5 });
has '+source_media_type'   => (isa => 'Maybe[source_media_type_enum]', lazy => 0, builder => sub { undef } );

sub to_template_hash {
    my $self = shift;

    my $hash = $self->to_hash;
    $hash->{rejection_reason_ids} = [];
    $hash->{creative_id} = delete $hash->{id};
    $hash->{bs_template_name} = undef;
    $hash->{composed_from} = $self->composed_from;
    $hash->{scale} = $self->get_scale_size();
    return $hash;
}



=head2 composed_from

    Из чего сделан креатив, нужно на фронте

=cut

sub composed_from {
    return shift->is_image ? 'image' : '';
}

sub get_scale_size {
    my $self = shift;
    return Direct::Model::CanvasCreativeScales::size2scale($self->width . 'x' . $self->height);
}

sub yabs_data {
    my $self = shift;
    if (@_) {
        my $yabs_data = shift;
        $self->_yabs_data(defined $yabs_data ? JSON->new->utf8(0)->encode($yabs_data) : undef);
        return $yabs_data;
    }

    return {} unless $self->_yabs_data;
    return JSON->new->utf8(0)->decode($self->_yabs_data);
}

sub detect_lang {
    return 'ru'; # https://st.yandex-team.ru/DIRECT-74879 
}

=head2 image_source_media_types

    Список source_media_type являющихся картинкой

    Используем
    * для передачи флага совместимости со старыми браузерами в БК
    * для передачи в модерацию информации о том, что сформировали креатив сами из примитивного источника (картинки)
    * для правильного отображения элементов управления на фронте (для картинко-баннера нужна одна ссылка вместо трех)

=cut

sub image_source_media_types {
    return @IMAGE_TYPES;
}

=head2 is_image

    true если баннер сформирован из изображения

=cut

sub is_image {
    my $self = shift;
    return unless $self->source_media_type;
    return any { $self->source_media_type eq $_ } $self->image_source_media_types;
}

1;
