package Direct::Model::Creative;

use Direct::Modern;
use Mouse;
use YAML::Syck qw//;
use Yandex::I18n;
use GeoTools qw/get_geo_names/;
use Yandex::HashUtils qw/hash_grep/;
use TextTools qw/iget_or/;
use BannerStorage::Dict;
use Direct::Model::CanvasCreativeScales qw//;

extends 'Yandex::ORM::Model::Base';

__PACKAGE__->_setup(
    default_table  => 'perf_creatives',

    fields => [
        id                        => { type => 'Id', column => 'creative_id', primary_key => 1 },
        client_id                 => { type => 'Id', column => 'ClientID' },
        stock_creative_id         => { type => 'Maybe[Id]' },
        creative_type             => { type => 'Enum', values => [qw/performance bannerstorage/] },
        business_type             => { type => 'Enum', values => [qw/retail hotels realty auto flights news other pharm/], default => 'retail' },
        name                      => { type => 'Str', length => 255 },
        width                     => { type => 'Int' },
        height                    => { type => 'Int' },
        alt_text                  => { type => 'Maybe[Str]', length => 255 },
        href                      => { type => 'Maybe[Str]', length => 1024 },
        preview_url               => { type => 'Str', length => 1024 },
        sum_geo                   => { type => 'Maybe[Str]' },
        status_moderate           => { type => 'Enum', values => [qw/New Ready Sending Sent Yes No Error AdminReject/], column => 'statusModerate', volatile => 1 },
        # менеджер работает на базовом типе и по всем полям, поэтому модель не должна падать при обращению к этому полю
        # При этом для генерации креативов фабрикой это поле нужно выбирать для всех, поэтому поле есть в базовом классе, но с дефолтным значением
        source_media_type         => { type => 'Undef', dummy_value => undef },
        moderate_send_time        => { type => 'Timestamp' },
        moderate_try_count        => { type => 'Int', default => 0 },
        _moderate_info            => { type => 'Maybe[Str]', column => 'moderate_info' },
        moderate_info_json        => { type => 'Maybe[Str]', column => 'moderate_info', dummy_value => undef },
        _additional_data          => { type => 'Maybe[Str]', column => 'additional_data' },
        template_id               => { type => 'Int' },
        version                   => { type => 'Maybe[Id]' },
        theme_id                  => { type => 'Maybe[Int]', },
        creative_group_id         => { type => 'Maybe[Int]', },
        group_create_time         => { type => 'Maybe[Timestamp]' },
        group_name                => { type => 'Maybe[Str]' },
        layout_id                 => { type => 'Maybe[Int]' },
        live_preview_url          => { type => 'Maybe[Str]' },
        moderation_comment        => { type => 'Maybe[Str]' },
        duration                  => { type => 'Maybe[Int]' },
        has_packshot              => { type => 'Maybe[Bool]', default => 0, coerce => 1 },
        is_adaptive               => { type => 'Bool', default => 0, column => 'is_adaptive'},
        is_bannerstorage_predeployed => { type => 'Bool', default => 0 },
    ],

    additional => [
        has_screenshot             => { type => 'Bool', default => 0 },
        preview_scale              => { type => 'Num', is => 'ro', builder => '_build_preview_scale' },
        template_name              => { type => 'Str', is => 'ro', builder => '_build_template_name' },
        theme                      => { type => 'Maybe[Str]', is => 'ro', builder => '_build_theme' },
        layout                     => { type => 'Maybe[Str]', ir => 'ro', builder => '_build_layout' },
        _mod_reason_yaml           => { type => 'Maybe[Str]', track_changes => 1 },
        has_banner_storage_reasons => { type => 'Bool', default => 0 },
    ],
);

# поле должно быть в @fields, т.к. мы по коду берем список полей из данного класса, по которому выгребаем
# данные креатиов для фабрики (которая может создать тип, для которое данное поле необходимо)
# При этом для fields у нас нет дефолта, поэтому при получении данных креатива
# из вне (BannerStorage например), когда мы пытаемся сохранить креатив в БД
# Manager тоже берет список полей из базовой модели и падает на попытке
# обратится к данному полю, потому что в источнике оно не было заполнено (и не
# должно было)
# Отсюда такая странная подпорка.

=head2 unstorable_attributes

список атрибутов, которые читаются из БД, но напрямую не модифицируются

=cut

sub unstorable_attributes {
    my ($class) = @_;
    croak 'is class-only method' if blessed($class);
    return [qw/moderate_send_time moderate_try_count sum_geo _moderate_info _additional_data/]
}

=head2 AVAILABLE_TEMPLATES

список доступных шаблонов креативов(один для всех пользователей)

=cut

use constant AVAILABLE_TEMPLATES => {
    # PRODUCTION
    320 => {name => iget_noop('Динамический баннер 728х90 "Много товаров с одним описанием и мозаикой"'), preview_scale => 0.5},
    330 => {name => iget_noop('Динамический баннер 240х400 "Много товаров с описаниями и каруселью"'), preview_scale => 0.5},
    331 => {name => iget_noop('Динамический баннер 240х400 "Много товаров с одним описанием и мозаикой"'), preview_scale => 0.5},
    332 => {name => iget_noop('Динамический баннер 240х400 "Один товар крупно с описанием и каруселью"'), preview_scale => 0.5},
    333 => {name => iget_noop('Динамический баннер 728х90 "Много товаров с описаниями и каруселью"'), preview_scale => 0.5},
    334 => {name => iget_noop('Динамический баннер 300x250 "Много товаров крупно с описанием по наведению и каруселью"'), preview_scale => 0.5},
    335 => {name => iget_noop('Динамический баннер 300x250 "Много товаров с описаниями и каруселью"'), preview_scale => 0.5},
    336 => {name => iget_noop('Динамический баннер 300x250 "Один товар крупно с описанием и каруселью"'), preview_scale => 0.5},
    337 => {name => iget_noop('Динамический баннер 728х90 "Много товаров крупно с описанием по наведению и каруселью"'), preview_scale => 0.5},
    692 => {name => iget_noop('Динамический баннер 240х400 "Большое изображение товара с описанием и каруселью"'), preview_scale => 0.5},
    694 => {name => iget_noop('Динамический баннер 300х250 "Большое изображение товара с описанием и каруселью"'), preview_scale => 0.5},
    558 => {name => iget_noop('Динамический баннер 728х90 "Много товаров с одним описанием и мозаикой"'), preview_scale => 0.5},
    706 => {name => iget_noop('Динамический баннер 300х600 "Много товаров с одним описанием и мозаикой"'), preview_scale => 0.5},
    707 => {name => iget_noop('Динамический баннер 300х500 "Много товаров с одним описанием и мозаикой"'), preview_scale => 0.5},
    708 => {name => iget_noop('Динамический баннер 970х250 "Много товаров с одним описанием и мозаикой"'), preview_scale => 0.5},
    713 => {name => iget_noop('Динамический баннер 240х600 "Много товаров с одним описанием и мозаикой"'), preview_scale => 0.5},
    1013 => {name => iget_noop('Видеоролик с интерактивом 2.0'), preview_scale => 0.5, feature_name => 'bannerstorage_template_1013_enabled'},
    1042 => {name => iget_noop('Видеоролик (пропуск на 15 сек)'), preview_scale => 0.5, feature_name => 'bannerstorage_template_1042_enabled'},
    1058 => {name => iget_noop('Видеоролик (до 30 сек), Moat'), preview_scale => 0.5, feature_name => 'bannerstorage_template_1058_enabled'},
    1060 => {name => iget_noop('Видеоролик до 30 сек. с интерактивом с верификаторами, пропуск с 15 сек'), preview_scale => 0.5, feature_name => 'bannerstorage_template_1060_enabled'},
    1062 => {name => iget_noop('Директ. Видеоролик (до 30 сек, параметризируемый), Moat'), preview_scale => 0.5, feature_name => 'bannerstorage_template_1062_enabled'},

    # TESTING
    560 => {name => iget_noop('Динамический баннер 240х400 "Много товаров с описаниями и каруселью"'), preview_scale => 0.5},
    553 => {name => iget_noop('Динамический баннер 240х400 "Много товаров с одним описанием и мозаикой"'), preview_scale => 0.5},
    561 => {name => iget_noop('Динамический баннер 240х400 "Один товар крупно с описанием и каруселью"'), preview_scale => 0.5},
    559 => {name => iget_noop('Динамический баннер 728х90 "Много товаров с описаниями и каруселью"'), preview_scale => 0.5},
    554 => {name => iget_noop('Динамический баннер 300x250 "Много товаров крупно с описанием по наведению и каруселью"'), preview_scale => 0.5},
    555 => {name => iget_noop('Динамический баннер 300x250 "Много товаров с описаниями и каруселью"'), preview_scale => 0.5},
    556 => {name => iget_noop('Динамический баннер 300x250 "Один товар крупно с описанием и каруселью"'), preview_scale => 0.5},
    557 => {name => iget_noop('Динамический баннер 728х90 "Много товаров крупно с описанием по наведению и каруселью"'), preview_scale => 0.5},

    748 => {name => iget_noop('Динамический баннер 240х400 "Недвижимость"'), preview_scale => 0.5},
    755 => {name => iget_noop('Динамический баннер 240x600 "Недвижимость"'), preview_scale => 0.5},
    756 => {name => iget_noop('Динамический баннер 300x250 "Недвижимость"'), preview_scale => 0.5},
    757 => {name => iget_noop('Динамический баннер 300x500 "Недвижимость"'), preview_scale => 0.5},
    758 => {name => iget_noop('Динамический баннер 300x600 "Недвижимость"'), preview_scale => 0.5},
    759 => {name => iget_noop('Динамический баннер 728x90 "Недвижимость"'), preview_scale => 0.5},
    760 => {name => iget_noop('Динамический баннер 1000x120 "Недвижимость"'), preview_scale => 0.4},

    #SUPER
    619 => {name => iget_noop('Динамический баннер 240x400 отели'), preview_scale => 0.5, only_for_super=>1},
    620 => {name => iget_noop('Динамический баннер 300x250 отели'), preview_scale => 0.5, only_for_super=>1},
    684 => {name => iget_noop('Динамический баннер 728x90 отели'),  preview_scale => 0.5, only_for_super=>1},
    547 => {name => iget_noop('Мультиролл, динамический'),          preview_scale => 0.5, only_for_super=>1},
    703 => {name => iget_noop('Динамический баннер 240х400 "Большое изображение товара с описанием и каруселью" (О сервисе)'), preview_scale => 0.5, only_for_super=>1},
    704 => {name => iget_noop('Динамический баннер 300х250 "Большое изображение товара с описанием и каруселью" (О сервисе)'), preview_scale => 0.5, only_for_super=>1},


    769 => {name => iget_noop('Динамический мобильный баннер 320х50 "С описанием и каруселью"'), preview_scale => 0.5, only_for_super=>1},
    770 => {name => iget_noop('Динамический мобильный баннер 320х50 "С кнопкой и ценой"'), preview_scale => 0.5, only_for_super=>1},
    771 => {name => iget_noop('Динамический баннер 240х400 (Дзен)'), preview_scale => 0.5, only_for_super=>1},
    772 => {name => iget_noop('Динамический баннер 300х250 (Дзен)'), preview_scale => 0.5, only_for_super=>1},

    812 => {name => iget_noop('Динамический баннер 240х400 "Авиабилеты"'), preview_scale => 0.5, only_for_super=>1},
    813 => {name => iget_noop('Динамический баннер 240х600 "Авиабилеты"'), preview_scale => 0.5, only_for_super=>1},
    814 => {name => iget_noop('Динамический баннер 300х500 "Авиабилеты"'), preview_scale => 0.5, only_for_super=>1},
    815 => {name => iget_noop('Динамический баннер 300х600 "Авиабилеты"'), preview_scale => 0.5, only_for_super=>1},
    816 => {name => iget_noop('Динамический баннер 300х250 "Авиабилеты"'), preview_scale => 0.5, only_for_super=>1},
    817 => {name => iget_noop('Динамический баннер 728х90 "Авиабилеты"'), preview_scale => 0.5, only_for_super=>1},
    818 => {name => iget_noop('Динамический баннер 1000х120 "Авиабилеты"'), preview_scale => 0.4, only_for_super=>1},
    819 => {name => iget_noop('Динамический баннер 970х250 "Авиабилеты"'), preview_scale => 0.5, only_for_super=>1},

   # NEW TEMPLATES
    782 => { only_for_super => 1 },
};

#Обозначение нефиксированного размера: DIRECT-72795
our $ADAPTIVE_SIZE = '0x0';
our $ADAPTIVE_SIZE_TITLE = iget_noop('Адаптивный');

our @AVAILABLE_SIZES = ($ADAPTIVE_SIZE, qw/728x90 240x400 300x250/);

=head2 all_templates

Возвращающий hashref на список доступных шаблонов

=cut

sub all_templates {
    my ($class, %opt) = @_;
    if ($opt{for_available_operations}) {
        return AVAILABLE_TEMPLATES;
    }
    return hash_grep { defined $_->{name} } AVAILABLE_TEMPLATES;
}


sub _build_preview_scale {
    my ($self) = @_;

    croak 'template_id not defined' unless $self->template_id();

    return $self->all_templates()->{$self->template_id()}->{preview_scale} // 0.5;
}

sub _build_template_name {
    my ($self) = @_;

    croak 'template_id not defined' unless $self->template_id();

    return iget_or($self->all_templates()->{$self->template_id()}->{name})
        || BannerStorage::Dict::get_dict('template')->{$self->template_id()}->{name}
        // croak 'template name not defined for template '.$self->template_id();
}

sub _build_theme {
    my $self = shift;

    return '' unless $self->has_theme_id() && $self->theme_id();

    return BannerStorage::Dict::get_dict('theme')->{$self->theme_id()}->{name};
}

sub _build_layout {
    my $self = shift;
    return '' unless $self->has_layout_id && $self->layout_id;

    return BannerStorage::Dict::get_dict('layout')->{$self->layout_id()}->{name};
}

override 'to_hash' => sub {
    my ($self) = @_;

    #Перед дампом в хеш инициализируем атрибуты с ленивой инициализацией
    $self->preview_scale();
    $self->template_name();
    $self->layout();
    $self->theme();

    my $hash = super();
    $hash->{geo_names} = $self->has_sum_geo && $self->sum_geo ? get_geo_names($self->sum_geo) : undef;
    return $hash;
};

=head2 rejection_reason_ids

Установка/получение причин отклонения креатива

rejection_reason_ids([1,2,3]) - установка причин отклонения
rejection_reason_ids - получение причин отклонения

=cut

sub rejection_reason_ids {
    my $self = shift;
    # set
    if (@_) {
        my @reasons_ids = sort {$a <=> $b} @{$_[0]};
        my @reasons_for_dump = map { +{id => $_} } @reasons_ids;
        $self->_mod_reason_yaml(@reasons_for_dump ?  YAML::Syck::Dump(\@reasons_for_dump) : undef);
        return @reasons_ids;
    }

    # отсутствие причин отклонения - это пустой массив ()
    return () unless $self->_has_mod_reason_yaml and defined $self->_mod_reason_yaml;

    my @reasons = @{YAML::Syck::Load($self->_mod_reason_yaml)};
    return map { $_->{id} } @reasons if @reasons && ref $reasons[0] eq 'HASH';
    return @reasons;
}

sub get_scale_size {
    my $self = shift;
    return Direct::Model::CanvasCreativeScales::size2scale($self->width . 'x' . $self->height);
}

sub to_template_hash {

    my $self = shift;

    my $hash = $self->to_hash;
    $hash->{rejection_reason_ids} = [$self->rejection_reason_ids];
    $hash->{creative_id} = delete $hash->{id};
    $hash->{bs_template_name} = delete $hash->{template_name};
    return $hash;
}

=head2 get_canvas_tags

Возвращает тег для канваса для этого креатива

=cut
sub get_canvas_tags {
    my $self = shift;
    return [];
}

1;
