package Direct::Model::Creative::Manager;

use Direct::Modern;
use Mouse;

extends 'Yandex::ORM::Model::Manager::Base';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils qw/xminus xisect/;
use List::MoreUtils qw/any/;
use Yandex::DateTime qw/now/;
use DateTime::Format::MySQL qw//;

=head2 items

Коллекция креативов, над которыми производятся операции
=cut

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::Creative]',
);

=head2 model_class

Класс модели
=cut

has 'model_class' => (
    is => 'ro',
    isa => 'Str',
    default => 'Direct::Model::Creative',
);

=head2 _client_creatives

HashRef { id_клиента => [креатив1, креатив2 ...], id_клиента2 => ... }
Используется для быстрого поиска креативов при обработке данных шарда
=cut

has '_client_creatives' => (
    is  => 'ro',
    isa => 'HashRef[ArrayRef[Direct::Model::Creative]]',
    lazy_build => 1,
);

=head2 _storable_attributes

Список сохраняемых в БД полей креативов.
Предполагается, что поле в БД имеет то же имя, что поле креатива,
исключение - _irregular_fields, см. ниже.
=cut

has '_storable_attributes' => (
    is => 'rw',
    isa => 'ArrayRef',
    lazy_build => 1,
);

=head2 _attr2field_name

HashRef задающий соответствие имен атрибутов креатива именам полей в БД,

=cut

has '_attr2field_name' => (
    is => 'ro',
    isa => 'HashRef',
    lazy_build => 1,
);

=head2 _update_triggers

При update'е,
для полей, значение которых задается не напрямую, а в зависимости от значений других полей
список { имя_поля => sql-код задающий значение, ...}

=cut

has '_update_triggers' => (
    is => 'ro',
    isa => 'HashRef',
    lazy_build => 1,
);

=head2 _update_with_triggers_flag

Флаг - добавлять/не добавлять при update'е заданные через
_update_triggers условно-определяемые поля к общему списку

=cut

has '_update_with_triggers_flag' => (
    is => 'rw',
    isa => 'Bool',
    default => 1,
);


=head2 Публичные методы
=cut

=head3 create_or_update

Создание/обновление креативов
usage: $manager->create_or_update();
    
=cut

sub create_or_update {
    my ($self) = @_;
    
    $self->_storable_attributes($self->_build__storable_attributes);
    $self->_update_with_triggers_flag(1);
    return $self->_create_or_update();
}

=head3 update_preview_url

Обновляет preview_url у креативов, другие поля при этом не меняются
usage: $manager->create_or_update();
    
=cut


sub update_preview_url {
    my ($self) = @_;

    $self->_update_with_triggers_flag(0);
    $self->_storable_attributes([qw/id preview_url/]);
    return $self->_create_or_update(without_insert => 1, without_bssync => 1);
}

=head2 Приватные методы
=cut

=head3 _create_or_update

Непосредственно производит создание и удаление записей
usage: $manager->_create_or_update( option_1=>val1, option_2=>val2);
Поддерживаемые опции:
    without_insert => 1|0 - производить/не производить добавление новых креативов в БД

=cut

sub _create_or_update {
    my ($self, %options) = @_;

    foreach_shard ClientID => [keys %{$self->_client_creatives}], sub {
        my ($shard, $shard_client_ids) = @_;
        my $creatives = $self->_prepare_creatives_data($shard => $shard_client_ids);
        do_in_transaction {
            $self->_insert( $shard => $creatives->{to_insert} ) if $creatives->{to_insert} && !$options{without_insert};
            $self->_update( $shard => $creatives->{to_update}, map {$_ => $options{$_}} qw/without_bssync/  ) if $creatives->{to_update};

            $self->_add_mod_reasons(  $shard => $creatives->{add_mod_reasons} )   if $creatives->{add_mod_reasons};
            $self->_erase_mod_reasons($shard => $creatives->{erase_mod_reasons} ) if $creatives->{erase_mod_reasons};
        };
    };
    return;
}


=head3 _prepare_creatives_data

Для заданного списка идентификаторов клиентов выбирает креативы из общей коллекции.
Для каждого из выбранных креативов определяет нужно ли создавать в БД новую запись или обновить существующую.

Подготавливает коллекции данных для последующей обработки.

Возвращает hashref
{
 to_insert => arrayref с данными для _insert,
 to_update => hashref с данными для _update
}
=cut

sub _prepare_creatives_data {
    my ($self, $shard, $client_ids) = @_;
    
    my $client_creatives = $self->_client_creatives();
    
    my (@creatives, @creative_ids);
    my (@erase_mod_reasons, @add_mod_reasons);
    for (@$client_ids) {
        push @creative_ids, map { $_->id } @{$client_creatives->{$_}};
        push @creatives, @{$client_creatives->{$_}};

        for my $c (@{$client_creatives->{$_}}) {
            my @ids = $c->rejection_reason_ids;
            if (@ids > 0) {
                push @add_mod_reasons, $c; 
            } else {
                push @erase_mod_reasons, $c->id;
            }
        }
 
    }
    my $exists_creatives = get_one_column_sql(PPC(shard => $shard),
                            ["SELECT creative_id FROM perf_creatives",
                                WHERE => {creative_id => \@creative_ids}]);

    my $new_creatives = xminus(\@creative_ids, $exists_creatives);

    my (@to_insert, @to_shcid_insert, %to_update);
    
    my %new_id = map { ($_ => undef) } @$new_creatives;

    for my $c (@creatives) {
        if (exists $new_id{$c->id}) {
            push @to_shcid_insert, [$c->client_id, $c->id];
            push @to_insert, [map { $c->$_() } @{$self->_storable_attributes()}];
        }
        else{
            $to_update{$c->id} = { map { $self->_db_field_name($_) => $c->$_() } @{$self->_storable_attributes()} };
            if (exists $c->{has_banner_storage_reasons} && $c->{has_banner_storage_reasons}) {
                # если из BannerStorage пришли причины, то сбрасываем {moderated_by_direct: true} в additional_data
                # чтобы причина отклонения на модерации выдавалась правильного типа
                $to_update{$c->id}->{additional_data} = undef;
            }
            if ($self->_update_with_triggers_flag()) {
                my $status_moderate = 'statusModerate';
                if (exists $to_update{$c->id}->{statusModerate}){
                    #В триггерах используем вместо записываемого в БД значения statusModerate (может быть изменено триггером) его оригинальное значение
                    $status_moderate = sql_quote($to_update{$c->id}->{statusModerate});
                }
                $to_update{$c->id}->{$_} = sprintf($self->_update_triggers()->{$_}, $status_moderate ) foreach keys %{$self->_update_triggers()};
            }
        }
    }

    
    return {
           @to_insert ? ( to_insert => [\@to_insert, \@to_shcid_insert, $new_creatives] ) : (),
           keys %to_update ? ( to_update => \%to_update) : (),
           @erase_mod_reasons ? (erase_mod_reasons => \@erase_mod_reasons) : (),
           @add_mod_reasons ? (add_mod_reasons => \@add_mod_reasons) : (),
    };
}

=head3 _insert

    Производит непосредственное добавление данных креативов в БД.
    Креативы, для которых уже есть записи в БД игнорируются.
    
    usage: $self->_insert( $shard, [
                [ [ creative_1_field_1, creative_1_field_2 ...], [ creative_2_field_1, creative_2_field_2 ...] ... ],
                [ [client_id_1, creative_id_1.1], [client_id_1, creative_id_1.2],... [client_id_2, creative_id_2.1], ... ],
                [ creative_id_1, creative_id_2 ...]
            ]);
    Возвращает число вставленных строк.

=cut

sub _insert {
    my ($self, $shard, $data) = @_;
    my ($creatives_data, $shcid_data, $new_cids) = @$data;
    
    do_mass_insert_sql(PPCDICT,
            'INSERT IGNORE INTO shard_creative_id(ClientID, creative_id) VALUES %s',
            $shcid_data,
            {max_row_for_insert => $self->_max_items_to_insert()},
    );
    
    my $fields = sql_fields( map { $self->_db_field_name($_) } @{$self->_storable_attributes()} );
    my $result = do_mass_insert_sql(PPC(shard => $shard),
            'INSERT IGNORE INTO perf_creatives ('. $fields .') VALUES %s',
            $creatives_data,
            {max_row_for_insert => $self->_max_items_to_insert()},
    );
    
    $self->_add_sync_tasks($shard, $new_cids);
    
    return $result;
}

=head3 _update

    Выполняет непосредственное обновление данных в БД
    usage: $self->_insert( $shard, {...} );
    {...} - структура для do_mass_update_sql
    Возвращает количество обновленных записей.

    Поддерживаемые опции:
        without_bssync - не создавать задания на синхронизацию креатива с Banner Storage

=cut

sub _update {
    my ($self, $shard, $ex_creatives_data, %options) = @_;

    my $is_changing_statusModerate = any {$_ eq 'status_moderate'} @{$self->_storable_attributes};
    my $exists_creatives;

    if ($is_changing_statusModerate) {
        $exists_creatives = get_hashes_hash_sql(PPC(shard => $shard),
            [ "SELECT creative_id, statusModerate, version FROM perf_creatives",
                WHERE => { creative_id => [ keys %$ex_creatives_data ] } ]);
    }

    my $byfield_options;
    if ($self->_update_with_triggers_flag) {
        $byfield_options = { map { $_ => {dont_quote_value => 1} } keys %{$self->_update_triggers() } };
    }

    my $result = do_mass_update_sql(PPC(shard => $shard),
        perf_creatives => 'creative_id',
        $ex_creatives_data,
        byfield_options => $byfield_options,
    );

    if ($is_changing_statusModerate) {
        my $current_creatives = get_all_sql(PPC(shard => $shard),
            [ "SELECT creative_id, statusModerate, version FROM perf_creatives",
                WHERE => { 'statusModerate__ne' => 'New',
                    creative_id                     => [ keys %$ex_creatives_data ]
                } ]);

        my @creatives_to_banner_bssync = map {$_->{creative_id}} grep {
            $exists_creatives->{$_->{creative_id}}->{statusModerate} ne $_->{statusModerate} &&
                ($_->{statusModerate} eq 'Yes' || $_->{statusModerate} eq 'No') ||
                defined $exists_creatives->{$_->{creative_id}}->{version} &&
                $_->{version} != $exists_creatives->{$_->{creative_id}}->{version}} @$current_creatives;
        #Чтобы не сбрасывать статус у баннеров, креативы у которых ранее не было значения поля version добавлено условие
        #defined $exists_creatives. Когда поле version будет заполнено условие можно убрать
        $self->bs_sync_banners($shard, \@creatives_to_banner_bssync);
    }

    $self->_add_sync_tasks($shard, [keys %$ex_creatives_data]) unless $options{without_bssync};



    return $result;
}

=head3 _add_sync_tasks

 Вызывает create_creative_sync_tasks, добавляя к полученному списку параметров full_sync=0 и max_rows из свойств инстанса

=cut

sub _add_sync_tasks {
    my ($self, $shard, $c_ids) = @_;

    my $full_sync = 0;
    $self->create_creative_sync_tasks($shard, $c_ids, $full_sync, $self->_max_items_to_insert());

    return;
}

=head3 create_creative_sync_tasks

 Для указанного списка идентификаторов креативов создает задания на синхронизацию.
 Если задание уже есть - оно помечается как новое и у него сбрасывается число выполненных попыток синхронизации
 usage: Direct::Model::Creative::Manager->create_creative_sync_tasks( $shard, [creative_id_1, creative_id_2 ...], $full_sync, $max_rows)
 
 $full_sync: 0|1 - выполнять при обработке задания только синхронизацию скриншотов или полную синхронизацию креатив
 $max_rows: максимальное число строк, которое можно вставить одним запросом
 
=cut

sub create_creative_sync_tasks {
    my ($class, $shard, $creative_ids, $full_sync, $max_rows) = @_;

    return do_mass_insert_sql(PPC(shard => $shard),
        qq/INSERT
            INTO creative_banner_storage_sync( creative_id, sync_status, attempts, full_sync )
            VALUES %s
        ON DUPLICATE KEY
            UPDATE sync_status = 'New', attempts=0, full_sync=values(full_sync)/,
        [map { [$_, 'New', 0, $full_sync] } @$creative_ids ],
        {max_row_for_insert => $max_rows}
    );
}


=head3 _db_field_name
    
    Для заданного имени поля креатива возвращает имя поля базы, в которое оно должно сохраняться
    usage: $db_field_name = $self->_db_field_name( 'creative_field_name')

=cut

sub _db_field_name {
    my ($self, $attr_name) = @_;
    
    my $field_name = $self->_attr2field_name()->{$attr_name};
    croak 'unknown field: '.$field_name unless $field_name;
    return $field_name;
}

=head3 _add_mod_reasons 

    Добавляем причины отклонения модерации для указанных креативов

=cut

sub _add_mod_reasons {
    my ($self, $shard, $creatives) = @_; 

    my $ctime = DateTime::Format::MySQL->format_datetime(now());
    my @data = map {
        my $c = $_;
        [$c->id, 'perf_creative', $c->client_id, 0, $c->status_moderate, $c->_mod_reason_yaml, $ctime];
    } @$creatives;
    do_mass_insert_sql(PPC(shard => $shard), 
        'INSERT INTO mod_reasons(id, type, ClientID, cid, statusModerate, reason, timeCreated)
        VALUES %s
        ON DUPLICATE KEY UPDATE
            reason=values(reason), statusModerate=values(statusModerate),
            timeCreated = NOW()',
        \@data);

    return;
}

=head3 _db_field_name
    
    Удаляем причины отклонения модерации для креативов, определяемых переданным списком id

=cut

sub _erase_mod_reasons {
    my ($self, $shard, $creative_ids) = @_; 

    do_delete_from_table(PPC(shard => $shard), 'mod_reasons',
        where => {id => $creative_ids, type => 'perf_creative'});

    return;
}

=head3 bs_sync_banners

    Сбрасываем statusBsSynced на всех не архивных баннерах креатива и не в архивных кампаниях

=cut

sub bs_sync_banners {
    my ($self, $shard, $creative_ids) = @_;

    my $banner_ids = get_one_column_sql(PPC(shard => $shard), [
            qq/SELECT bp.bid FROM banners_performance bp
            join banners b on bp.bid = b.bid
            join campaigns c on b.cid = c.cid/,
            WHERE => { 'b.statusArch' => 'No',
                'c.archived'          => 'No',
                'b.statusModerate__ne' => 'New',
                'bp.creative_id'      => $creative_ids }
        ]);

    if (scalar @$banner_ids > 0) {
        do_update_table(PPC(shard => $shard), 'banners',
            {
                statusBsSynced => 'No',
                LastChange__dont_quote => 'LastChange',
            }, where => { bid => $banner_ids });
    }

    return;
}

=head3 Билдеры
=cut

sub _build__client_creatives {
    my ($self) = @_;

    my %client_creatives;
    push @{ $client_creatives{$_->client_id} }, $_ foreach @{$self->items};
    return \%client_creatives;
}

sub _build__storable_attributes {
    my ($self) = @_;
    return xminus( [keys %{$self->_attr2field_name()}], $self->model_class()->unstorable_attributes() );
}

sub _build__attr2field_name {
    my ($self) = @_;
    my %fields = map {( $_->{attr_name} => $_->{alias} ) } values %{$self->model_class->get_table_schema('perf_creatives')};
    return \%fields;
}

sub _build__update_triggers {
    # Placeholder'ы заменяются на значение statusModerate если statusModerate меняется запросом, или на поле statusModerate,
    # если это поле явным образом не апдейтится
    return {
        moderate_try_count => q/IF(sum_geo IS NOT NULL AND statusModerate != 'Ready' AND %1$s IN ('New', 'Ready'), 0, moderate_try_count)/,
        moderate_send_time => q/IF(sum_geo IS NOT NULL AND statusModerate = 'New', moderate_send_time, IF(sum_geo IS NOT NULL AND statusModerate NOT IN ('New', 'Ready') AND %1$s IN ('New', 'Ready'), NOW() + INTERVAL 10 MINUTE, moderate_send_time))/,
        statusModerate     => q/IF(sum_geo IS NOT NULL AND statusModerate = 'New', statusModerate, IF(sum_geo IS NOT NULL AND statusModerate != 'New' AND %1$s = 'New', 'Ready', %1$s ))/,
    }
}

1;
