package Direct::Model::DynamicCondition;

use Direct::Modern;
use Mouse;
use Mouse::Util::TypeConstraints;

use JSON;
use HashingTools qw/url_hash_utf8/;

use Direct::Model::DynamicCondition::Rule;
use Direct::Model::DynamicCondition::FeedRule;

extends 'Yandex::ORM::Model::Base';

subtype 'DynamicRulesArrayRef'
    => as 'ArrayRef[Direct::Model::DynamicCondition::Rule | Direct::Model::DynamicCondition::FeedRule]';

__PACKAGE__->_setup(
    default_table  => 'bids_dynamic',

    fields => [
        id                  => { type => 'Id', column => 'dyn_id', primary_key => 1 },
        adgroup_id          => { type => 'Id', column => 'pid' },
        price               => { type => 'Num', default => '0.00' },
        price_context       => { type => 'Num', default => '0.00' },
        autobudget_priority => { type => 'Maybe[Int]', column => 'autobudgetPriority' },
        status_bs_synced    => { type => 'Enum', values => [qw/No Sending Yes/], column => 'statusBsSynced', default => 'No', volatile => 1 },

        condition_id        => { type => 'Id', column => 'dyn_cond_id' },
        condition_name      => { type => 'Str', column => 'condition_name', length => 50, table => 'dynamic_conditions' },

        from_tab            => { type => 'Enum', values => [qw/tree condition all-products/], column => 'from_tab', default => 'condition', track_changes => 1 },

        # Hidden fields
        _opts               => { type => 'Str', column => 'opts', default => '' },
        _condition_hash     => { type => 'Num', column => 'condition_hash', table => 'dynamic_conditions' },
        _condition_json     => { type => 'Str', column => 'condition_json', table => 'dynamic_conditions' },
    ],

    additional => [
        is_suspended => { type => 'Bool', track_changes => 1, trigger => \&_on_is_suspended_changed },
        available => { type => 'Bool', trigger => \&_on_available_changed, builder => sub { 0 } },
        filter_type => { type => 'Str' },
        feed_id => { type => 'Maybe[Id]' },
    ],

    relations => [
        adgroup   => { type => 'Direct::Model::AdGroupDynamic' },
        condition => { type => 'DynamicRulesArrayRef', trigger => \&_on_condition_changed },
    ],

    state_flags => [qw/
        bs_sync_banners
        bs_sync_adgroup
        moderate_adgroup
        set_adgroup_bl_status
        update_adgroup_last_change
        freeze_autobudget_alert
    /],
);

sub dyn_id { shift->id(@_) }
sub dyn_cond_id { shift->condition_id(@_) }

around BUILDARGS => sub {
    my ($orig, $class) = (shift, shift);
    my %args = @_ == 1 && ref($_[0]) eq 'HASH' ? %{$_[0]} : @_;

    $args{is_suspended} = scalar($args{_opts} =~ /\bsuspended\b/) if defined $args{_opts};

    if (defined $args{_condition_json}) {
        eval {
            $args{condition} = JSON->new->utf8(0)->decode($args{_condition_json});
            if (ref $args{condition} eq 'HASH'){
                my $available = delete $args{condition}->{available};
                $args{available} = $available ? 1 : 0;
            }
            1;
        } or do { croak "Cannot apply `condition_json`: $@"; };
    };
    
    my $filter_type = $args{filter_type};
    my $rule_class = $args{feed_id} ? 'Direct::Model::DynamicCondition::FeedRule' : 'Direct::Model::DynamicCondition::Rule';
    my $rule_ctor = $args{feed_id}
        ? sub { my $r = shift; $rule_class->from_json_key_val({ $r => $args{condition}->{$r} }, filter_type => $filter_type)  } 
        : sub { my $r = shift; $rule_class->from_json_key_val($r) }
        ;
    my @condition = ref $args{condition} eq 'HASH' ? ( sort keys %{$args{condition}} ) : @{$args{condition}//[]};
    # @condition = grep { ref $_ ne '' } @condition;
    $args{condition} = [ map { $rule_ctor->($_) } @condition ];
    
    $class->$orig(%args);
};

sub _on_is_suspended_changed {
    my ($self, $new) = @_;
    $self->_opts($new ? 'suspended' : '') if $self->_constructed || !$self->_has_opts;
}

sub _on_available_changed {
    my ($self) = @_;
    return $self->_on_condition_changed($self->condition // []);
}

sub _on_condition_changed {
    my ($self, $new) = @_;
    
    if ($self->_constructed || !$self->_has_condition_json ) {
        my $rule_type = $self->_get_rule_type();
        if ($rule_type eq 'Direct::Model::DynamicCondition::Rule') {
            $self->_condition_json(JSON->new->utf8(0)->canonical->encode([map { +{$_->to_json_key_val} } @$new]));
        }
        elsif ($rule_type eq 'Direct::Model::DynamicCondition::FeedRule'){
            my %condition = map { $_->to_json_key_val } @$new;
            $condition{'available'} = "true" if $self->available;
            $self->_condition_json(JSON->new->utf8(0)->canonical->encode(\%condition));
        }
        else {
            croak 'Unknown type of rule: ', $rule_type;
        }
    }
    $self->_condition_hash($self->_calc_condition_hash) if $self->_constructed || !$self->_has_condition_hash;
}

sub _get_rule_type {
    my ($self) = @_;
    return $self->adgroup->get_class_for_condition_rules() if $self->has_adgroup;
    unless (@{$self->condition}) {
        return $self->has_feed_id ? 'Direct::Model::DynamicCondition::FeedRule' : 'Direct::Model::DynamicCondition::Rule';
    }
    return ref $self->condition->[0];
}

sub _calc_condition_hash {
    my ($self) = @_;
    url_hash_utf8(JSON->new->utf8(0)->canonical->encode([ ($self->available ? (available => $self->available) : ()) , map { $_->to_hash } @{$self->condition}]));
}

=head2 is_deleted

Возвращает признак удаленности условия: отсутствует id (dyn_id) и есть condition_id (dyn_cond_id).
Имеет смысл для условий, полученных из БД (с полным набором полей).

=cut

sub is_deleted { return !$_[0]->has_id && $_[0]->has_condition_id }

sub is_condition_changed { shift->_is_condition_hash_changed(@_) }

sub get_condition_uhash {
    my ($self) = @_;

    my %rules_uniq;
    my $rule_type = $self->_get_rule_type();
    if ($rule_type eq 'Direct::Model::DynamicCondition::Rule') {
        $rules_uniq{$_->{type}}->{$_->{kind} // ''}->{join('', sort @{$_->{value} // []})} = 1 for map { $_->to_hash } @{$self->condition};
    }
    elsif ($rule_type eq 'Direct::Model::DynamicCondition::FeedRule'){
        $rules_uniq{$_->{field}}->{$_->{relation} // ''}->{ref $_->{value} ? join('', sort @{$_->{value} // []}) : $_->{value}} = 1 for map { $_->to_hash } @{$self->condition};
        $rules_uniq{available} = $self->available if $self->available;
    }
    else {
        croak 'Unknown type of rule: ', $rule_type;
    }

    return url_hash_utf8(JSON->new->utf8(0)->canonical->encode(\%rules_uniq));
}

sub to_hash {
    my $dyn_cond = shift->SUPER::to_hash;
    delete @{$dyn_cond}{qw/_opts _condition_hash _condition_json/};
    return $dyn_cond;
}

sub to_template_hash {
    my $dyn_cond = shift->to_hash;
    $dyn_cond->{dyn_id} = delete $dyn_cond->{id};
    $dyn_cond->{dyn_cond_id} = delete $dyn_cond->{condition_id};
    $dyn_cond->{autobudgetPriority} = delete $dyn_cond->{autobudget_priority} if exists $dyn_cond->{autobudget_priority};
    return $dyn_cond;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Direct::Model::DynamicCondition - Модель (класс) условий нацеливания для динамических групп

=cut
