package Direct::Model::DynamicCondition::Manager;

use Direct::Modern;
use Mouse;

extends 'Direct::Model::ShowCondition::Manager';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use AutobudgetAlerts;

use List::MoreUtils qw/uniq/;

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::DynamicCondition]',
);

=head2 create

Создание в БД записей для соответствующих объектов (условий нацеливания).

=cut

sub create {
    my ($self) = @_;

    my @bids_dynamic_columns = Direct::Model::DynamicCondition->get_db_columns_list('bids_dynamic');

    # Обработаем динамические условия по шардам
    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $dyn_conds_in_shard) = ($chunk->{shard}, $chunk->{pid});

        do_in_transaction {
            # Сохраним условия (без ставок)
            $self->_save_dynamic_conditions($shard, $dyn_conds_in_shard);

            # Сохраним ставки
            $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'bids_dynamic', \@bids_dynamic_columns, $dyn_conds_in_shard);

            # Сохраним дополнительные параметры фильтров: `from_tab`
            $self->_save_additional_params($shard, $dyn_conds_in_shard, 'create');

            # Обработаем флаги
            $self->_do_update_adgroups($shard, $dyn_conds_in_shard);
            $self->_do_update_banners($shard, $dyn_conds_in_shard);
        };
    };

    $_->reset_state() for @{$self->items};

    return;
}

=head2 update

Обновление в БД записей для соответствующих объектов (условий нацеливания).

=cut

sub update {
    my ($self) = @_;

    # Обработаем динамические условия по шардам
    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $dyn_conds_in_shard) = ($chunk->{shard}, $chunk->{pid});

        # досюда должны доходить фильтры, которые уже дедуплицировали по условиям внутри группы.
        # если в одной группе будет несколько одинаковых условий, метод либо упадет, либо произведет фильтр с
        # нулевым dyn_cond_id
        # если в продакшене варнингов не найдется, нужно поменять на croak
        my %seen_condition_hashes;
        for (@$dyn_conds_in_shard) {
            if ($seen_condition_hashes{$_->adgroup_id} && $seen_condition_hashes{$_->adgroup_id}{$_->_condition_hash}) {
                warn "duplicate dynamic conditions in adgroup ".$_->adgroup_id;
            }
            $seen_condition_hashes{$_->adgroup_id}{$_->_condition_hash} = 1;
        }

        do_in_transaction {
            # Сохраним изменившееся условия (без ставок)
            $self->_save_dynamic_conditions($shard, [
                grep { $_->_is_condition_hash_changed || $_->is_condition_name_changed } @$dyn_conds_in_shard
            ]);

            # Сохраним ставки
            # Чтобы не поймать "Duplicate entry for key 'i_pid_dyn_cond_id'" когда одно из обновляемых динамических условий совпадает с другим до обновления
            # временно отключим проверку внешних ключей
            do_sql(PPC(shard => $shard), 'SET SESSION foreign_key_checks = 0');
            # и сделаем все старые значения dyn_cond_id, которые будут обновлены, уникальными 
            my @condition_id_changed = map {$_->id} grep { $_->is_condition_id_changed() } @$dyn_conds_in_shard;
            do_sql(PPC(shard => $shard),['UPDATE bids_dynamic SET dyn_cond_id=dyn_id', WHERE => {dyn_id => \@condition_id_changed} ]) if @condition_id_changed;
            $self->_update_one_table_in_db(PPC(shard => $shard), bids_dynamic => 'dyn_id', $dyn_conds_in_shard);

            # магия закончена, снова включаем проверку на внешние ключи
            do_sql(PPC(shard => $shard), 'SET SESSION foreign_key_checks = 1');
        

            # Сохраним дополнительные параметры фильтров: `from_tab`
            $self->_save_additional_params($shard, $dyn_conds_in_shard, 'update');

            # Обработаем флаги
            $self->_do_update_adgroups($shard, $dyn_conds_in_shard);
            $self->_do_update_banners($shard, $dyn_conds_in_shard);
            $self->_proceed_empty_adgroups_and_banners($shard, $dyn_conds_in_shard);
        };
    };

    $_->reset_state() for @{$self->items};

    return;
}

=head2 delete

Удаление в БД записей для соответствующих объектов (условий нацеливания).

=cut

sub delete {
    my ($self) = @_;

    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $dyn_conds_in_shard) = ($chunk->{shard}, $chunk->{pid});

        do_in_transaction {
            # Ограничений на удаление из bids_dynamic у нас нет
            do_delete_from_table(PPC(shard => $shard), 'bids_dynamic', where => {dyn_id => [map { $_->id } @$dyn_conds_in_shard]});

            # Обработаем флаги
            $self->_do_update_adgroups($shard, $dyn_conds_in_shard);
            $self->_do_update_banners($shard, $dyn_conds_in_shard);
            $self->_proceed_empty_adgroups_and_banners($shard, $dyn_conds_in_shard);
        };
    };

    return;
}

sub _save_dynamic_conditions {
    my ($self, $shard, $dyn_conds) = @_;

    return unless @$dyn_conds;

    # Создадим хеш, вида: adgroup_id => {condition_hash => condition}
    my %dyn_conds_by_gid2hash;
    $dyn_conds_by_gid2hash{ $_->adgroup_id }{ $_->_condition_hash } = $_ for @$dyn_conds;

    # [Шаг 1] Выберем существующие условия нацеливания с группировкой по adgroup_id
    my $existent_dyn_conds = get_all_sql(PPC(shard => $shard), [
        "SELECT dyn_cond_id, pid, condition_hash, condition_name FROM dynamic_conditions",
        where => {
            _OR => [map { (_AND => { pid => $_, condition_hash => [keys %{$dyn_conds_by_gid2hash{$_}}] }) } keys %dyn_conds_by_gid2hash],
        }
    ]);
    my %existent_dyn_conds_by_gid2hash;
    $existent_dyn_conds_by_gid2hash{ $_->{pid} }{ $_->{condition_hash} } = $_ for @$existent_dyn_conds;

    # [Шаг 2.1] Определим условия для вставки
    my @dyn_conds_to_insert;
    for (@$dyn_conds) {
        if (exists $existent_dyn_conds_by_gid2hash{ $_->adgroup_id }{ $_->_condition_hash }) {
            my $db_dyn_cond = $existent_dyn_conds_by_gid2hash{ $_->adgroup_id }{ $_->_condition_hash };
            $_->condition_id($db_dyn_cond->{dyn_cond_id});
            # Пропускаем условия, уже содержащиеся в базе с тем же самым именем
            next if $_->condition_name eq $db_dyn_cond->{condition_name};
        } else {
            $_->condition_id(0);
        }

        push @dyn_conds_to_insert, $_;
    }

    return unless @dyn_conds_to_insert;

    # [Шаг 2.2] Вставка
    # Идентификаторы запрашиваем только для новых условий
    # Если у существующего условия изменилось имя, то вставляем в БД со старым идентификатором
    my $new_dyn_cond_ids = get_new_id_multi('dyn_cond_id', scalar(grep { !$_->condition_id } @dyn_conds_to_insert));
    do_mass_insert_sql(PPC(shard => $shard), q{
        INSERT INTO dynamic_conditions (dyn_cond_id, pid, condition_name, condition_hash, condition_json)
        VALUES %s ON DUPLICATE KEY UPDATE condition_name = VALUES(condition_name)
    }, [map { [
        $_->condition_id || shift(@$new_dyn_cond_ids),
        $_->adgroup_id, $_->condition_name, $_->_condition_hash, $_->_condition_json,
    ] } @dyn_conds_to_insert]);

    # [Шаг 3] Выберем вставленные условия
    %dyn_conds_by_gid2hash = ();
    $dyn_conds_by_gid2hash{ $_->adgroup_id }{ $_->_condition_hash } = $_ for @dyn_conds_to_insert;
    my $result_dyn_conds = get_all_sql(PPC(shard => $shard), [
        "SELECT dyn_cond_id, pid, condition_hash, condition_name FROM dynamic_conditions",
        where => {
            _OR => [map { (_AND => { pid => $_, condition_hash => [keys %{$dyn_conds_by_gid2hash{$_}}] }) } keys %dyn_conds_by_gid2hash],
        }
    ]);
    for (@$result_dyn_conds) {
        my $dyn_cond = $dyn_conds_by_gid2hash{ $_->{pid} }{ $_->{condition_hash} };
        $dyn_cond->condition_id($_->{dyn_cond_id});
        $dyn_cond->condition_name($_->{condition_name});
    }

    return;
}

sub _do_update_adgroups {
    my ($self, $shard, $dyn_conds) = @_;

    my (%adgroup_ids, %case_values, %cid_freeze_autobudget_alerts);
    for my $dyn_cond (@$dyn_conds) {
        my $adgroup_id = $dyn_cond->adgroup_id;

        # Принудительное изменение/сохранение времени модификации группы
        if (defined $dyn_cond->do_update_adgroup_last_change) {
            $adgroup_ids{phrases}->{$adgroup_id} = 1;
            $case_values{phrases}{LastChange}->{$adgroup_id} = $dyn_cond->do_update_adgroup_last_change ? 'NOW()' : 'LastChange';
        }

        # Отправка группы в БК
        if ($dyn_cond->do_bs_sync_adgroup) {
            $adgroup_ids{phrases}->{$adgroup_id} = 1;
            $case_values{phrases}{statusBsSynced}->{$adgroup_id} = sql_quote('No');
            $case_values{phrases}{LastChange}->{$adgroup_id} //= 'LastChange';
        }

        # Модерация группы
        if ($dyn_cond->do_moderate_adgroup) {
            $adgroup_ids{phrases}->{$adgroup_id} = 1;
            $case_values{phrases}{statusModerate}->{$adgroup_id} = sql_quote('Ready');
            $case_values{phrases}{statusPostModerate}->{$adgroup_id} = "IF(statusPostModerate = 'Rejected', 'Rejected', 'No')";
            $case_values{phrases}{LastChange}->{$adgroup_id} //= 'LastChange';
        }

        # Выставление статуса генерации фраз в BannerLand
        if (my $status = $dyn_cond->do_set_adgroup_bl_status) {
            $adgroup_ids{adgroups_dynamic}->{$adgroup_id} = 1;
            $case_values{adgroups_dynamic}{statusBlGenerated}->{$adgroup_id} = sql_quote($status);
        }

        if ($dyn_cond->do_freeze_autobudget_alert) {
            $cid_freeze_autobudget_alerts{$dyn_cond->adgroup->campaign_id} = 1;
        }
    }

    for my $table (keys %adgroup_ids) {
        my %case_values_table = %{$case_values{$table}};

        do_update_table(PPC(shard => $shard), $table, {
            (map { $_ => sql_case(pid => $case_values_table{$_}, default__dont_quote => $_, dont_quote_value => 1) } keys %case_values_table)
        }, where => {pid => [keys %{$adgroup_ids{$table}}]}, dont_quote => [keys %case_values_table]);
    }

    AutobudgetAlerts::update_on_new_phrases_add($_) for keys %cid_freeze_autobudget_alerts;

    return;
}

sub _do_update_banners {
    my ($self, $shard, $dyn_conds) = @_;
    if (my @adgroup_ids_to_bssync_banners = keys %{ +{map { $_->adgroup_id => 1 } grep { $_->do_bs_sync_banners } @$dyn_conds} }) {
        do_update_table(PPC(shard => $shard), 'banners', {
            statusBsSynced => 'No',
            LastChange__dont_quote => 'LastChange',
        }, where => {pid => \@adgroup_ids_to_bssync_banners}) if @adgroup_ids_to_bssync_banners;
    }
    return;
}

sub _proceed_empty_adgroups_and_banners {

    my ($self, $shard, $dyn_conds) = @_;
    # Проверяем не стали ли группы пустыми.
    my @uniq_adgroup_ids = uniq map { $_->adgroup_id } @$dyn_conds;
    my @empty_adgroup_ids = uniq @{get_one_column_sql(PPC(shard => $shard), ["select pid from phrases p", where => {'p.pid' => \@uniq_adgroup_ids},
                                                                    "and !exists(select 1 from bids_dynamic bi where bi.pid=p.pid and !FIND_IN_SET('suspended', bi.opts)) FOR UPDATE"])};

    # Пустым группам и баннерам сбрасываем bs_synced.
    $self->clear_bs_synced_for_adgroups(\@empty_adgroup_ids);

    return;
}

sub _save_additional_params {
    my ($self, $shard, $dyn_conds, $action) = @_;

    $dyn_conds = [grep { $_->has_from_tab } @$dyn_conds] if $action eq 'create';
    $dyn_conds = [grep { $_->has_from_tab && $_->is_from_tab_changed } @$dyn_conds] if $action eq 'update';
    return if !@$dyn_conds;

    my $key = 'dyn_cond:from_tab';
    my $pid2cid = get_hash_sql(PPC(shard => $shard), [
        'SELECT pid, cid FROM phrases', WHERE => {pid => [map {$_->adgroup_id} @$dyn_conds]}        
    ]);
    my %dyn_conditions_by_cid;
    push @{$dyn_conditions_by_cid{$pid2cid->{$_->adgroup_id}}}, $_ for @$dyn_conds;

    for my $cid (keys %dyn_conditions_by_cid) {
        my $from_tab_values = {};
        $from_tab_values->{$_->id} = { from_tab => $_->from_tab } for @{$dyn_conditions_by_cid{$cid}};
        do_mass_update_sql(PPC(shard => $shard), 'bids_dynamic', 'dyn_id', $from_tab_values);
    }

    return;
}

__PACKAGE__->meta->make_immutable;

1;
