package Direct::Model::Feed;

use Direct::Modern;
use Mouse;

extends 'Yandex::ORM::Model::Base';

use Direct::Encrypt qw/encrypt_text/;


=head2 BUSINESS_TYPE_VALUES

Константа: список доступных значений business_type фида

=cut

sub BUSINESS_TYPE_VALUES {
    return [qw/retail hotels realty auto flights news other/];
}

=head2 FEED_TYPE_VALUES

Константа: список доступных значений feed_type

=cut

sub FEED_TYPE_VALUES {
    return [qw/GoogleHotels GoogleFlights GoogleMerchant AutoRu YandexRealty TravelBooking CriteoTurkey AliExpress YandexMarket YandexZen Alibaba
            TurkeyTeknosa TurkeyJollyTur TurkeyClient1 TurkeyClient2 TurkeyClient3 TurkeyClient4 TurkeyClient5 TurkeyClient6 TurkeyClient7 TurkeyClient8
            TurkeyClient9 TurkeyClient10 TurkeyClient11 TurkeyClient12 TurkeyClient13 TurkeyClient14 TurkeyClient15 TurkeyClient16 TurkeyClient17
            TurkeyClient18 TurkeyClient19 TurkeyClient20 TurkeyClient21 TurkeyClient22 TurkeyClient23 TurkeyClient24 TurkeyClient25
            YandexCustom GoogleCustom GoogleTravel
            /];
}

=head2 VALID_TYPES_FOR_BUSINESS

    соответствие типов бизнеса и допустимых типов фидов
    подсмотрено в data3/desktop.blocks/i-utils/__feed-filter-data/i-utils__feed-filter-data.utils.js

=cut
sub VALID_TYPES_FOR_BUSINESS {
    return {
        retail => {
            map { $_ => 1 } qw/
                YandexMarket
                GoogleMerchant
                CriteoTurkey
                AliExpress
                Alibaba
                TurkeyTeknosa
                TurkeyJollyTur
                TurkeyClient1
                TurkeyClient2
                TurkeyClient3
                TurkeyClient4
                TurkeyClient5
                TurkeyClient6
                TurkeyClient7
                TurkeyClient8
                TurkeyClient9
                TurkeyClient10
                TurkeyClient11
                TurkeyClient12
                TurkeyClient13
                TurkeyClient14
                TurkeyClient15
                TurkeyClient16
                TurkeyClient17
                TurkeyClient18
                TurkeyClient19
                TurkeyClient20
                TurkeyClient21
                TurkeyClient22
                TurkeyClient23
                TurkeyClient24
                TurkeyClient25
            /
        },
        hotels => {
            map { $_ => 1 } qw/
                GoogleHotels
                TravelBooking
            /
        },
        auto => {
            AutoRu => 1,
        },
        realty => {
            YandexRealty => 1,
        },
        flights => {
            GoogleFlights => 1,
        },
        news => {
            YandexZen => 1,
        },
        other => {
            map { $_ => 1 } qw/
                YandexCustom
                GoogleCustom
                GoogleTravel
                YandexMarket
            /
        }
    };
}

=head2 UPDATE_STATUS_VALUES

Константа: список доступных значений update_status

=cut

sub UPDATE_STATUS_VALUES {
    return [qw/New Outdated Updating Done Error/];
}

__PACKAGE__->_setup(
    default_table  => 'feeds',

    fields => [
        id                         => { type => 'Id', column => 'feed_id', primary_key => 1 },
        client_id                  => { type => 'Id', column => 'ClientID' },
        feed_type                  => { type => 'Maybe[Enum]', values => FEED_TYPE_VALUES},
        business_type              => { type => 'Enum', values => BUSINESS_TYPE_VALUES, default => 'retail' },
        source                     => { type => 'Enum', values => [qw/url file site/] },
        name                       => { type => 'Str', length => 255 },
        url                        => { type => 'Maybe[Str]', length => 1024 },
        filename                   => { type => 'Maybe[Str]', length => 255 },
        login                      => { type => 'Maybe[Str]', length => 255 },
        encrypted_password         => { type => 'Maybe[Str]', length => 255 },
        email                      => { type => 'Maybe[Str]', length => 255 },
        refresh_interval           => { type => 'Int', default => 0 },
        is_remove_utm              => { type => 'Int', default => 0 },
        update_status              => { type => 'Enum', values => UPDATE_STATUS_VALUES, default => 'New' },
        cached_file_hash           => { type => 'Maybe[Str]', length => 22 },
        fetch_errors_count         => { type => 'Int', default => 0 },
        offers_count               => { type => 'Maybe[Int]' },
        last_change                => { type => 'Timestamp', column => 'LastChange' },
        last_refreshed             => { type => 'Maybe[Str]' },
        offer_examples             => { type => 'Maybe[Str]' },
    ],

    additional => [
        content                    => { type => 'Maybe[Str]' },
    ],

    relations => [
        campaigns                  => { type => 'ArrayRef[Direct::Model::Campaign]' },
        categories                 => { type => 'ArrayRef[HashRef]' },
        history                    => { type => 'ArrayRef[Direct::Model::FeedHistoryItem]' },
    ],

    state_flags => [qw/
        bs_sync_banners
    /],
);

=head2 set_password

Шифрует переданный пароль и сохраняет его в поле 'encrypted_password'.

=cut
sub set_password {
    my ($self, $password) = @_;
    $self->encrypted_password(defined $password ? encrypt_text($password) : undef);
}

=head2 add_history

Добавляет новый элемент в историю загрузок фида.

=cut
sub add_history {
    my ($self, $item) = @_;
    $self->history($self->has_history ? [@{$self->history}, $item] : [$item]);
}

=head2 add_campaign

Добавляет кампанию, в которой используется фид, к списку.

=cut
sub add_campaign {
    my ($self, $camp) = @_;
    $self->campaigns($self->has_campaigns ? [@{$self->campaigns}, $camp] : [$camp]);
}

=head2 to_template_hash

Хэш с данными для клиентского интерфейса.

=cut
sub to_template_hash {
    my ($self) = @_;
    my $hash = $self->to_hash;

    my @int_fields_not_null = qw/client_id id refresh_interval fetch_errors_count offers_count/;
    $hash->{$_} = int($hash->{$_} // 0) for grep { exists $hash->{$_} } @int_fields_not_null;

    $hash->{feed_id} = delete $hash->{id} if exists $hash->{id};

    if ($self->has_update_status) {
        $hash->{display_status} = $hash->{update_status};
    }

    return $hash;
}

1;
