package Direct::Model::Feed::Manager;

use Direct::Modern;
use Mouse;
use POSIX qw/strftime/;

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;

use Direct::Model::Feed;
use Direct::Model::FeedHistoryItem;

extends 'Yandex::ORM::Model::Manager::Base';

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::Feed]',
);

=head2 create

Создание в БД записей для соответствующих объектов (фидов).

=cut
sub create {
    my ($self) = @_;

    my $feed_ids = get_new_id_multi('feed_id', scalar(@{$self->items}));
    my @columns = Direct::Model::Feed->get_db_columns_list('feeds');

    for my $chunk (sharded_chunks(ClientID => $self->items, by => sub { $_->client_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{ClientID});

        $_->id(shift @$feed_ids) for @$shard_items;

        do_in_transaction {
            $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'feeds', \@columns, $shard_items);
            $self->_create_history($shard, $shard_items);
        };
    }

    $_->reset_state for @{$self->items};

    return;
}

=head2 update

Обновление в БД записей для соответствующих объектов (фидов).

=cut
sub update {
    my ($self) = @_;

    for my $chunk (sharded_chunks(ClientID => $self->items, by => sub { $_->client_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{ClientID});

        do_in_transaction {
            $self->_update_one_table_in_db(PPC(shard => $shard), feeds => 'feed_id', $shard_items);
            $self->_create_history($shard, $shard_items);
            $self->_do_update_banners($shard, $shard_items);
        };
    }

    $_->reset_state for @{$self->items};

    return;
}

=head2 _create_history

Создаёт записи в perf_feed_history для несохранённых элементов истории.

=cut
sub _create_history {
    my ($self, $shard, $feeds) = @_;

    my @new_history_items;
    for my $feed (@$feeds) {
        next unless $feed->has_history;

        for my $history_item (@{$feed->history}) {
            next if $history_item->has_id && $history_item->id;
            $history_item->feed_id($feed->id);
            $history_item->created_at(strftime('%Y-%m-%d %H:%M:%S', localtime())) if !$history_item->has_created_at;
            push @new_history_items, $history_item;
        }
    }

    return if !@new_history_items;

    # id - автоинкрементное поле, поэтому пропускаем его
    my @columns = grep { $_ ne 'id' } Direct::Model::FeedHistoryItem->get_db_columns_list('perf_feed_history');
    $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'perf_feed_history', \@columns, \@new_history_items);

    return;
}

sub _do_update_banners {
    my ($self, $shard, $feeds) = @_;

    $feeds = [grep { $_->do_bs_sync_banners } @$feeds];
    return if !@$feeds;

    my @feed_ids = map { $_->id } @$feeds;
    do_sql(PPC(shard => $shard), [q{
        UPDATE adgroups_performance gperf JOIN banners b ON (b.pid = gperf.pid)
        SET b.statusBsSynced = 'No', b.LastChange = b.LastChange
        }, WHERE => {
            'gperf.feed_id' => \@feed_ids,
        },
    ]);
    do_sql(PPC(shard => $shard), [
        "UPDATE adgroups_dynamic gd JOIN banners b ON b.pid = gd.pid
        SET b.statusBsSynced = 'No', b.LastChange = b.LastChange",
        where => {
            'gd.feed_id' => \@feed_ids,
        }
    ]);

    return;
}

1;
