package Direct::Model::ImageFormat;

use Direct::Modern;
use Mouse;
use HashingTools qw/md5_base64ya/;
use Direct::Validation::Image;
use BannerImages;

use Yandex::Avatars::MDS;

extends 'Yandex::ORM::Model::Base';

__PACKAGE__->_setup(
    default_table  => 'banner_images_formats',

    fields => [
        hash                      => { type => 'Str', column => 'image_hash', primary_key => 1 },
        mds_group_id              => { type => 'Maybe[Id]' },
        namespace                 => { type => 'Enum', values => ['direct' , 'direct-picture'], default => 'direct' },
        image_type                => { type => 'Maybe[Enum]', values => ['small','regular','wide','image_ad','logo'] },
        width                     => { type => 'Int' },
        height                    => { type => 'Int' },
        formats                   => { type => 'Str' },
        avatars_host              => { type => 'Enum', values => ['avatars.mdst.yandex.net','avatars.mds.yandex.net'] },
    ],

    additional => [
        image => { type => 'Str' },
        info  => { type => 'HashRef' },
        name  => { type => 'Maybe[Str]', },
        image_uri  => { type=> 'Str', is => 'ro', builder => '_build_image_uri' },
    ],
);

sub BUILD
{
    my ($self) = @_;
    if ($self->has_image) {
        if (!$self->has_hash) {
            $self->hash(md5_base64ya($self->image));
        }
        if (!$self->has_info) {
            $self->info(Direct::Validation::Image::check_image($self->image));
        }
        if (!$self->has_image_type && !$self->info->{error}) {
            $self->image_type($self->_guess_type_from_info());
        }
        if ($self->info->{error}) {
            return;
        }
        if (!$self->has_width) {
            $self->width($self->info->{width});
        }
        if (!$self->has_height) {
            $self->height($self->info->{height});
        }
    }
}

sub _guess_type_from_info
{
    my $self = shift;
    my ($width, $height) = ($self->info->{width}, $self->info->{height});
    my $type = BannerImages::banner_image_check_size($width, $height);
    $type = 'image_ad' if (!$type || $type eq 'small') && Direct::Validation::Image::is_image_size_valid_for_image_ad($width, $height);
    return $type;
}

sub _build_image_uri {
    my ($self) = @_;

    my $avatars = Yandex::Avatars::MDS->new(namespace => $self->{namespace}, get_host => $self->{avatars_host});
    return $avatars->avatars_url('get', short => 1)."/$self->{mds_group_id}/$self->{hash}/orig";
}

sub get_scale_size
{
    my $self = shift;
    my %size2scale = (
        '240x600' => 0.5,
        '300x250' => 0.8,
        '300x300' => 0.5,
        '300x500' => 0.5,
        '300x600' => 0.5,
        '320x480' => 0.5,
        '336x280' => 0.7,
        '480x320' => 0.5,
        '640x200' => 0.35,
        '640x960' => 0.35,
        '640x100' => 0.35,
        '728x90'  => 0.3,
        '960x640' => 0.25,
        '970x250' => 0.2,
        '1000x120' => 0.35,
    );
    my $wh = $self->width . 'x' . $self->height;
    return $size2scale{$wh} // 0.5;
}

=head2 size_matches

Проверяет, что размеры двух картинок совпадают

=cut

sub size_matches
{
    my ($self, $other) = @_;
    return $self->width == $other->width && $self->height == $other->height;
}

sub to_template_hash
{
    my $self = shift;
    return {
        hash => $self->hash,
        width => $self->width,
        height => $self->height,
        mds_group_id => $self->mds_group_id,
        group_id => $self->mds_group_id, # для совместимости со старым кодом фронта
        scale => $self->get_scale_size(),
        name => ( $self->has_name ? $self->name : '' ),
    }
}

sub to_stat_hash
{
    my $self = shift;
    my $hash = $self->to_template_hash();
    my $pool_item = {
        image_hash => $self->hash,
        image_type => $self->image_type,
        mds_group_id => $self->mds_group_id,
        namespace => $self->namespace,
    };
    $hash->{uri} = BannerImages::get_image_relative_url($pool_item);
    return $hash;
}

override 'to_hash' => sub {
    my ($self) = @_;

    # Перед дампом в хеш инициализируем атрибуты с ленивой инициализацией
    $self->image_uri();

    my $hash = super();
    return $hash;
};

1;
