package Direct::Model::Keyword::Manager;

use Direct::Modern;
use Mouse;

extends 'Direct::Model::ShowCondition::Manager';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use List::MoreUtils qw/uniq/;

use Models::AdGroup qw//;
use Common qw//;
use AutobudgetAlerts qw//;
use Primitives qw//;
use PrimitivesIds qw//;

use Direct::Model::Keyword;
use Direct::Banners;

has 'items' => (is  => 'ro', isa => 'ArrayRef[Direct::Model::Keyword]');

=head2 create

Создание в БД записей для соответствующих объектов (ключевых фраз).

=cut

sub create {
    my ($self) = @_;

    my @bids_columns = Direct::Model::Keyword->get_db_columns_list('bids');

    # Обработаем объекты по шардам
    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});

        do_in_transaction {
            $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'bids', \@bids_columns, $shard_items);

            # Сохраним дополнительные параметры фраз: param1/param2/bs_history
            $self->_save_additional_params($shard, $shard_items, 'create');

            # Обработаем флаги
            $self->_do_update_adgroups($shard, $shard_items);
            $self->_do_update_banners($shard, $shard_items);
            $self->_do_clear_auto_price_queue($shard, $shard_items);
            $self->_do_schedule_forecast($shard, $shard_items);
            $self->_do_freeze_autobudget_alert($shard, $shard_items);
        };
    }

    $_->reset_state() for @{$self->items};

    return;
}

=head2 update

Обновление в БД записей для соответствующих объектов (ключевых фраз).

=cut

sub update {
    my ($self) = @_;
    # Обработаем объекты по шардам
    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});

        do_in_transaction {
            $self->_update_one_table_in_db(PPC(shard => $shard), bids => 'id', $shard_items);

            # Сохраним дополнительные параметры фильтров: param1/param2/bs_history
            $self->_save_additional_params($shard, $shard_items, 'update');

            # Обработаем флаги
            $self->_do_update_adgroups($shard, $shard_items);
            $self->_do_update_banners($shard, $shard_items);
            $self->_proceed_empty_adgroups_and_banners($shard, $shard_items);
            $self->_do_clear_auto_price_queue($shard, $shard_items);
            $self->_do_schedule_forecast($shard, $shard_items);
            $self->_do_freeze_autobudget_alert($shard, $shard_items);
        };
    };

    $_->reset_state() for @{$self->items};

    return;
}

=head2 update

Удаление из БД записей для соответствующих объектов (ключевых фраз).

=cut

sub delete {
    my ($self) = @_;

    # Обработаем объекты по шардам
    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});

        do_in_transaction {
            my @kw_ids = map { $_->id } @$shard_items;
            my %cid2kw_ids;
            foreach my $keyword (@$shard_items) {
                push @{$cid2kw_ids{$keyword->campaign_id}}, $keyword->id;
            }

            for my $table (qw/bids_href_params bids_phraseid_history bids_manual_prices/) {
                do_delete_from_table(PPC(shard => $shard), $table, where => {
                    _OR => [map { (_AND => { cid => $_, id => $cid2kw_ids{$_} }) } keys %cid2kw_ids],
                });
            }

            do_delete_from_table(PPC(shard => $shard), 'bids', where => {id => \@kw_ids});

            # Обработаем флаги
            $self->_proceed_empty_adgroups_and_banners($shard, $shard_items);
            $self->_do_update_adgroups($shard, $shard_items);
            $self->_do_update_banners($shard, $shard_items);
            $self->_do_clear_auto_price_queue($shard, $shard_items);
            $self->_do_schedule_forecast($shard, $shard_items);
        };
    };

    $_->reset_state() for @{$self->items};

    return;
}

sub _save_additional_params {
    my ($self, $shard, $keywords, $action) = @_;

    if ($action eq 'create') {
        # bids_href_params
        $self->_insert_to_secondary_table_in_db(PPC(shard => $shard),
            'bids_href_params', 'bids' => [qw/cid id/],
            [Direct::Model::Keyword->get_db_columns_list('bids_href_params')],
            [grep { ($_->has_href_param1 && defined $_->href_param1) || ($_->has_href_param2 && defined $_->href_param2) } @$keywords],
        );

        # bids_phraseid_history
        $self->_insert_to_secondary_table_in_db(PPC(shard => $shard),
            'bids_phraseid_history', 'bids' => [qw/cid id/],
            [Direct::Model::Keyword->get_db_columns_list('bids_phraseid_history')],
            [grep { $_->has_bs_history && defined $_->bs_history } @$keywords],
        );
    }
    elsif ($action eq 'update') {
        # bids_href_params
        my @keywords__href_params = grep { $_->has_href_param1 && $_->has_href_param2 && ($_->is_href_param1_changed || $_->is_href_param2_changed) } @$keywords;
        if (@keywords__href_params) {
            do_mass_insert_sql(PPC(shard => $shard), q{
                INSERT INTO `bids_href_params` (`cid`, `id`, `param1`, `param2`)
                VALUES %s ON DUPLICATE KEY UPDATE
                    param1 = VALUES(param1),
                    param2 = VALUES(param2)
            }, [
                map { [$_->campaign_id, $_->id, $_->href_param1, $_->href_param2] } @keywords__href_params
            ], {sleep => 1, max_row_for_insert => 5000});
        }

        # bids_phraseid_history
        my @keywords__bs_history = grep { $_->has_bs_history && $_->is_bs_history_changed } @$keywords;

        if (my @rows = map { [$_->campaign_id, $_->id, $_->bs_history] } grep { defined $_->bs_history } @keywords__bs_history) {
            do_mass_insert_sql(PPC(shard => $shard), q{
                INSERT INTO `bids_phraseid_history` (`cid`, `id`, `phraseIdHistory`)
                VALUES %s ON DUPLICATE KEY UPDATE
                    phraseIdHistory = VALUES(phraseIdHistory),
                    update_time = NOW()
            }, \@rows, {sleep => 1, max_row_for_insert => 5000});
        }

        if (my @keywords_to_delete_bs_history = grep { !defined $_->bs_history } @keywords__bs_history) {
            my %keyword_ids_by_cid;
            push @{$keyword_ids_by_cid{ $_->campaign_id }}, $_->id for @keywords_to_delete_bs_history;

            do_delete_from_table(PPC(shard => $shard), 'bids_phraseid_history', where => {
                _OR => [map { (_AND => {cid => $_, id => $keyword_ids_by_cid{$_}}) } keys %keyword_ids_by_cid],
            });
        }
    }

    return;
}

sub _do_update_adgroups {
    my ($self, $shard, $keywords) = @_;

    my %update_adgroups;
    for my $keyword (@$keywords) {
        # Отправка группы в БК
        if ($keyword->do_bs_sync_adgroup) {
            $update_adgroups{$keyword->adgroup_id}->{statusBsSynced} = sql_quote('No');
            $update_adgroups{$keyword->adgroup_id}->{LastChange} = 'LastChange';
        }

        # Отправка группы на Модерацию
        if ($keyword->do_moderate_adgroup) {
            $update_adgroups{$keyword->adgroup_id}->{statusModerate} = sql_quote('Ready');
            $update_adgroups{$keyword->adgroup_id}->{statusPostModerate} = "IF(statusPostModerate = 'Rejected', 'Rejected', 'No')";
            $update_adgroups{$keyword->adgroup_id}->{LastChange} = 'LastChange';
        }

        # Состояние прогноза показов
        # Внимание! Флаг учитываение еще в функции _do_schedule_forecast
        if ($keyword->do_schedule_forecast) {
            $update_adgroups{$keyword->adgroup_id}->{statusShowsForecast} = sql_quote('New');
            $update_adgroups{$keyword->adgroup_id}->{LastChange} = 'LastChange';
        }

        # Изменение времени модификации группы (важно, чтобы условие было в конце)
        if ($keyword->do_update_adgroup_last_change) {
            $update_adgroups{$keyword->adgroup_id}->{LastChange} = 'NOW()';
        }

    }

    my %byfield_options;
    $byfield_options{$_} = {default__dont_quote => $_, dont_quote_value => 1} for map { keys %$_ } values %update_adgroups;

    do_mass_update_sql(PPC(shard => $shard), 'phrases', 'pid', \%update_adgroups, byfield_options => \%byfield_options);

    return;
}

sub _do_update_banners {
    my ($self, $shard, $keywords) = @_;

    if (my @gids = uniq(map { $_->adgroup_id } grep { $_->do_moderate_template_banners } @$keywords)) {
        # Отправка шаблонных баннеров на модерацию
        my $template_bids_by_gid = Models::AdGroup::adgroups_template_banners(\@gids, non_draft => 1, non_archived => 1);
        if (my @bids = map {@$_} values %$template_bids_by_gid) {
            do_update_table(PPC(shard => $shard), 
                'banners b
                LEFT JOIN banner_images bim on bim.bid = b.bid
                LEFT JOIN banner_display_hrefs bdh on bdh.bid = b.bid
                LEFT JOIN banners_performance bp on bp.bid = if(b.banner_type = "text", b.bid, NULL)
                ',
                {
                'b.statusModerate' => 'Ready',
                'b.phoneflag__dont_quote' => 'IF(b.vcard_id IS NOT NULL,"Ready","New")', # отправляем на модерацию визитку
                'b.statusSitelinksModerate__dont_quote' => 'IF(b.sitelinks_set_id IS NOT NULL,"Ready","New")', # отправляем на модерацию сайтлинки
                'b.statusBsSynced' => 'No',
                'b.statusPostModerate__dont_quote' => "IF(b.statusPostModerate = 'Rejected', 'Rejected', 'No')",
                'b.LastChange__dont_quote' => 'NOW()',
                'bim.statusModerate' => 'Ready', # отправляем на модерацию картинку
                'bdh.statusModerate' => 'Ready',
                'bp.statusModerate' => 'Ready', # отправляем видео-дополнение
            }, where => {
                'b.bid' => \@bids,
            });
            Direct::Banners::delete_minus_geo(bid => \@bids);
        }
    }

    # Удаление флажков постмодерации и автомодерации
    if (my @gids = uniq(map { $_->adgroup_id } grep { $_->do_clear_banners_moderation_flags } @$keywords)) {
        if (my @bids = @{PrimitivesIds::get_bids(pid => \@gids)}) {
            do_delete_from_table(PPC(shard => $shard), 'post_moderate', where => {bid => \@bids});
            do_delete_from_table(PPC(shard => $shard), 'auto_moderate', where => {bid => \@bids});
        }
    }

    return;
}

sub _do_clear_auto_price_queue {
    my ($self, $shard, $keywords) = @_;

    my @campaign_ids = uniq(map { $_->campaign_id } grep { $_->do_clear_auto_price_queue } @$keywords);
    Common::clear_auto_price_queue(\@campaign_ids) if @campaign_ids;

    return;
}

sub _do_schedule_forecast {
    my ($self, $shard, $keywords) = @_;

    # Внимание! Флаг еще учитывается в функции _do_update_adgroups (для обновления phrases.statusShowsForecast)

    my @campaign_ids = uniq(map { $_->campaign_id } grep { $_->do_schedule_forecast } @$keywords);
    Primitives::schedule_forecast_multi(\@campaign_ids) if @campaign_ids;

    return;
}

sub _do_freeze_autobudget_alert {
    my ($self, $shard, $keywords) = @_;

    my @campaign_ids = uniq(map { $_->campaign_id } grep { $_->do_freeze_autobudget_alert } @$keywords);
    AutobudgetAlerts::update_on_new_phrases_add($_) for @campaign_ids;

    return;
}

sub _proceed_empty_adgroups_and_banners {

    my ($self, $shard, $keywords) = @_;
    # Проверяем не стали ли группы пустыми.
    my @uniq_adgroup_ids = uniq map { $_->adgroup_id } @$keywords;

    my @empty_adgroup_ids = uniq @{get_one_column_sql(PPC(shard => $shard), ["select pid from phrases p", where => {'p.pid' => \@uniq_adgroup_ids},
                                                                    "and !(exists(select 1 from bids bi where bi.pid=p.pid and bi.is_suspended=0) ".
                                                                    "     OR exists(select 1 from bids_retargeting bir where bir.pid=p.pid and bir.is_suspended=0)) FOR UPDATE"])};
    # Пустым группам и баннеракм сбрасываем bs_synced.
    $self->clear_bs_synced_for_adgroups(\@empty_adgroup_ids);

    return;
}

__PACKAGE__->meta->make_immutable;

1;
