package Direct::Model::MobileContent;

use Direct::Modern;
use Mouse;

use MobileContent qw/icon_hash2url/;
use JSON;

extends 'Yandex::ORM::Model::Base';

__PACKAGE__->_setup(
    default_table  => 'mobile_content',

    fields => [
        id                      => { type => 'Id', column => 'mobile_content_id', primary_key => 1 },
        client_id               => { type => 'Id', column => 'ClientID', default => 0 },
        store_content_id        => { type => 'Str', length => 200 },
        store_country           => { type => 'Str', length => 2 },
        os_type                 => { type => 'Enum', values => [qw/Android iOS/] },
        content_type            => { type => 'Enum', values => [qw/app movie book music newsstand podcast/] },
        bundle_id               => { type => 'Maybe[Str]' },
        is_available            => { type => 'Bool', default => 0 },
        create_time             => { type => 'Timestamp' },
        modify_time             => { type => 'Timestamp' },
        status_bs_synced        => { type => 'Enum', values => [qw/No Sending Yes/], default => 'No', column => 'statusBsSynced', volatile => 1 },
        name                    => { type => 'Maybe[Str]', length => 200 },
        prices_json             => { type => 'Maybe[Str]' },
        rating                  => { type => 'Maybe[Num]' },
        rating_votes            => { type => 'Maybe[Int]' },
        icon_hash               => { type => 'Maybe[Str]' },
        icon_status_moderate    => { type => 'Enum', values => [qw/Yes No Sent Sending Ready/], column => 'statusIconModerate', default => 'Ready', volatile => 1 },
        min_os_version          => { type => 'Maybe[Str]', length => 10 },
        app_size_bytes          => { type => 'Maybe[Int]' },
        available_actions       => { type => 'Maybe[Set]', values => [qw/download update watch read play listen buy rent/] },
        publisher_domain_id     => { type => 'Maybe[Id]' },
        genre                   => { type => 'Maybe[Str]', length => 200 },
        age_label               => { type => 'Maybe[Enum]', values => [qw/18+ 16+ 12+ 6+ 0+/] },
    ],
    
    additional => [
        publisher_domain        => { type => 'Maybe[Str]' },
    ],
);

sub get_uniq_key {
    my ($self) = @_;
    return [$self->client_id, $self->store_content_id, $self->store_country, $self->os_type, $self->content_type];
}

sub icon_url {
    my ($self) = @_;
    return icon_hash2url($self->os_type, $self->icon_hash); 
}

sub get_country_prices {
    my ($self) = @_;
    return {} unless $self->prices_json;
    my $all_prices = from_json($self->prices_json);
    return $all_prices->{$self->store_country} || {}; 
}

=head2 get_store_app_id

Получить ключевой идентификатор приложения в сторе

Метод можно вызывать как статический
Direct::Model::MobileContent->get_store_app_id({
    os_type => ...,
    store_content_id => ...,
    bundle_id => ...,
})
и на экземпляре класса
$self->get_store_app_id()

Результат:
    $store_app_id - id приложения или undef если id не определен

=cut

sub get_store_app_id {
    my $content;
    if (ref $_[0] eq __PACKAGE__) {
        $content = {
            os_type => $_[0]->os_type,
            store_content_id => $_[0]->store_content_id,
            bundle_id => $_[0]->bundle_id, 
        }
    } else {
        $content = $_[1];
    } 
    
    if ($content->{os_type} eq 'Android') {
        return $content->{store_content_id};
    } elsif ($content->{os_type} eq 'iOS') {
        return $content->{bundle_id};
    }
    
    return undef;    
}

sub to_template_hash {
    
    my $self = shift;
    
    my $mobile_content = $self->to_hash;
    $mobile_content->{mobile_content_id} = $self->id; 
    $mobile_content->{icon_url} = $self->icon_url;
    $mobile_content->{is_show_icon} = $self->icon_status_moderate ne 'No' ? 1 : 0;
    $mobile_content->{prices} = $self->get_country_prices;
    $mobile_content->{available_actions} = $self->available_actions ? [@{$self->available_actions}] : [];
    $mobile_content->{store_app_id} = $self->get_store_app_id();
    delete @{$mobile_content}{qw/prices_json/};
    
    return $mobile_content;
}

1;
