package Direct::Model::PerformanceFilter;

use Direct::Modern;
use Mouse;
use Mouse::Util::TypeConstraints;

use Yandex::ORM::Helpers qw/mysql_timestamp_around is_json_eq_without_quotes/;

use JSON;
use HashingTools qw/url_hash_utf8/;

use Direct::Model::PerformanceFilter::Rule;

extends 'Yandex::ORM::Model::Base';

subtype 'PerformanceRulesArrayRef'
    => as 'ArrayRef[Direct::Model::PerformanceFilter::Rule]';
    
__PACKAGE__->_setup(
    default_table  => 'bids_performance',

    fields => [
        id                  => { type => 'Id', column => 'perf_filter_id', primary_key => 1 },
        adgroup_id          => { type => 'Id', column => 'pid' },
        filter_name         => { type => 'Str', column => 'name', length => 255 },
        price_cpc           => { type => 'Num', default => '0.00' },
        price_cpa           => { type => 'Num', default => '0.00' },
        autobudget_priority => { type => 'Maybe[Int]', column => 'autobudgetPriority' },
        target_funnel       => { type => 'Enum', values => [qw/same_products product_page_visit new_auditory/] },
        now_optimizing_by   => { type => 'Enum', values => [qw/CPC CPA/], default => 'CPC' },
        last_change         => { type => 'Timestamp', column => 'LastChange' },
        status_bs_synced    => { type => 'Enum', values => [qw/No Sending Yes/], column => 'statusBsSynced', default => 'No', volatile => 1 },
        is_suspended        => { type => 'Bool', default => 0 },
        is_deleted          => { type => 'Bool', default => 0 },
        ret_cond_id         => { type => 'Maybe[Id]' },
        from_tab            => { type => 'Enum', values => [qw/tree condition all-products/], column => 'from_tab', default => 'condition', track_changes => 1 },

        # Hidden fields
        _condition_json     => { type => 'Str', column => 'condition_json', custom_eq => \&_is_equal_condition_json },
    ],

    additional => [
        campaign_id         => { type => 'Id' },
        available           => { type => 'Bool', trigger => \&_on_condition_changed, builder => sub { 0 } },
        prev_id             => { type => 'Id', comment => 'Идентификатор предыдущего фильтра (в случае обновления и смены id)' },
        filter_type         => { type => 'Str' },
    ],

    relations => [
        adgroup         => { type => 'Direct::Model::AdGroupPerformance' },
        condition       => { type => 'PerformanceRulesArrayRef', trigger => \&_on_condition_changed, coerce => 1 },
        retargeting     => { type => 'Maybe[Direct::Model::RetargetingCondition]', trigger => \&_on_retargeting_changed },
    ],

    state_flags => [qw/
        bs_sync_banners
        bs_sync_adgroup
        set_adgroup_bl_status
        update_adgroup_last_change
        freeze_autobudget_alert
    /],
);

around BUILDARGS => sub {
    my ($orig, $class) = (shift, shift);
    my %args = @_ == 1 && ref($_[0]) eq 'HASH' ? %{$_[0]} : @_;

    if (defined $args{_condition_json}) {
        my $raw_condition;
        eval { $raw_condition = JSON->new->utf8(0)->decode($args{_condition_json}); 1; } or do { croak "Cannot apply `condition_json`: $@"; };
        if (exists $raw_condition->{available}){
            $args{available} = !!$raw_condition->{available};
            delete $raw_condition->{available};
        }
        $args{condition} = $raw_condition;
        $args{available} = 0 unless exists $args{available};
    }

    my $filter_type = $args{filter_type};
    if (ref $args{condition} eq 'HASH') {
        $args{condition} = [
            map { Direct::Model::PerformanceFilter::Rule->from_json_key_val({ $_ => $args{condition}->{$_} }, filter_type => $filter_type) }
            # костыль: на переходный период пропускаем categoryId для тех фильтров, у которых он не описан
            grep {$_ !~ /^categoryId\b/ || _is_category_allowed($filter_type)}
            sort keys %{$args{condition}} 
        ];
    }

    $class->$orig(%args);
};

sub _is_equal_condition_json {
    my ($old, $new) = @_;
    return is_json_eq_without_quotes($old, $new);
}


# delete after deploy
sub _is_category_allowed {
    my $filter_type = shift;

    state $is_allowed = {};
    return $is_allowed->{$filter_type}  if exists $is_allowed->{$filter_type};

    require FilterSchema;
    my $schema = FilterSchema->new(filter_type => $filter_type)->schema;

    $is_allowed->{$filter_type} = exists $schema->{definitions}->{rule}->{categoryId};
    return $is_allowed->{$filter_type};
}


sub _on_condition_changed {
    my ($self, $new) = @_;

    if ($self->_constructed || !$self->_has_condition_json) {
        my %condition = map { $_->to_json_key_val } ($self->has_condition ? @{$self->condition} : ());
        # serialize `available` attribute if true
        $condition{'available'} = "true" if $self->has_available && $self->available;
        $self->_condition_json(JSON->new->utf8(0)->canonical->encode(\%condition));
    }
}

sub is_condition_changed { shift->_is_condition_json_changed(@_) }

# Allow set to `now`
around last_change => mysql_timestamp_around('last_change');

sub _on_retargeting_changed {
    my ($self, $new) = @_;
    $self->ret_cond_id($self->retargeting ? $self->retargeting->id : undef);
}

sub to_hash {
    my $hash = shift->SUPER::to_hash;
    delete @{$hash}{qw/_condition_json/};
    return $hash;
}

sub to_template_hash {
    my $hash = shift->to_hash;
    $hash->{perf_filter_id} = int(delete $hash->{id}) if exists $hash->{id};
    $hash->{autobudgetPriority} = delete $hash->{autobudget_priority} if exists $hash->{autobudget_priority};
    return $hash;
}

sub get_condition_uhash {
    my ($self) = @_;
    my %rules_uniq;
 
    $rules_uniq{$_->{field}}->{$_->{relation} // ''}->{ref $_->{value} ? join("\n", sort @{$_->{value} // []}) : $_->{value} // "\n"} = 1 for map { $_->to_hash } @{$self->condition};
    $rules_uniq{_target_funnel} = $self->target_funnel if $self->has_target_funnel && $self->target_funnel;
    $rules_uniq{available} = $self->available if $self->available;
    $rules_uniq{ret_cond_id} = $self->ret_cond_id if $self->ret_cond_id;

    return url_hash_utf8(JSON->new->utf8(0)->canonical->encode(\%rules_uniq));
}

1;
