package Direct::Model::PerformanceFilter::Manager;

use Direct::Modern;
use Mouse;

extends 'Direct::Model::ShowCondition::Manager';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use AutobudgetAlerts;

use List::MoreUtils qw/uniq/;

use Direct::Model::PerformanceFilter;

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::PerformanceFilter]',
);

=head2 create

Создание в БД записей для соответствующих объектов (перфоманс фильтров).

=cut

sub create {
    my ($self) = @_;

    my @columns = Direct::Model::PerformanceFilter->get_db_columns_list('bids_performance');

    # Обработаем объекты по шардам
    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});
        do_in_transaction {
            $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'bids_performance', \@columns, $shard_items);

            # Сохраним дополнительные параметры фильтров: `from_tab`
            $self->_save_additional_params($shard, $shard_items, 'create');

            # Обработаем флаги
            $self->_do_update_adgroups($shard, $shard_items);
            $self->_do_update_banners($shard, $shard_items);
        };
    }

    $_->reset_state() for @{$self->items};

    return;
}

=head2 update

Обновление в БД записей для соответствующих объектов (перфоманс фильтров).

=cut

sub update {
    my ($self) = @_;

    # Обработаем объекты по шардам
    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});

        do_in_transaction {
            $self->_update_one_table_in_db(PPC(shard => $shard), bids_performance => 'perf_filter_id', $shard_items);

            # Сохраним дополнительные параметры фильтров: `from_tab`
            $self->_save_additional_params($shard, $shard_items, 'update');

            # Обработаем флаги
            $self->_do_update_adgroups($shard, $shard_items);
            $self->_do_update_banners($shard, $shard_items);
            $self->_proceed_empty_adgroups_and_banners($shard, $shard_items);
        };
    };

    $_->reset_state() for @{$self->items};

    return;
}

sub _save_additional_params {
    my ($self, $shard, $perf_filters, $action) = @_;

    $perf_filters = [grep { $_->has_from_tab } @$perf_filters] if $action eq 'create';
    $perf_filters = [grep { $_->has_from_tab && $_->is_from_tab_changed } @$perf_filters] if $action eq 'update';
    return if !@$perf_filters;

    my $key = 'perf_filter:from_tab';
    my %perf_filters_by_cid; push @{$perf_filters_by_cid{$_->campaign_id}}, $_ for @$perf_filters;

    for my $cid (keys %perf_filters_by_cid) {
        my $from_tab_values = {};
        $from_tab_values->{$_->id} = { from_tab => $_->from_tab } for @{$perf_filters_by_cid{$cid}};
        do_mass_update_sql(PPC(shard => $shard), 'bids_performance', 'perf_filter_id', $from_tab_values);
    }

    return;
}

sub _do_update_adgroups {
    my ($self, $shard, $perf_filters) = @_;

    my (%case_values, %update_fields, %cid_freeze_autobudget_alerts);
    for my $perf_filter (@$perf_filters) {
        my $adgroup_id = $perf_filter->adgroup_id;
        # Принудительное изменение/сохранение времени модификации группы
        if (defined $perf_filter->do_update_adgroup_last_change) {
            $case_values{phrases}{$adgroup_id}{LastChange} = $perf_filter->do_update_adgroup_last_change ? 'NOW()' : 'LastChange';
            $update_fields{phrases}{LastChange} = 1;
        }

        # Отправка группы в БК
        if ($perf_filter->do_bs_sync_adgroup) {
            $case_values{phrases}{$adgroup_id}{statusBsSynced} = sql_quote('No');
            $case_values{phrases}{$adgroup_id}{LastChange} //= 'LastChange';
            $update_fields{phrases}{LastChange} = $update_fields{phrases}{statusBsSynced} = 1;
        }

        if (my $status = $perf_filter->do_set_adgroup_bl_status) {
            $case_values{adgroups_performance}{$adgroup_id}{ statusBlGenerated } = sql_quote($status);
            $update_fields{adgroups_performance}{statusBlGenerated} = 1;
        }

        if ($perf_filter->do_freeze_autobudget_alert) {
            $cid_freeze_autobudget_alerts{$perf_filter->adgroup->campaign_id} = 1;
        }
    }

    for my $table (keys %case_values) {
        my %case_values_table = %{$case_values{$table}};
    
        AutobudgetAlerts::update_on_new_phrases_add($_) for keys %cid_freeze_autobudget_alerts;

        if (%case_values_table) {
            do_mass_update_sql(PPC(shard => $shard), $table, pid => \%case_values_table,
                byfield_options => { map { $_ => { dont_quote_value => 1 } } keys %{$update_fields{$table}} }
            );
        }
    }

    return;
}

sub _do_update_banners {
    my ($self, $shard, $perf_filters) = @_;

    if (my @adgroup_ids_to_bssync_banners = keys %{ +{map { $_->adgroup_id => 1 } grep { $_->do_bs_sync_banners } @$perf_filters} }) {
        do_update_table(PPC(shard => $shard), 'banners', {
            statusBsSynced => 'No',
            LastChange__dont_quote => 'LastChange',
        }, where => {pid => \@adgroup_ids_to_bssync_banners}) if @adgroup_ids_to_bssync_banners;
    }
    return;
}

sub _proceed_empty_adgroups_and_banners {

    my ($self, $shard, $keywords) = @_;
    # Проверяем не стали ли группы пустыми.
    my @uniq_adgroup_ids = uniq map { $_->adgroup_id } @$keywords;
    my @empty_adgroup_ids = uniq @{get_one_column_sql(PPC(shard => $shard), ["select pid from phrases p", where => {'p.pid' => \@uniq_adgroup_ids},
                                                                    "and !exists(select 1 from bids_performance bip where bip.pid=p.pid and bip.is_suspended=0 and bip.is_deleted=0) FOR UPDATE"])};

    # Пустым группам и баннерам сбрасываем bs_synced.
    $self->clear_bs_synced_for_adgroups(\@empty_adgroup_ids);

    return;
}


__PACKAGE__->meta->make_immutable;

1;
