package Direct::Model::Pixel;

use Direct::Modern;
use Direct::Model::AdGroup;

use List::Util qw/first/;
use List::MoreUtils qw/any/;

use Mouse;
use Mouse::Util::TypeConstraints;

extends 'Yandex::ORM::Model::Base';

use Yandex::URL qw/get_host/;
use Yandex::HashUtils qw/hash_merge/;

use constant KIND_AUDIT => 'audit';
use constant KIND_AUDIENCE => 'audience';

use constant ACCESS_FOR_ALL => 1;
use constant ACCESS_PERMISSIBLE => 2;

use constant CRITERION_TYPE_NONE => 'none';
use constant CRITERION_TYPE_PRIVATE => 'private';
use constant CRITERION_TYPE_PUBLIC => 'public';
use constant CRITERION_TYPE_YANDEX => 'yandex';

my $DEFAULT_RANDOM_PLACEHOLDER = '(\%(?:25)?)(?i:(?:aw_)?random)\1';

our %PROVIDERS_DATA = (
    yndx_audience => {
        name => 'Yandex.Audience',
        domain => 'mc.yandex.ru',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://mc\.yandex\.ru/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIENCE,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_PRIVATE,
    },
    adfox => {
        name => 'Adfox',
        domain_regexp => qr/^(ads\.adfox\.ru|amc\.yandex\.ru|mc\.admetrica\.ru)$/,
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr/^https:\/\/(?:ads\.adfox\.ru|amc\.yandex\.ru|mc\.admetrica\.ru)\/.+$DEFAULT_RANDOM_PLACEHOLDER/,
        kind => KIND_AUDIT,
        access => ACCESS_FOR_ALL,
    },
    tns => {
        name => 'TNS',
        domain_regexp => qr|^(?:.+\.)?tns-counter\.ru$|,
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://(?:.+\.)?tns-counter\.ru/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    adriver => {
        name => 'Adriver',
        domain => 'ad.adriver.ru',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://ad\.adriver\.ru/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    sizmek => {
        name => 'Sizmek',
        domain => 'bs.serving-sys.com',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://bs\.serving-sys\.com/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    weborama => {
        name => 'Weborama',
        domain_regexp => qr!^wcm(?:(?:-ru\.frontend)|\.solution)\.weborama\.fr$!,
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr!^https://wcm(?:(?:-ru\.frontend)|\.solution)\.weborama\.fr/.+$DEFAULT_RANDOM_PLACEHOLDER!,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    dcm => {
        name => 'DCM',
        domain => 'ad.doubleclick.net',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://ad\.doubleclick\.net/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    gemius => {
        name => 'Gemius',
        domain => 'gdeby.hit.gemius.pl',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://gdeby\.hit\.gemius\.pl/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    adjust => {
        name => 'Adjust',
        domain => 'view.adjust.com',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://view\.adjust\.com/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    mediascope => {
        name => 'Mediascope',
        domain => 'verify.yandex.ru',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://verify\.yandex\.ru/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    omi => {
        name => 'OMI',
        domain => 'verify.yandex.ru',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://verify\.yandex\.ru/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    mail_ru_top_100 => {
        name => 'Mail.ru top-100',
        domain_regexp => qr!(?:[a-z0-9A-Z\-]+\.)?mail\.ru$!,
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://(?:[a-z0-9A-Z\-]+\.)?mail\.ru/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    dv => {
        name => 'DoubleVerify',
        domain => 'tps.doubleverify.com',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://tps\.doubleverify\.com/visit\.jpg.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    ias => {
        name => 'IAS',
        domain => 'pixel.adsafeprotected.com',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://pixel\.adsafeprotected\.com/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    appsflyer => {
        name => 'AppsFlyer',
        domain => 'impression.appsflyer.com',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://impression\.appsflyer\.com/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
    adloox => {
        name => 'adloox',
        domain => 'pixel.adlooxtracking.com',
        random => $DEFAULT_RANDOM_PLACEHOLDER,
        regexp => qr|^https://pixel\.adlooxtracking\.com/.+$DEFAULT_RANDOM_PLACEHOLDER|,
        kind => KIND_AUDIT,
        access => ACCESS_PERMISSIBLE,
        criterion_type_for_unknown => CRITERION_TYPE_YANDEX,
    },
);

our $PROVIDER_BY_DOMAIN = { map { $PROVIDERS_DATA{$_}{domain} => $_ } grep { exists $PROVIDERS_DATA{$_}{domain} } keys %PROVIDERS_DATA };
our $PROVIDER_BY_DOMAIN_REGEXP = { map { $_ => $PROVIDERS_DATA{$_}{domain_regexp} } grep { exists $PROVIDERS_DATA{$_}{domain_regexp} } keys %PROVIDERS_DATA };

our $AUDIT_PROVIDERS = [ grep { $PROVIDERS_DATA{$_}{kind} eq KIND_AUDIT } keys %PROVIDERS_DATA ];
our $AUDIT_PROVIDERS_ACCESSIBLE_FOR_ALL = [ grep { $PROVIDERS_DATA{$_}{kind} eq KIND_AUDIT && $PROVIDERS_DATA{$_}{access} eq ACCESS_FOR_ALL } keys %PROVIDERS_DATA ];

our $PERMISSIBLE_PROVIDERS = [ grep { $_ ne 'omi' && $PROVIDERS_DATA{$_}{access} eq ACCESS_PERMISSIBLE } keys %PROVIDERS_DATA ];
our $PERMISSIBLE_CAMPAIGN_TYPES = [qw/cpm_banner cpm_deals cpm_yndx_frontpage cpm_price/];
our $PERMISSIBLE_CRITERION_TYPES = [CRITERION_TYPE_NONE, CRITERION_TYPE_PRIVATE, CRITERION_TYPE_PUBLIC, CRITERION_TYPE_YANDEX];

__PACKAGE__->_setup(
    default_table => 'banner_pixels',

    fields => [
        banner_id       => {type => 'Id', column => 'bid', primary_key => 1},
        id              => {type => 'Id', column => 'pixel_id', primary_key => 1},
        url             => {type => 'Str', column => 'pixel_url', length => 1024},
    ],
    additional => [
        kind => {type => 'Maybe[Enum]', values => [qw/audience audit/], builder => '_build_kind'},
        provider => {type => 'Maybe[Enum]', 
                     values => [qw/yndx_audience tns adfox adriver sizmek weborama dcm gemius mediascope omi adjust mail_ru_top_100 dv ias appsflyer adloox/],
                     builder => '_build_provider'},
        campaign_id => {type => 'Id'},
        adgroup_type => {type => Direct::Model::AdGroup::ADGROUP_TYPE}
    ],
);

sub _build_provider {
    my ($self) = @_;

    my $host = get_host($self->url) // '';
    return $PROVIDER_BY_DOMAIN->{$host} // (first { $host =~ $PROVIDER_BY_DOMAIN_REGEXP->{$_} } keys %$PROVIDER_BY_DOMAIN_REGEXP);
}


=head2  build_mediascope_audit_pixel($tmsec_prefix, $cid, $bid, $adgroup_typ)

   Сконструировать объект Direct::Model::Pixel для автоматического пикселя mediascope

=cut

sub build_mediascope_audit_pixel {
    my ($tmsec_prefix, $cid, $bid, $adgroup_type) = @_;

    my $pixel_url = sprintf 'https://{DRND}.verify.yandex.ru/verify?platformid=1&msid=%s_5-%d-%d', $tmsec_prefix, $cid, $bid;
    return Direct::Model::Pixel->new(
        banner_id => $bid, id => 1,
        url => $pixel_url,
        kind => 'audit', provider => 'mediascope',
        campaign_id => $cid, adgroup_type => $adgroup_type
    );
}

=head2  build_omi_audit_pixel($cid, $bid, $adgroup_typ)

   Сконструировать объект Direct::Model::Pixel для автоматического пикселя omi

=cut
sub build_omi_audit_pixel {
    my ($cid, $bid, $adgroup_type) = @_;

    my $pixel_url = sprintf 'https://verify.yandex.ru/verify?platformid=9';
    return Direct::Model::Pixel->new(
        banner_id => $bid, id => 1,
        url => $pixel_url,
        kind => 'audit', provider => 'omi',
        campaign_id => $cid, adgroup_type => $adgroup_type
    );
}
=head2 uniq_key()

    Возвращает строку, по которой идентифицируем совпадающие пикселы у клиента

=cut

sub uniq_key {
    my ($self, $banner_id, $url) = @_;
    return join ',', defined $banner_id ? ($banner_id, $url) : ($self->banner_id, $self->url);
}

=head2 _build_kind()

    Возвращает тип пиксела: пиксел аудиторий или аудита

=cut

sub _build_kind {
    my ($self) = @_;
    return if !$self->provider;
    return $PROVIDERS_DATA{$self->provider}{kind};
}

=head2 is_provider_matches_kind()

    Возвращает истину, если тип провайдера совпадает с заданным типом пиксела

=cut

sub is_provider_matches_kind() {
    my ($self) = @_;
    return $self->provider && $self->kind && $self->kind eq $self->_build_kind ? 1 : 0;
}

=head2 is_kind_audience()

    Возвращает истину для пиксела Я.Аудитории

=cut

sub is_kind_audience() {
    my ($self) = @_;
    return $self->kind && $self->kind eq KIND_AUDIENCE ? 1 : 0;
}

=head2 need_to_check_provider_permission($campaign_type, $is_yandex_page, $has_private_data)

    Возвращает истину если надо проверять разрешение для провайдера

=cut

sub need_to_check_provider_permission() {
    my ($self, $campaign_type, $is_yandex_page, $has_private_criterion) = @_;

    my $need_to_check_provider_permission;
    if ($PROVIDERS_DATA{$self->provider}{access} eq ACCESS_FOR_ALL) {
        $need_to_check_provider_permission = 0;
    } elsif ($self->kind eq KIND_AUDIENCE) {
        $need_to_check_provider_permission = ($campaign_type eq 'cpm_deals' && !$is_yandex_page) ? 1 : 0;
    } elsif ($campaign_type eq 'cpm_deals' && defined $is_yandex_page && !$is_yandex_page && !$has_private_criterion) {
        $need_to_check_provider_permission = 0;
    } else {
        $need_to_check_provider_permission = 1;
    }

    return $need_to_check_provider_permission;
}

=head2 is_valid_provider_url()

    Возвращает истину если задан корректный url для определенного провайдера

=cut

sub is_valid_provider_url() {
    my ($self) = @_;
    if ($self->provider) {
        return $self->url =~ $PROVIDERS_DATA{$self->provider}{regexp} ? 1 : 0;
    }
    return 0;
}

=head2 get_accessible_audit_provider_names($campaign_type, $is_yandex_page, $has_private_criterion, $permitted_providers)

    Возвращает список названий провайдеров пикселов аудита разрешеных клиенту

=cut

sub get_accessible_audit_provider_names() {
    my ($self, $campaign_type, $is_yandex_page, $has_private_criterion, $permitted_providers) = @_;
    my @accessible_providers = ($campaign_type eq 'cpm_deals' && defined $is_yandex_page && !$is_yandex_page && !$has_private_criterion)
        ? @$AUDIT_PROVIDERS
        : (@$AUDIT_PROVIDERS_ACCESSIBLE_FOR_ALL, grep { $PROVIDERS_DATA{$_}{kind} eq KIND_AUDIT } @$permitted_providers);
    return [ sort map { $PROVIDERS_DATA{$_}{name} } @accessible_providers ];
}


=head2 to_template_hash()

Преобразование из объекта в хеш, для отдачи фронтенду

=cut

sub to_template_hash {
    my ($self) = @_;

    my %hash;
    foreach my $field (qw/id url kind provider/) {
        $hash{$field} = $self->$field;
    }

    return \%hash;
}

=head2 url_with_subst_random()

    Возвращает url, с замененным для БК рандомным параметром

=cut

my %VERIFY_ADGROUP_TYPE = (
    cpm_video => 1,
    cpm_banner => 2,
    default => 0,
);

sub url_with_subst_random {
    my ($self) = @_;

    my $url = $self->url;
    $url =~ s/$PROVIDERS_DATA{$self->provider}{random}/\{DRND\}/g;

    my %additional_params;
    if (any {$self->provider eq $_} qw/mediascope omi/) {
        hash_merge \%additional_params, {
            page => '{PAGE}',
            BID => $self->banner_id,
            REF => '{TREF}',
            DRND => '{DRND}',
            DTYPE => '{DEVICE_TYPE}',
            SESSION => '{Kad-session-id}',
            hitlogid => '{HLID}',
            CID => $self->campaign_id,
            BTYPE => exists $VERIFY_ADGROUP_TYPE{$self->adgroup_type} ? $VERIFY_ADGROUP_TYPE{$self->adgroup_type} : $VERIFY_ADGROUP_TYPE{default}
        };
    } elsif ($self->provider eq 'adfox') {
        hash_merge \%additional_params, {
            c_id => '{HLID}',
            b_id => '{BID}',
            o_id => '{OID}',
        };
    }
    if (%additional_params) {
        $url .= '&' . join '&', map { sprintf '%s=%s', $_, $additional_params{$_} } sort keys %additional_params;
    }

    return $url;
}


1;
