package Direct::Model::Retargeting::Manager;

use Direct::Modern;
use Mouse;

extends 'Direct::Model::ShowCondition::Manager';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;

use List::MoreUtils qw/uniq/;

use Direct::Model::Retargeting;

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::Retargeting]',
);

our %ALLOWED_RETARGETING_TO_ADGROUP_TYPES = (
    base            => 1,
    mobile_content  => 1,
    cpm_banner      => 1,
    cpm_video       => 1,
);

=head2 create

Создание в БД записей для соответствующих объектов (ретаргетинг/ставки).

=cut

sub create {
    my ($self) = @_;

    {;
        my @adgroup_ids = uniq(map { $_->adgroup_id } @{$self->items});
        my $adgroup_id2type = get_hash_sql(PPC(pid => \@adgroup_ids), ["SELECT pid, adgroup_type FROM phrases", where => {pid => SHARD_IDS}]);
        for my $retargeting (@{$self->items}) {
            my $adgroup_type = $adgroup_id2type->{$retargeting->adgroup_id};
            next if defined $adgroup_type &&
                $ALLOWED_RETARGETING_TO_ADGROUP_TYPES{$adgroup_type};
            croak sprintf(
                "retargetings can be added only to adgroup with type: %s",
                join(', ', keys %ALLOWED_RETARGETING_TO_ADGROUP_TYPES),
            );
        }
    }

    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});

        do_in_transaction {
   
            # Создадим записи в таблице `bids_retargeting`
            $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'bids_retargeting', undef, $shard_items);

            # Обработаем флаги
            $self->_do_update_adgroups($shard, $shard_items);
        };
    }

    $_->reset_state() for @{$self->items};

    return;
}

=head2 update

Обновление в БД записей для соответствующих объектов (ретаргетинг/ставки).

=cut

sub update {
    my ($self) = @_;

    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});

        do_in_transaction {
            $self->_update_one_table_in_db(PPC(shard => $shard), bids_retargeting => 'ret_id', $shard_items);

            # Обработаем флаги
            $self->_do_update_adgroups($shard, $shard_items);
            $self->_proceed_empty_adgroups_and_banners($shard, $shard_items);
        };
    }

    $_->reset_state() for @{$self->items};

    return;
}

=head2 delete

Удаление из БД записей для соответствующих объектов (ретаргетинг/ставки).

=cut

sub delete {
    my ($self) = @_;

    for my $chunk (sharded_chunks(pid => $self->items, by => sub { $_->adgroup_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});

        do_in_transaction {

            do_delete_from_table(PPC(shard => $shard),
                'bids_retargeting',
                where => { ret_id => [map { $_->id } @$shard_items] },
            );

            # Обработаем флаги
            $self->_do_update_adgroups($shard, $shard_items);
            $self->_proceed_empty_adgroups_and_banners($shard, $shard_items);
        };
    }

    $_->reset_state() for @{$self->items};

    return;
}

sub _do_update_adgroups {
    my ($self, $shard, $retargetings) = @_;

    my %update_adgroups;
    for my $retargeting (@$retargetings) {
        # Отправка группы в БК
        if ($retargeting->do_bs_sync_adgroup) {
            $update_adgroups{$retargeting->adgroup_id}->{statusBsSynced} = sql_quote('No');
            $update_adgroups{$retargeting->adgroup_id}->{LastChange} = 'LastChange';
        }

        # Изменение времени модификации группы (важно, чтобы условие было в конце)
        if ($retargeting->do_update_adgroup_last_change) {
            $update_adgroups{$retargeting->adgroup_id}->{LastChange} = 'NOW()';
        }
    }

    return if !%update_adgroups;

    my %byfield_options;
    $byfield_options{$_} = {default__dont_quote => $_, dont_quote_value => 1} for map { keys %$_ } values %update_adgroups;

    do_mass_update_sql(PPC(shard => $shard), 'phrases', 'pid', \%update_adgroups, byfield_options => \%byfield_options);

    return;
}

sub _proceed_empty_adgroups_and_banners {

    my ($self, $shard, $keywords) = @_;
    # Проверяем не стали ли группы пустыми.
    my @uniq_adgroup_ids = uniq map { $_->adgroup_id } @$keywords;
    my @empty_adgroup_ids = uniq @{get_one_column_sql(PPC(shard => $shard), ["select pid from phrases p", where => {'p.pid' => \@uniq_adgroup_ids},
                                                                    "and !(exists(select 1 from bids bi where bi.pid=p.pid and bi.is_suspended=0) ".
                                                                    "     OR exists(select 1 from bids_retargeting bir where bir.pid=p.pid and bir.is_suspended=0)) FOR UPDATE"])};

    # Пустым группам и баннерам сбрасываем bs_synced.
    $self->clear_bs_synced_for_adgroups(\@empty_adgroup_ids);

    return;
}

__PACKAGE__->meta->make_immutable;

1;
