package Direct::Model::RetargetingCondition;

use Direct::Modern;
use Mouse;
use Mouse::Util::TypeConstraints;

use Yandex::ORM::Helpers qw/mysql_timestamp_around/;

use JSON;
use List::MoreUtils qw/all any/;

use Direct::Model::RetargetingCondition::Rule;

extends 'Yandex::ORM::Model::Base';
with 'Direct::Model::Role::Update';

subtype 'RetargetingRulesArrayRef'
    => as 'ArrayRef[Direct::Model::RetargetingCondition::Rule]'; # => where { scalar(@$_) > 0 };

coerce 'RetargetingRulesArrayRef'
    => from 'ArrayRef[HashRef]'
    => via { [map { Direct::Model::RetargetingCondition::Rule->new($_) } @$_] };

__PACKAGE__->_setup(
    default_table  => 'retargeting_conditions',

    fields => [
        id                                  => { type => 'Id', column => 'ret_cond_id', primary_key => 1 },
        client_id                           => { type => 'Id', column => 'ClientID' },
        condition_name                      => { type => 'Maybe[Str]', length => 4096, default => '' },
        condition_desc                      => { type => 'Maybe[Str]', length => 4096, default => '' },
        retargeting_conditions_type         => { type => 'Maybe[Enum]', values => [qw/interests metrika_goals dmp ab_segments brandsafety shortcuts/], default => 'metrika_goals' },
        is_deleted                          => { type => 'Bool', default => 0 },
        last_change                         => { type => 'Timestamp', column => 'modtime' },
        properties                          => { type => 'Set', values => ['negative', 'interest', 'autoretargeting'] },

        # Hidden fields
        _condition_json                     => { type => 'Str', column => 'condition_json' },
    ],

    additional => [
        is_accessible                       => { type => 'Bool' },
        is_used                             => { type => 'Bool' },
    ],

    relations => [
        condition                           => { type => 'RetargetingRulesArrayRef', trigger => \&_on_condition_changed, coerce => 1 },
    ],

    state_flags => [qw/
        bs_sync_adgroups
        bs_sync_multipliers
        update_adgroups_last_change
        clear_goals
    /],
);

around BUILDARGS => sub {
    my ($orig, $class) = (shift, shift);
    my %args = @_ == 1 && ref($_[0]) eq 'HASH' ? %{$_[0]} : @_;

    if (defined $args{_condition_json}) {
        eval { $args{condition} = JSON->new->utf8(0)->decode($args{_condition_json}); 1; } or do { croak "Cannot apply `condition_json`: $@"; };
    }

    $class->$orig(%args);
};

# Allow set to `now`
around last_change => mysql_timestamp_around('last_change');

sub _on_condition_changed {
    my ($self, $new) = @_;

    if ($self->_constructed || !$self->_has_condition_json) {
        my @condition = map { $_->to_hash(json => 1) } @$new;
        $self->_condition_json(JSON->new->utf8(0)->canonical->encode(\@condition));
    }
    if ($self->_constructed || !$self->has_properties) {
        my %properties_map = (map { $_ => 1 } ($self->has_properties ? @{$self->properties} : ()));
        $properties_map{negative} = all { $_->type eq 'not' } @$new;
        $self->properties([sort grep { $properties_map{$_} } keys %properties_map]);
    }

    return;
}

sub is_condition_changed { shift->_is_condition_json_changed(@_) }

sub is_negative { (any { $_ eq 'negative' } @{shift->properties}) ? 1 : 0 }

sub get_using_goal_ids {
    my $self = shift;
    my @ids;
    for my $rule (@{$self->condition}) {
        push @ids, map { $_->goal_id } @{$rule->goals};
    }
    return \@ids;
}

override to_hash => sub {
    my ($self) = @_;
    my $hash = super();
    delete @{$hash}{qw/_condition_json/};
    return $hash;
};

override to_db_hash => sub {
    my ($self) = @_;
    my $hash = super();
    $hash->{is_accessible} = 0 + $self->is_accessible if $self->has_is_accessible; # force to int
    return $hash;
};

sub to_template_hash {
    my $hash = shift->to_hash;
    $hash->{ret_cond_id} = delete $hash->{id} if exists $hash->{id};
    return $hash;
}

1;
