package Direct::Model::RetargetingCondition::Manager;

use Direct::Modern;
use Mouse;

extends 'Yandex::ORM::Model::Manager::Base';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils qw/hash_merge/;
use List::MoreUtils qw/any/;

use BS::ResyncQueue qw//;

use Direct::Model::RetargetingCondition;

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::RetargetingCondition]',
);

has 'skip_goals_update' => (is => 'ro', isa => 'Bool', default => 0);

=head2 create

Создание в БД записей для соответствующих объектов (условий ретаргетинга).

=cut

sub create {
    my ($self) = @_;

    for my $chunk (sharded_chunks(ClientID => $self->items, by => sub { $_->client_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{ClientID});

        do_in_transaction {
            _lowlevel_check_integrity_inside_transaction($shard, $shard_items, 'create');

            # Создадим записи в таблице `retargeting_conditions`
            $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'retargeting_conditions', undef, $shard_items);
        };

        # Обновим цели
        $self->_update_goals($shard, $shard_items, 'create');
    }

    $_->reset_state() for @{$self->items};

    return;
}

=head2 update

Обновление в БД записей для соответствующих объектов (условий ретаргетинга).

=cut

sub update {
    my ($self) = @_;

    for my $chunk (sharded_chunks(ClientID => $self->items, by => sub { $_->client_id })) {
        my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{ClientID});

        # Защита от изменения `negative` свойства
        if (any { $_->is_properties_changed } @$shard_items) {
            my $ex_props_by = get_hash_sql(PPC(shard => $shard), [
                "SELECT ret_cond_id, properties FROM retargeting_conditions",
                WHERE => {ret_cond_id => [map { $_->id } @$shard_items]},
            ]);
            if (any { $_->is_negative ^ ($ex_props_by->{$_->id} =~ /\bnegative\b/ ? 1 : 0) } @$shard_items) {
                croak "Cannot change `negative` property";
            }
        }

        do_in_transaction {
            _lowlevel_check_integrity_inside_transaction($shard, $shard_items, 'update');

            $self->_update_one_table_in_db(PPC(shard => $shard), retargeting_conditions => 'ret_cond_id', $shard_items);
        };

        # Обновим цели
        $self->_update_goals($shard, $shard_items, 'update');

        # Обработаем флаги
        $self->_do_update_adgroups($shard, $shard_items);
        $self->_do_bs_sync_multipliers($shard, $shard_items);
    };

    $_->reset_state() for @{$self->items};

    return;
}

sub _lowlevel_check_integrity_inside_transaction {
    my ($shard, $ret_conds, $action) = @_;

    # условия ретаргетинга с целями таргетирования по интересам не имеют названий (null)
    $ret_conds = [grep {!$_->has_properties || !any {$_ eq 'interest'} @{ $_->properties }} @$ret_conds];
    # ограничения по названию не действуют на условия для кампаний cpm_banner типа соцдем
    $ret_conds = [grep {$_->has_retargeting_conditions_type && !$_->retargeting_conditions_type eq 'interests'} @$ret_conds];

    $ret_conds = [grep { $_->is_condition_name_changed } @$ret_conds] if $action eq 'update';
    return if !@$ret_conds;

    my $ret_conds_by_client_id;
    push @{$ret_conds_by_client_id->{ $_->client_id }}, $_ for @$ret_conds;

    my $should_fail = get_one_field_sql(PPC(shard => $shard), [
        "SELECT 1 FROM retargeting_conditions",
        WHERE => {
            _OR => [
                map { (
                    _AND => {ClientID => $_, condition_name => [map { $_->condition_name } @{$ret_conds_by_client_id->{$_}}]}
                ) } keys %$ret_conds_by_client_id
            ],
            is_deleted => 0,
        },
        "LIMIT 1",
    ]);

    croak "Found retargeting condition(s) with the same name" if $should_fail;

    return;
}

sub _update_goals {
    my ($self, $shard, $ret_conds, $action) = @_;

    return if $self->skip_goals_update;

    if (my @ret_conds_to_clear_goals = grep { $_->do_clear_goals } @$ret_conds) {
        do_delete_from_table(PPC(shard => $shard), 'retargeting_goals', where => {
            ret_cond_id => [map { $_->id } @ret_conds_to_clear_goals],
        });
    }

    my @ret_cond_ids = map { $_->id } grep { $action eq 'create' ? 1 : $_->is_condition_changed } grep { !$_->do_clear_goals } @$ret_conds;
    return if !@ret_cond_ids;

    my $ex_ret_goals = get_all_sql(PPC(shard => $shard), [
        "SELECT ret_cond_id, goal_id, goal_type, is_accessible FROM retargeting_goals", WHERE => {ret_cond_id => \@ret_cond_ids},
    ]);
    my $ex_ret_goals_by = {};
    $ex_ret_goals_by->{ $_->{ret_cond_id} }{ $_->{goal_id} } = $_ for @$ex_ret_goals;

    my @ret_goals_to_save;
    for my $ret_cond (@$ret_conds) {
        for my $goal (map { $_->to_hash } map { @{$_->goals} } @{$ret_cond->condition}) {
            my $ex_goal = ($ex_ret_goals_by->{ $ret_cond->id }{ $goal->{goal_id} } //= {});

            if (!$ex_goal->{goal_id} || !$ex_goal->{is_accessible} || $ex_goal->{goal_type} ne $goal->{goal_type}) {
                # is_accessible выставляем в 1
                push @ret_goals_to_save, [$ret_cond->id, $goal->{goal_id}, $goal->{goal_type}, 1];
            }

            hash_merge $ex_goal, {
                ret_cond_id => $ret_cond->id, goal_id => $goal->{goal_id}, goal_type => $goal->{goal_type}, is_accessible => 1, _keep => 1,
            };
        }
    }

    # Встановим новые / обновим старые цели
    do_mass_insert_sql(PPC(shard => $shard), "
        INSERT INTO retargeting_goals (ret_cond_id, goal_id, goal_type, is_accessible) VALUES %s
        ON DUPLICATE KEY UPDATE
            is_accessible = VALUES(is_accessible),
            goal_type = VALUES(goal_type)
    ", \@ret_goals_to_save);

    # Удалим старые цели
    for my $ret_cond_id (keys %$ex_ret_goals_by) {
        my $ret_cond_goals = $ex_ret_goals_by->{$ret_cond_id};
        delete $ret_cond_goals->{$_} for grep { $ret_cond_goals->{$_}->{_keep} } keys %$ret_cond_goals;
        delete $ex_ret_goals_by->{$ret_cond_id} if !%$ret_cond_goals;
    }
    do_delete_from_table(PPC(shard => $shard), 'retargeting_goals', where => {
        _OR => [
            map { (_AND => {ret_cond_id => $_, goal_id => [keys %{$ex_ret_goals_by->{$_}}]}) } keys %$ex_ret_goals_by
        ],
    }) if %$ex_ret_goals_by;

    return;
}

sub _do_update_adgroups {
    my ($self, $shard, $ret_conds) = @_;

    $ret_conds = [grep { $_->do_bs_sync_adgroups || $_->do_update_adgroups_last_change } @$ret_conds];
    return if !@$ret_conds;

    # Получим идентификаторы групп, к которым привязаны условия ретаргетинга
    my @ret_cond_ids = map { $_->id } @$ret_conds;
    my $used_adgroups = get_all_sql(PPC(shard => $shard), [
        "(SELECT bret.pid, bret.ret_cond_id, 'default' AS type FROM bids_retargeting bret", WHERE => {'bret.ret_cond_id' => \@ret_cond_ids}, ")",
        "UNION ALL",
        "(SELECT bperf.pid, bperf.ret_cond_id, 'performance' AS type FROM bids_performance bperf", WHERE => {'bperf.ret_cond_id' => \@ret_cond_ids}, ")",
    ]);

    return if !@$used_adgroups;

    my $adgroups_by_ret_cond_id = {};
    push @{$adgroups_by_ret_cond_id->{$_->{ret_cond_id}}}, $_ for @$used_adgroups;

    my (%update_adgroups, @perf_gids_to_bs_sync_banners);

    for my $ret_cond (@$ret_conds) {
        for my $adgroup (@{$adgroups_by_ret_cond_id->{ $ret_cond->id } // []}) {
            # Отправка группы в БК
            if ($ret_cond->do_bs_sync_adgroups) {
                $update_adgroups{ $adgroup->{pid} }->{statusBsSynced} = sql_quote('No');
                $update_adgroups{ $adgroup->{pid} }->{LastChange} //= 'LastChange';

                # Для перфоманс групп нужно еще сбросить statusBsSynced на всех баннерах
                if ($adgroup->{type} eq 'performance') {
                    push @perf_gids_to_bs_sync_banners, $adgroup->{pid};
                }
            }

            # Изменение времени модификации группы
            if ($ret_cond->do_update_adgroups_last_change) {
                $update_adgroups{ $adgroup->{pid} }->{LastChange} = 'NOW()';
            }
        }
    }

    my %byfield_options;
    $byfield_options{$_} = {default__dont_quote => $_, dont_quote_value => 1} for map { keys %$_ } values %update_adgroups;

    do_mass_update_sql(PPC(shard => $shard), 'phrases', 'pid', \%update_adgroups, byfield_options => \%byfield_options);

    # Для перфоманс групп нужно еще сбросить statusBsSynced на всех баннерах
    if (@perf_gids_to_bs_sync_banners) {
        do_sql(PPC(shard => $shard), [
            "UPDATE banners SET statusBsSynced = 'No', LastChange = LastChange", WHERE => {pid => \@perf_gids_to_bs_sync_banners},
        ]);
    }

    return;
}

sub _do_bs_sync_multipliers {
    my ($self, $shard, $ret_conds) = @_;

    $ret_conds = [grep { $_->do_bs_sync_multipliers } @$ret_conds];
    return if !@$ret_conds;

    my $data = get_all_sql(PPC(shard => $shard), ["
        SELECT DISTINCT hm.cid, hm.pid
        FROM retargeting_multiplier_values rmv
        JOIN hierarchical_multipliers hm ON (hm.hierarchical_multiplier_id = rmv.hierarchical_multiplier_id)
    ", WHERE => {
        'rmv.ret_cond_id' => [map { $_->id } @$ret_conds],
    }]);

    BS::ResyncQueue::bs_resync($data) if @$data;

    return;
}

__PACKAGE__->meta->make_immutable;

1;
