package Direct::Model::SitelinksSet::Manager;

use Direct::Modern;
use Yandex::ORM::Types;
use Mouse;

extends 'Yandex::ORM::Model::Manager::Base';

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use List::UtilsBy qw/uniq_by partition_by/;
use JavaIntapi::GenerateObjectIds;

use Direct::Model::Sitelink;
use Direct::Model::SitelinksSet;

has 'items' => (is  => 'ro', isa => 'ArrayRef[Direct::Model::SitelinksSet]');

=head2 save

Сохранение в БД записей для соответствующих объектов (наборы сайтлинков).
Т.к. сайтлинки иммутабельны, работа с ними в БД представлена только одним методом.

=cut

sub save {
    my ($self) = @_;

    my @columns = Direct::Model::SitelinksSet->get_db_columns_list('sitelinks_sets');

    foreach_shard ClientID => $self->items, by => sub { $_->client_id }, sub {
        my ($shard, $sitelinks_sets) = @_;

        $self->_save_sitelinks($shard, [map { @{$_->links} } @$sitelinks_sets]);

        # Сохраняем только новые/изменившееся наборы сайтлинков
        $sitelinks_sets = [grep { $_->calculate_hash; !$_->has_id || !$_->id || $_->is_changed } @$sitelinks_sets];
        return if !@$sitelinks_sets;

        my %sitelinks_sets_by_client_id = partition_by { $_->client_id } @$sitelinks_sets;
        my $uhash2id = get_hash_sql(PPC(shard => $shard), [
            q{SELECT CONCAT_WS(':', ClientID, links_hash), sitelinks_set_id FROM sitelinks_sets},
            WHERE => {
                _OR => [map { (
                    _AND => {ClientID => $_, links_hash => [map { $_->hash } @{$sitelinks_sets_by_client_id{$_}}]}
                ) } keys %sitelinks_sets_by_client_id],
            },
        ]);

        # Сохраним новые уникальные объекты
        if (my @sitelinks_sets_to_save = uniq_by { $_->get_uhash } grep { !$uhash2id->{$_->get_uhash} } @$sitelinks_sets) {
            my %new_sets_by_client_id = partition_by { $_->client_id } @sitelinks_sets_to_save;
            while (my ($client_id, $new_sets) = each %new_sets_by_client_id) {
                my $new_ids = JavaIntapi::GenerateObjectIds->new(object_type => 'sitelinks_set',
                        count => scalar @$new_sets, client_id => $client_id)->call();
                $_->id((0 + @$new_ids) ? shift @$new_ids : die "Couldn't get new sitelinks_set_id") for @$new_sets;
            }

            $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'sitelinks_sets', \@columns, \@sitelinks_sets_to_save);

            my @rows_to_insert;
            for my $sitelinks_set (@sitelinks_sets_to_save) {
                for (my $idx = 0; $idx < scalar(@{$sitelinks_set->links}); $idx++) {
                    push @rows_to_insert, [$sitelinks_set->id, $sitelinks_set->links->[$idx]->id, $idx];
                }
            }

            do_mass_insert_sql(PPC(shard => $shard),
                'INSERT IGNORE INTO sitelinks_set_to_link (sitelinks_set_id, sl_id, order_num) VALUES %s',
                \@rows_to_insert
            );

            # Дополним наш uhash2id новыми айдишниками
            $uhash2id->{$_->get_uhash} = $_->id for @sitelinks_sets_to_save;
        }

        $_->id($uhash2id->{$_->get_uhash}) for @$sitelinks_sets;
    };

    $_->reset_state() for @{$self->items};

    return;
}

sub _save_sitelinks {
    my ($self, $shard, $sitelinks) = @_;

    # Выберем только новые/изменившееся сайтлинки
    $sitelinks = [grep { !$_->has_id || !$_->id || $_->is_changed } @$sitelinks];

    return if !@$sitelinks;

    my $uhash2id = get_hash_sql(PPC(shard => $shard), [
        q{SELECT CONCAT_WS(':', hash, title, href, tl_id) uhash, sl_id FROM sitelinks_links FORCE INDEX(hash)},
        WHERE => {
            _OR => [map { (
                _AND => {hash => $_->hash, title => $_->title, href => $_->href, tl_id => $_->tl_id},
            ) } @$sitelinks],
        },
    ]);

    # Сохраним новые уникальные объекты
    if (my @sitelinks_to_save = uniq_by { $_->get_uhash } grep { !$uhash2id->{$_->get_uhash} } @$sitelinks) {
        my $new_ids = JavaIntapi::GenerateObjectIds->new(object_type => 'sitelink',
                count => scalar(@sitelinks_to_save))->call();
        $_->id(shift @$new_ids) for @sitelinks_to_save;

        my @columns = Direct::Model::Sitelink->get_db_columns_list('sitelinks_links');
        $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'sitelinks_links', \@columns, \@sitelinks_to_save);

        # Дополним наш uhash2id новыми айдишниками
        $uhash2id->{$_->get_uhash} = $_->id for @sitelinks_to_save;
    }

    $_->id($uhash2id->{$_->get_uhash}) for @$sitelinks;

    $_->reset_state() for @$sitelinks;

    return;
}

1;
