package Direct::Model::VCard::Manager;

use Direct::Modern;
use Mouse;

extends 'Yandex::ORM::Model::Manager::Base';

use Settings;

use List::MoreUtils qw/uniq/;

use CommonMaps qw//;
use VCards qw//;
use OrgDetails qw//;
use PrimitivesIds qw//;
use Property;
use GeoTools qw//;
use JavaIntapi::CreateVcards qw//;
use Yandex::DBTools qw/get_hashes_hash_sql/;

use Direct::Model::VCard;

has 'items' => (
    is  => 'ro',
    isa => 'ArrayRef[Direct::Model::VCard]',
);

=head2 save

Сохранение в БД записей для соответствующих объектов (визитки).
Т.к. визитки иммутабельны, работа с ними в БД представлена только одним методом.

=cut

sub save {
    my ($self, $operator_uid) = @_;

    if (Property->new($VCards::USE_JAVA_VCARDS_IN_SMART_PROPERTY_NAME)->get() 
            && $operator_uid) {
        return $self->_java_save($operator_uid);
    }

    my $uid2client_id = PrimitivesIds::get_key2clientid(uid => [uniq(map { $_->user_id } @{$self->items})]);
    my %vcards_by_uid;

    for my $vcard (@{$self->items}) {
        my $is_new_vcard = !$vcard->has_id || !$vcard->id;

        next unless $is_new_vcard || $vcard->is_changed;

        if (
            $is_new_vcard ||
            $vcard->is_country_changed || $vcard->is_city_changed || $vcard->is_street_changed ||
            $vcard->is_house_changed || $vcard->is_building_changed
        ) {
            my $map = CommonMaps::check_address_map($vcard->to_db_hash, {ClientID => $uid2client_id->{$vcard->user_id}});
            $vcard->address_id($map->{aid});
            $vcard->geo_id(GeoTools::get_geoid_by_cityname($vcard->city));
        }

        if ($is_new_vcard || $vcard->is_ogrn_changed) {
            my $org_details_id = OrgDetails::add_org_details({uid => $vcard->user_id, ogrn => $vcard->ogrn});
            $vcard->_org_details_id($org_details_id);
        }

        push @{$vcards_by_uid{ $vcard->user_id }}, $vcard;
    }

    while (my ($uid, $vcards) = each %vcards_by_uid) {
        my $vcard_hashes = VCards::create_vcards($uid, [map { $_->to_db_hash } @$vcards]);
        for (my $i = 0; $i <= $#$vcards; $i++) {
            $vcards->[$i]->id($vcard_hashes->[$i]->{vcard_id});
        }
    }

    $_->reset_state() for @{$self->items};

    return;
}


sub _java_save {
    my ($self, $operator_uid) = @_;

    my $uid2client_id = PrimitivesIds::get_key2clientid(uid => [uniq(map { $_->user_id } @{$self->items})]);
    my %vcards_by_client_id;

    for my $vcard (@{$self->items}) {
        my $is_new_vcard = !$vcard->has_id || !$vcard->id;
        next unless $is_new_vcard || $vcard->is_changed;
        push @{$vcards_by_client_id{ $uid2client_id->{$vcard->user_id} }}, $vcard;
    }

    while (my ($client_id, $vcards) = each %vcards_by_client_id) {
        my $vcard_ids = JavaIntapi::CreateVcards->new(items => [ map { $_->to_hash } @$vcards ], 
                operator_uid => $operator_uid, client_id => $client_id)->call();
        my $vcards_by_id = get_hashes_hash_sql(PPC(ClientID => $client_id), [
            "select vcard_id, address_id, geo_id, org_details_id from vcards",
             where => { vcard_id => $vcard_ids },]);

        for (my $i = 0; $i <= $#$vcards; $i++) {
            my ($vcard, $vcard_id) = ($vcards->[$i], $vcard_ids->[$i]);
            $vcard->id($vcard_id);

            next unless $vcards_by_id->{ $vcard_id };
            $vcard->address_id($vcards_by_id->{ $vcard_id }->{address_id});
            $vcard->geo_id($vcards_by_id->{ $vcard_id }->{geo_id});
            $vcard->_org_details_id($vcards_by_id->{ $vcard_id }->{org_details_id});
        }
    }
    $_->reset_state() for @{$self->items};
    return;
}

__PACKAGE__->meta->make_immutable;

1;
