package Direct::Model::VideoAddition;

use Direct::Modern;
use Mouse;
use Mouse::Util::TypeConstraints;
use Carp;
use JSON;
use Lang::Guess qw/analyze_text_lang/;
use List::MoreUtils qw/any/;
# use Direct::Model::CanvasCreativeScales ();

extends 'Direct::Model::Creative';


our $FAKE_VIDEO_WIDTH = 640;
our $FAKE_VIDEO_HEIGHT = 360;
our $PREVIEW_WIDTH = 300;

our %COMPATIBLE_LAYOUT_IDS = (
    text => [1 .. 5],
    mobile_content => [7, 11 .. 17],
    cpm_video => [6],
    cpc_video => [51 .. 55, 61 .. 67],
    cpm_outdoor => [101 .. 150],
    cpm_indoor => [201..250],
    cpm_audio => [301..350]
);

our %CANVAS_TAGS = (
    text => 'text',
    mobile_content => 'mobile_content',
    cpm_video => 'cpm',
    cpc_video => 'cpc',
    cpm_outdoor => 'cpm_outdoor',
    cpm_indoor => 'cpm_indoor',
    cpm_audio => 'cpm_audio'
);


has '+creative_type'       => (isa => enum(['video_addition', 'bannerstorage']), lazy => 0, builder => sub { 'video_addition' });
has '+template_id'         => (isa => 'Maybe[Int]');
has '+template_name'       => (isa => 'Undef');
has '+layout'              => (isa => 'Undef', lazy => 0, builder => sub { undef });
has '+width'               => (isa => 'Maybe[Int]');
has '+height'              => (isa => 'Maybe[Int]');
has '+preview_scale'       => (lazy => 0, builder => sub { $PREVIEW_WIDTH / $FAKE_VIDEO_WIDTH });

has resource_type     => ( is => 'rw', isa => enum(['creative', 'media']), lazy => 0, builder => sub {'creative'} );
has _used_resources   => ( is => 'rw', isa => 'Maybe[Str]' );

has yacontext_categories => ( is => 'rw', isa => 'Maybe[ArrayRef[Int]]' );

sub _build_preview_scale { }

sub _build_template_name { }

sub get_scale_size {
    my $self = shift;
    return $PREVIEW_WIDTH / $FAKE_VIDEO_WIDTH;
    #return Direct::Model::CanvasCreativeScales::size2scale($FAKE_VIDEO_WIDTH . 'x' . $FAKE_VIDEO_HEIGHT);
}

sub to_template_hash {
    my $self = shift;

    my $hash = $self->to_hash;
    $hash->{has_packshot} = int($hash->{has_packshot}) if defined $hash->{has_packshot};
    $hash->{resource_url} = $hash->{preview_url};
    delete $hash->{status_moderate}; # это статус из perf_creatives, он всегда New, настоящий статус в banners_performance
    # дублируем, потому что фронт кое-где ожидает id
    $hash->{creative_id} = $hash->{id};
    $hash->{width} = $self->get_width();
    $hash->{height} = $self->get_height();
    $hash->{scale} = $self->get_scale_size();
    $hash->{ration} = "16/9";
    $hash->{canvas_tags} = $self->get_canvas_tags();

    $hash->{additional_data} = $self->additional_data_to_template();
    $hash->{duration} = $self->get_duration();
    delete $hash->{_additional_data};

    return $hash;
}

sub TO_JSON
{
    return shift->to_template_hash();
}

sub detect_lang {
    my $self = shift;
    my $info = $self->moderate_info;
    return analyze_text_lang(map { $_->{text} } @{$info->{texts} || []}) || 'ru';
}

sub moderate_info {
    my $self = shift;
    if (@_) {
        my $moderate_info = shift;
        $self->_moderate_info(defined $moderate_info ? JSON->new->utf8(0)->encode($moderate_info) : undef);
        return $moderate_info;
    }

    return {} unless $self->_moderate_info;
    return JSON->new->utf8(0)->decode($self->_moderate_info);
}

#В существующем поле duration хранится округленное значение длительности.
#Для outdoor пытаемся взять реальное значение из additional_data (для outdoor 3:1 - это 7,5 сек). Если не находим, берем округленное
sub get_duration {
    my $self = shift;
    return $self->is_outdoor_creative ? $self->additional_data->{duration} : $self->duration;
}

sub get_height {
    my $self = shift;
    my $audio_range = $COMPATIBLE_LAYOUT_IDS{cpm_audio};
    if (any {$self->layout_id && $self->layout_id == $_} @$audio_range) {
        return $FAKE_VIDEO_HEIGHT;# квадратная картинка
    }
    if (defined $self->{_moderate_info} && $self->moderate_info && $self->moderate_info->{aspects} &&
            $self->moderate_info->{aspects}[0]->{height} > 0) {
        if ($self->moderate_info->{aspects}[0]->{height} == $self->moderate_info->{aspects}[0]->{width}) {
            return $FAKE_VIDEO_HEIGHT;
        }
        if ($self->moderate_info->{aspects}[0]->{height} > $self->moderate_info->{aspects}[0]->{width}) {
            return 600;
        }
        return 640 * $self->moderate_info->{aspects}[0]->{height} / $self->moderate_info->{aspects}[0]->{width};
    }
    return $FAKE_VIDEO_HEIGHT;
}
sub get_width {
    my $self = shift;
    my $audio_range = $COMPATIBLE_LAYOUT_IDS{cpm_audio};
    if (any {$self->layout_id && $self->layout_id == $_} @$audio_range) {
        return $FAKE_VIDEO_HEIGHT;# квадратная картинка
    }
    if (defined $self->{_moderate_info} && $self->moderate_info && $self->moderate_info->{aspects} &&
            $self->moderate_info->{aspects}[0]->{height} > 0) {
        if ($self->moderate_info->{aspects}[0]->{height} == $self->moderate_info->{aspects}[0]->{width}) {
            return $FAKE_VIDEO_HEIGHT;
        }
        if ($self->moderate_info->{aspects}[0]->{height} > $self->moderate_info->{aspects}[0]->{width}) {
            return 600 * $self->moderate_info->{aspects}[0]->{width} / $self->moderate_info->{aspects}[0]->{height};
        }
        return 640;
    }
    return $FAKE_VIDEO_WIDTH;
}
#использовать для отдачи в интерфейс
sub additional_data_to_template {
    my $self = shift;

    my $result = {};
    return $result if !$self->is_outdoor_creative;

    my $additional_data = $self->additional_data;
    if ($additional_data->{formats}) {
        $result->{formats} = [grep {defined $_->{width} && defined $_->{height}} @{$additional_data->{formats}}];
    }
    return $result;
}

sub is_outdoor_creative {
    my $self = shift;
    my $oudoor_range = $COMPATIBLE_LAYOUT_IDS{cpm_outdoor};
    return any {$self->layout_id && $self->layout_id == $_} @$oudoor_range
}

sub additional_data {
    my $self = shift;
    if (@_) {
        my $additional_data = shift;
        $self->_additional_data(defined $additional_data ? JSON->new->utf8(0)->encode($additional_data) : undef);
        return $additional_data;
    }

    return {} unless $self->_additional_data;
    return JSON->new->utf8(1)->decode($self->_additional_data);
}

sub get_canvas_tags {
    my $self = shift;
    my @result;
    foreach my $key (keys %COMPATIBLE_LAYOUT_IDS) {
        if (any {$self->layout_id && $_ eq $self->layout_id} @{$COMPATIBLE_LAYOUT_IDS{$key}}) {
            push @result, $CANVAS_TAGS{$key};
        }
    }
    return \@result;
}

1;
