package Direct::Monitor::Daily;

=head1 NAME
    
    Direct::Monitor::Daily - сборище мониторингов, которые должны запускаться раз в сутки

=cut

use strict;
use warnings;

use base qw/Direct::Monitor::Base/;

use Settings;
use Yandex::DBTools;
use Yandex::Overshard;
use ShardingTools;
use Yandex::HashUtils;
use List::Util qw/sum/;
use Currencies;
use Currency::Format;
use Currency::Rate;
use TimeTarget;

use utf8;

sub calc_vcards_values :Monitor
{
    # sql'и для подсчета интересных значений
    my %target = (
        # количество медиаплановых объявлений с визитками
        'direct.vcards.mediaplan_banners_with_vcards_total_count' => {
            desc => 'общее число медиаплановых блоков с визитками',
            sql => "SELECT COUNT(*) FROM mediaplan_banners WHERE vcard_id IS NOT NULL",
        },

        # ... и в медиаплановых 
        'direct.vcards.broken_links_count_mediaplan' => {
            desc => 'количество битых ссылок на визитки из медиапланов',
            sql => 
            "SELECT count(*) 
               FROM mediaplan_banners mb 
                    LEFT JOIN vcards vc ON vc.vcard_id = mb.vcard_id AND vc.cid = mb.cid
              WHERE mb.vcard_id IS NOT NULL AND vc.vcard_id IS NULL",
        },

        # Количество визиток, у которых geo_id = 0, а должно быть не нулевым (указан город, для которого мы можем определить регион)
        'direct.vcards.with_geo_id_0_count' => {
            desc => 'количество визиток с ошибочным значением geo_id = 0',
            sql => 
            "SELECT COUNT(*)
               FROM vcards v
                    JOIN geo_regions g ON g.name = v.city
              WHERE geo_id = 0",
        },
    );
    
    # считаем все, что считается простыми запросами...
    my $res;
    for my $id (sort keys %target){
        $res->{$id}->{value} = overshard_sum(PPC(shard => 'all'), $target{$id}->{sql});
        hash_copy $res->{$id}, $target{$id}, qw/desc old_name units/;
    }
    
    return $res;
}


#...................................................................................................
# https://jira.yandex-team.ru/browse/DIRECT-4944
sub fair_auction_values :Monitor
{
    my %target_desc =(
        fair_auction_agencies_count => 'агентств с честными торгами',
        fair_auction_agency_camps_count => 'агентских кампаний с честными торгами',
        fair_auction_agency_clients_count => 'субклиентов с честными торгами',
        fair_auction_all_camps_count => 'всего кампаний с честными торгами',
        fair_auction_all_clients_count => 'всего клиентов с честными торгами',
        fair_auction_managers_count => 'менеджеров с честными торгами', 
        fair_auction_manager_camps_count => 'обслуживаемых менеджером кампаний с честными торгами',
        fair_auction_manager_clients_count => 'обслуживаемых менеджером клиентов с честными торгами',
        fair_auction_self_camps_count => 'самоходных кампаний с честными торгами', 
        fair_auction_self_clients_count => 'самоходных клиентов с честными торгами', 
    );
    my $sql_fair_auction = {
        overshard => {
            group => '',
            sum => [ qw/  fair_auction_all_camps_count
                fair_auction_agency_camps_count   fair_auction_manager_camps_count
                fair_auction_self_camps_count     fair_auction_all_clients_count
                fair_auction_agency_clients_count fair_auction_manager_clients_count
                fair_auction_self_clients_count 
                /,
            ],
            count_distinct => [qw/fair_auction_agencies_count fair_auction_managers_count/],
        },
        sql => q/
        select count(*) as fair_auction_all_camps_count
             , sum(IF(c.AgencyUID > 0, 1, 0)) as fair_auction_agency_camps_count
             , sum(IF(c.ManagerUID > 0, 1, 0)) as fair_auction_manager_camps_count
             , sum(IF(c.ManagerUID is null and c.AgencyUID is null, 1, 0)) as fair_auction_self_camps_count
             , count(distinct c.uid) as fair_auction_all_clients_count
             , count(distinct IF(c.AgencyUID > 0, c.uid, null)) as fair_auction_agency_clients_count
             , count(distinct IF(c.ManagerUID > 0, c.uid, null)) as fair_auction_manager_clients_count
             , count(distinct IF(c.ManagerUID is null and c.AgencyUID is null, c.uid, null)) as fair_auction_self_clients_count
             , group_concat(distinct c.AgencyUID) as fair_auction_agencies_count
             , group_concat(distinct c.ManagerUID) as fair_auction_managers_count
        from camp_options co
          join campaigns c on c.cid = co.cid
          left join campaigns wc on wc.cid = c.wallet_cid
        where co.fairAuction = 'Yes'
          and c.statusEmpty = 'No'
          and c.sum - c.sum_spent + IF(c.wallet_cid, wc.sum - wc.sum_spent, 0) > 0.1
          and c.statusShow = 'Yes'
        /
    };

    do_sql(PPC(shard => 'all'), "SET SESSION group_concat_max_len = 10000000");
    my $raw_fair_auction = overshard_get_one_line($sql_fair_auction->{overshard}, PPC(shard => 'all'), $sql_fair_auction->{sql}) || {};

    my $result = {};
    while (my ($id, $value) = each %$raw_fair_auction) {
        $result->{$id}->{value} = $value;
        $result->{$id}->{desc} = $target_desc{$id};
    }

    return $result;
}

# статистика по клиентам
sub calc_clients_stats :Monitor
{
    my $stats = overshard_get_one_line({ group => '', sum => [ qw/total without_country_count/ ] }, PPC(shard => 'all'), q/
        SELECT COUNT(*) AS total
             , SUM(IF(country_region_id = 0, 1, 0)) AS without_country_count
          FROM (
                    SELECT IFNULL(cl.country_region_id,0) AS country_region_id
                      FROM users u
                           LEFT JOIN clients cl USING(ClientID)
                  GROUP BY u.ClientID
               ) t
    /) // {};

    my $client_types = overshard_get_one_line(
        { group => '', sum => [ qw/work_clients work_clients_agency work_clients_manager work_clients_self/ ] },
        PPC(shard => 'all'), q/
            SELECT COUNT(*) AS work_clients
                 , SUM(IF(t.agency, 1, 0)) AS work_clients_agency
                 , SUM(IF(t.manager, 1, 0)) AS work_clients_manager
                 , SUM(IF(t.agency = 0 AND t.manager = 0, 1, 0)) AS work_clients_self
              FROM (
                        SELECT SUM(IF(c.AgencyUID, 1, 0)) AS agency
                             , SUM(IF(c.ManagerUID, 1, 0)) AS manager
                          FROM campaigns c
                               LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid
                         WHERE c.statusEmpty = 'No'
                               AND c.sum - c.sum_spent + IF(wc.cid, wc.sum - wc.sum_spent, 0) > 0.1
                               AND c.statusShow = 'Yes'
                      GROUP BY c.uid
                    ) t
    /) // {};

    return {
        'direct.clients.total' => {
            value => $stats->{total},
            desc  => 'Количество клиентов - всего (уникальных ClientID)',
        },
        'direct.clients.with_unknown_country' => {
            value => $stats->{without_country_count},
            desc  => 'Количество клиентов - с неизвестной страной (country_region_id = 0)',
        },
        'direct.clients.work' => {
            old_name    => 'work_clients',
            desc        => 'активных клиентов',
            value       => $client_types->{work_clients},
        },
        'direct.clients.work_self' => {
            old_name    => 'work_clients_self',
            desc        => 'активных самостоятельных клиентов',
            value       => $client_types->{work_clients_self},
        },
        'direct.clients.work_agency' => {
            old_name    => 'work_clients_agency',
            desc        => 'активных агентских клиентов',
            value       => $client_types->{work_clients_agency},
        },
        'direct.clients.work_manager' => {
            old_name    => 'work_clients_manager',
            desc        => 'активных, обслуживаемых менеджером клиентов',
            value       => $client_types->{work_clients_manager},
        },
    };
}


# выставлены новые стратегии
sub calc_new_startegy_values :Monitor
{
    my $res = {};

    my %indication = (
        # Недельный пакет кликов
        camps_with_strategy_autobudget_limit_clicks => {
            desc => 'Стратегии: кампаний с деньгами и со стратегией "Недельный пакет кликов"',
            desc_clients => 'Стратегии: клиентов с деньгами и со стратегией "Недельный пакет кликов"',
            sql => " (c.strategy_name='autobudget_week_bundle') ",
        },

        # кампаний с "Средняя цена клика (за неделю)"
        camps_with_strategy_autobudget_avg_bid  => {
            desc => 'Стратегии: кампаний с деньгами и со стратегией "Средняя цена клика (за неделю)"',
            desc_clients => 'Стратегии: клиентов с деньгами и со стратегией "Средняя цена клика (за неделю)"',
            # !!! раньше было некорректное условие, сюда попадали все кампании с параметром avg_bid
            sql => " (c.strategy_name='autobudget_avg_click') ",
        },

        # кампаний с "Средняя цена конверсии (за неделю)"
        camps_with_strategy_autobudget_avg_cpa  => {
            desc => 'Стратегии: кампаний с деньгами и со стратегией "Средняя цена конверсии (за неделю)"',
            desc_clients => 'Стратегии: клиентов с деньгами и со стратегией "Средняя цена конверсии (за неделю)"',
            sql => " (c.strategy_name='autobudget_avg_cpa') ",
        },

        # кампаний с "Средняя цена установки"
        camps_with_strategy_autobudget_avg_cpi  => {
            desc => 'Стратегии: кампаний с деньгами и со стратегией "Средняя цена установки"',
            desc_clients => 'Стратегии: клиентов с деньгами и со стратегией "Средняя цена установки"',
            sql => " (c.strategy_name='autobudget_avg_cpi') ",
        },

        # кампаний с "CPA оптимизацией"
        camps_with_cpa_optimization  => {
            desc => 'Стратегии: кампаний с деньгами и со стратегией "Недельный бюджет (по конверсии)"',
            desc_clients => 'Стратегии: клиентов с деньгами и со стратегией "Недельный бюджет (по конверсии)"',
            # !!! раньше было некорректное условие, сюда попадали все кампании с параметром goal_id
            sql => " (c.strategy_name='autobudget' and if(json_type(c.strategy_data->>'\$.goal_id')='NULL', NULL, c.strategy_data->>'\$.goal_id') is not null) ",
        },

        camps_with_strategy_autobudget_roi  => {
            desc => 'Стратегии: кампаний с деньгами и со стратегией "Рентабельность рекламы"',
            desc_clients => 'Стратегии: клиентов с деньгами и со стратегией "Рентабельность рекламы"',
            sql => " c.strategy_name='autobudget_roi' ",
        },
    );

    for my $key (keys %indication) {
        # Присоединяем другие таблицы по необходимости (для уменьшения времени выборки)
        my @joins_sql;
        push @joins_sql, "left join camp_options co on c.cid = co.cid" if $indication{$key}->{sql} =~ /\bco\./;

        my $values = overshard_get_one_line( { group => '', sum => [ $key, "${key}_manager", "${key}_agency", "${key}_self", "${key}_uniq_clients" ] },
            PPC(shard => 'all'),
            "select SUM(IF( $indication{$key}->{sql}, 1, 0)) as $key, 
                    SUM(IF( $indication{$key}->{sql} and c.ManagerUID is not null, 1, 0)) as ${key}_manager,
                    SUM(IF( $indication{$key}->{sql} and c.AgencyUID  is not null, 1, 0 )) as ${key}_agency,
                    SUM(IF( $indication{$key}->{sql} and c.ManagerUID is null and c.AgencyUID is null , 1, 0 )) as ${key}_self,
                    COUNT(distinct c.uid) as ${key}_uniq_clients
             from campaigns c
               left join campaigns wc on wc.cid = c.wallet_cid
               ".join("\n", @joins_sql)."
             where c.sum - c.sum_spent + IF(c.wallet_cid, wc.sum - wc.sum_spent, 0) >= 0.01
               and c.autobudget = 'Yes'
               and $indication{$key}->{sql}
            "
        );

        $res->{$_}->{value} = $values->{$_} || 0 for keys %$values;
        $res->{$key}->{desc} = $indication{$key}->{desc}." - всего";
        $res->{"${key}_manager"}->{desc} = $indication{$key}->{desc}." - менеджерских";
        $res->{"${key}_agency"}->{desc} = $indication{$key}->{desc}." - агентских";
        $res->{"${key}_self"}->{desc} = $indication{$key}->{desc}." - самостоятельных";
        $res->{"${key}_uniq_clients"}->{desc} = $indication{$key}->{desc_clients};
    }

    return $res;
}

# выставлена стратегия "показ в блоке справа"
sub calc_strategy_no_premium_values :Monitor
{
    my $res = {};

    my %indication = (
        # кампаний со стратегией "показ в блоке справа"
        camps_with_strategy_no_premium  => {
            desc => 'Стратегии: кампаний с деньгами и со стратегией "показ в блоке справа"',
            desc_clients => 'Стратегии: клиентов с деньгами и со стратегией "показ в блоке справа"',
            sql => " c.strategy_name='no_premium' ",
        },
    );

    for my $key (keys %indication) {
        my $values = overshard_get_one_line( { group => '', sum => [ $key, "${key}_manager", "${key}_agency", "${key}_self", "${key}_uniq_clients" ] },
            PPC(shard => 'all'),
            "select SUM(IF( $indication{$key}->{sql}, 1, 0)) as $key,
                    SUM(IF( $indication{$key}->{sql} and c.ManagerUID is not null, 1, 0)) as ${key}_manager,
                    SUM(IF( $indication{$key}->{sql} and c.AgencyUID  is not null, 1, 0 )) as ${key}_agency,
                    SUM(IF( $indication{$key}->{sql} and c.ManagerUID is null and c.AgencyUID is null , 1, 0 )) as ${key}_self,
                    COUNT(distinct c.uid) as ${key}_uniq_clients
             from campaigns c
               left join campaigns wc on wc.cid = c.wallet_cid
             where c.sum - c.sum_spent + IF(c.wallet_cid, wc.sum - wc.sum_spent, 0) >= 0.01
               and $indication{$key}->{sql}
            "
        );

        $res->{$_}->{value} = $values->{$_} || 0 for keys %$values;
        $res->{$key}->{desc} = $indication{$key}->{desc}." - всего";
        $res->{"${key}_manager"}->{desc} = $indication{$key}->{desc}." - менеджерских";
        $res->{"${key}_agency"}->{desc} = $indication{$key}->{desc}." - агентских";
        $res->{"${key}_self"}->{desc} = $indication{$key}->{desc}." - самостоятельных";
        $res->{"${key}_uniq_clients"}->{desc} = $indication{$key}->{desc_clients};
    }

    return $res;
}

sub _rates {
    
    my $sql = join ' AND ', @_; 
    my $query = <<EOS;
        SELECT
            COUNT(*) `all`,
            SUM(IF($sql AND c.ManagerUID IS NOT NULL, 1, 0)) `manager`,
            SUM(IF($sql AND c.AgencyUID IS NOT NULL, 1, 0 )) `agency`,
            SUM(IF($sql AND c.ManagerUID IS NULL AND c.AgencyUID IS NULL, 1, 0 )) `self`,
            COUNT(DISTINCT c.uid) uniq_clients
        FROM
            campaigns c
            LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid
            JOIN camp_options co ON c.cid = co.cid
        WHERE
            c.sum - c.sum_spent + IF(c.wallet_cid, wc.sum - wc.sum_spent, 0) >= 0.1
        AND
            $sql
EOS
    
    return overshard_get_one_line( { group => '', sum => [ qw/all manager agency self uniq_clients/ ] }, PPC(shard => 'all'), $query);
}

# показатели по стратегии "Независимое размещение"
sub calc_strategy_different_places :Monitor
{
    my %indicators = (
        different_places => {
            desc => 'Стратегии: кампаний с деньгами и со стратегией "Независимое размещение"',
            desc_clients => 'Стратегии: клиентов с деньгами и со стратегией "Независимое размещение"',
            sql => q[co.strategy = 'different_places'], 
            kinds => {that => [qw/all manager agency self uniq_clients/]}
        },
        'different_places.search_stop' => {
            desc => 'Стратегии: кампаний с деньгами и со стратегией "Независимое размещение (%s)"',
            desc_clients => 'Стратегии: клиентов с деньгами и со стратегией "Независимое размещение (%s)"',
            sql => q[co.strategy = 'different_places' AND c.platform = 'context'],
            kinds => [
                {   
                    name => '',
                    label => 'поиск отключен', 
                    that => [qw/all manager agency self uniq_clients/]
                },
                {
                    name => 'autobudget_max_clicks',
                    label => 'недельный бюджет клики - в РСЯ',
                    # !!! условие поменялось, было некорректное (попадали другие стратегии)
                    sql => q[c.strategy_name = 'autobudget' and if(json_type(c.strategy_data->>'$.goal_id')='NULL', NULL, c.strategy_data->>'$.goal_id') IS NULL],
                    that => 'all'
                },
                {
                    name => 'cpa_optimization',
                    label => 'недельный бюджет конверсии - в РСЯ',
                    # !!! условие поменялось, было некорректное (попадали другие стратегии)
                    sql => q[c.strategy_name = 'autobudget' and if(json_type(c.strategy_data->>'$.goal_id')='NULL', NULL, c.strategy_data->>'$.goal_id') IS NOT NULL],
                    that => 'all'
                },
                {
                    name => 'autobudget_avg_bid',
                    label => 'средняя цена клика - в РСЯ',
                    # !!! условие поменялось, было некорректное (попадали другие стратегии)
                    sql => q[c.strategy_name = 'autobudget_avg_click'],
                    that => 'all'
                },
                {
                    name => 'autobudget_limit_clicks',
                    label => 'недельный пакет кликов - в РСЯ',
                    sql => q[c.strategy_name='autobudget_week_bundle'],
                    that => 'all'
                },
                {
                    name => 'autobudget',
                    label => 'поиск отключен, в РСЯ любая автостратегия',
                    sql => q[c.autobudget = 'Yes'],
                    that => [qw/all manager agency self uniq_clients/]
                },
            ]
        }
    );
    
    my %postfix = (
        all => 'всего', manager => 'менеджерских',
        agency => 'агентских', self => 'самостоятельных'
    );

    my %index;
    while (my ($key, $desc) = each %indicators) {

        foreach my $kind (ref $desc->{kinds} eq 'ARRAY' ? @{$desc->{kinds}} : $desc->{kinds}) {

            my $rates = _rates(grep {$_} $desc->{sql}, $kind->{sql});
            foreach my $role (ref $kind->{that} ? @{$kind->{that}} : $kind->{that}) {
                
                my $name = join '.', 'direct.camp_strategies', $key, $kind->{name} ? "$$kind{name}_$role" : $role;
                my $note = sprintf $desc->{$role =~ /uniq_clients/ ? 'desc_clients' : 'desc'},
                        $kind->{label};
                $note .= " - $postfix{$role}" if exists $postfix{$role};
                
                $index{$name} = {
                    value => $rates->{$role} || 0,
                    desc => $note
                }
            }
        }
    }
        
    return \%index;    
}

# замеры по использованию Дополнительных релевантных фраз
sub calc_broad_match_use :Monitor
{
    my $res = {};

    my $count = overshard_get_one_line({
            group => '',
            sum => [ qw/camps_count logins_count camps_count_by_rate_minimal
                camps_count_by_rate_optimal camps_count_by_rate_maximum
            /]
        },
        PPC(shard => 'all'), "
        SELECT COUNT(*) as camps_count
             , COUNT(DISTINCT(login)) logins_count
             , COUNT(IF(co.broad_match_rate = 'minimal', 1, NULL)) as camps_count_by_rate_minimal
             , COUNT(IF(co.broad_match_rate = 'optimal', 1, NULL)) as camps_count_by_rate_optimal
             , COUNT(IF(co.broad_match_rate = 'maximum', 1, NULL)) as camps_count_by_rate_maximum
        FROM camp_options co
          JOIN campaigns c ON c.cid = co.cid
          LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid
        JOIN users u ON u.uid = c.uid
        WHERE co.broad_match_flag = 'Yes'
          AND c.sum + IF(c.wallet_cid, wc.sum, 0) > 0
          AND c.archived = 'No'
          AND c.statusActive = 'Yes'
    ");

    my $banners_count = overshard_sum(PPC(shard => 'all'), "
        SELECT COUNT(*)
        FROM banners b
          JOIN campaigns c ON c.cid = b.cid
          LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid
          JOIN camp_options co ON co.cid = c.cid
        WHERE co.broad_match_flag = 'Yes'
          AND c.sum + IF(c.wallet_cid, wc.sum, 0) > 0
          AND c.archived = 'No'
          AND c.statusActive = 'Yes'
          AND b.statusActive = 'Yes'
    ");

    $res->{'broadmatch.camps_count'} = {
        value => $count->{camps_count},
        desc  => "Дополнительные релевантные фразы — количество активных кампаний"
    };

    $res->{'broadmatch.logins_count'} = {
        value => $count->{logins_count},
        desc  => "Дополнительные релевантные фразы — количество уникальных пользователей"
    };

    $res->{'broadmatch.banners_count'} = {
        value => $banners_count,
        desc  => "Дополнительные релевантные фразы — количество активных объявлений"
    };

    $res->{'broadmatch.camps_count_by_rate_minimal'} = {
        value => $count->{camps_count_by_rate_minimal},
        desc => "Дополнительные релевантные фразы — количество активных кампаний (минимальный)",
    };

    $res->{'broadmatch.camps_count_by_rate_optimal'} = {
        value => $count->{camps_count_by_rate_optimal},
        desc => "Дополнительные релевантные фразы — количество активных кампаний (оптимальный)",
    };

    $res->{'broadmatch.camps_count_by_rate_maximum'} = {
        value => $count->{camps_count_by_rate_maximum},
        desc => "Дополнительные релевантные фразы — количество активных кампаний (максимальный)",
    };

    return $res;
}

# данные по кампаниям с установленной датой окончания
sub calc_finish_date_values :Monitor
{
    my $res = overshard_get_one_line({ group => '', sum => [qw/camps_cnt camps_stopped_cnt clients_cnt stopped_sum_rest/] },
        PPC(shard => 'all'), q/
        SELECT STRAIGHT_JOIN
               COUNT(*) AS camps_cnt,
               COUNT( IF(DATE(NOW()) > DATE(c.finish_time), 1, NULL) ) AS camps_stopped_cnt,
               COUNT( DISTINCT u.ClientID ) AS clients_cnt,
               SUM( IF(DATE(NOW()) > DATE(c.finish_time), c.sum - c.sum_spent + IF(c.wallet_cid, wc.sum - wc.sum_spent, 0), 0) ) AS stopped_sum_rest
        FROM campaigns c
        LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid
        JOIN users u ON u.uid = c.uid
        WHERE c.sum - c.sum_spent + IF(c.wallet_cid, wc.sum - wc.sum_spent, 0) > 0.01
          AND YEAR(c.finish_time)
    /);
    return {
        'direct.finish_date.campaigns.with_money' => {
            value => $res->{camps_cnt},
            desc => 'Количество кампаний с деньгами и установленной Датой окончания',
        },
        'direct.finish_date.campaigns.stopped_by_finish_date' => {
            value => $res->{camps_stopped_cnt},
            desc => 'Количество кампаний с деньгами и установленной Датой окончания (просроченные) - остановленные по достижению даты окончания',
        },
        'direct.finish_date.clients.with_money' => {
            value => $res->{clients_cnt},
            desc => 'Количество клиентов с деньгами и установленной Датой окончания',
        },
        'direct.finish_date.money.sum_rest' => {
            value => $res->{stopped_sum_rest},
            desc => 'Сумма непотраченных средств на кампаниях, остановленных по достижению даты окончания',
        },
    };
}

# данные по кампаниям с дневным бюджетом
sub calc_day_budget_values :Monitor
{
    my $res = overshard_get_one_line({
            group => '',
            sum => [ qw/camps_total camps_agency camps_manager
                camps_byown clients_total clients_byown
            /],
            count_distinct => [qw/clients_agency clients_manager/],
        },
        PPC(shard => 'all'), q/
        SELECT STRAIGHT_JOIN
            COUNT( DISTINCT c.cid ) AS camps_total,
            COUNT( DISTINCT IF(IFNULL(c.AgencyUID, 0) > 0, c.cid, NULL) ) AS camps_agency,
            COUNT( DISTINCT IF(IFNULL(c.ManagerUID,0) > 0, c.cid, NULL) ) AS camps_manager,
            COUNT( DISTINCT IF(NOT IFNULL(c.AgencyUID,0) > 0 AND NOT IFNULL(c.ManagerUID,0) > 0, c.cid, NULL) ) AS camps_byown,
            COUNT( DISTINCT u.ClientID ) AS clients_total,
            GROUP_CONCAT( DISTINCT IF(IFNULL(c.AgencyUID,0) > 0, u.ClientID, NULL) ) AS clients_agency,
            GROUP_CONCAT( DISTINCT IF(IFNULL(c.ManagerUID,0) > 0, u.ClientID, NULL) ) AS clients_manager,
            COUNT( DISTINCT IF(NOT IFNULL(c.AgencyUID,0) > 0 AND NOT IFNULL(c.ManagerUID,0) > 0, u.ClientID, NULL) ) AS clients_byown
        FROM campaigns c
        LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid
        JOIN users u ON u.uid = c.uid
        WHERE
            c.day_budget > 0
            AND c.sum - c.sum_spent + IF(c.wallet_cid, wc.sum - wc.sum_spent, 0) > 0.01
    /);
    # пока, в соответствии с тикетом DIRECT-13130, помещаем значения в категорию Разное
    # в дальнейшем можно будет заменить подчёркивания на пробелы и перенести в иерархическое представление
    return {
        'direct_day_budget_campaigns_with_money_total' => {
            value => $res->{camps_total},
            desc => 'Дневной бюджет: кампаний с деньгами - всего',
        },
        'direct_day_budget_campaigns_with_money_agency' => {
            value => $res->{camps_agency},
            desc => 'Дневной бюджет: кампаний с деньгами - агентских',
        },
        'direct_day_budget_campaigns_with_money_manager' => {
            value => $res->{camps_manager},
            desc => 'Дневной бюджет: кампаний с деньгами - менеджерских',
        },
        'direct_day_budget_campaigns_with_money_byown' => {
            value => $res->{camps_byown},
            desc => 'Дневной бюджет: кампаний с деньгами - самостоятельных',
        },
        'direct_day_budget_clients_with_money_total' => {
            value => $res->{clients_total},
            desc => 'Дневной бюджет: клиентов с деньгами - всего',
        },
        'direct_day_budget_clients_with_money_agency' => {
            value => $res->{clients_agency},
            desc => 'Дневной бюджет: клиентов с деньгами - агентских',
        },
        'direct_day_budget_clients_with_money_manager' => {
            value => $res->{clients_manager},
            desc => 'Дневной бюджет: клиентов с деньгами - менеджерских',
        },
        'direct_day_budget_clients_with_money_byown' => {
            value => $res->{clients_byown},
            desc => 'Дневной бюджет: клиентов с деньгами - самостоятельных',
        },
    };
}

# замеры по использованию Меток
sub calc_tags :Monitor
{
    my $res = {};

    my $users_count = overshard_sum(PPC(shard => 'all'), "SELECT COUNT(DISTINCT(uid)) FROM users ".
                                            "JOIN campaigns USING(uid) ".
                                            "JOIN tag_campaign_list USING(cid) WHERE statusEmpty='No'");

    # кажется, до DIRECT-68731 тут было неправильное условие:
    # в all_camps_count считалось количество лёгких кампаний, хотя ниже по коду подразумевается проф
    my $all_camps_count = overshard_sum(PPC(shard => 'all'), "SELECT  COUNT(*) FROM campaigns JOIN users_options USING(uid) ".
                                            "WHERE statusEmpty='No'") || 1;

    my $tag_camps_count = overshard_sum(PPC(shard => 'all'), "SELECT COUNT(cid) FROM ".
                                            "(SELECT DISTINCT(cid) FROM tag_campaign_list GROUP BY cid HAVING COUNT(*)>2) c ".
                                            "JOIN campaigns USING(cid) JOIN users_options USING(uid) WHERE statusEmpty='No'");

    $res->{'tags.users_count'} = {
        value => $users_count,
        desc  => "Метки — количество пользователей, использующих метки"
    };

    $res->{'tags.camps_count'} = {
        value => $tag_camps_count,
        desc  => "Метки — количество кампаний с 2 и более метками"
    };

    $res->{'tags.camps_percent'} = {
        value => $tag_camps_count * 100 / $all_camps_count,
        desc  => "Метки — процент кампаний с 2 и более метками (только Проф. интерфейс)",
        units => "proc"
    };

    return $res;
}

# --------------------------------------------------------------------
# статистика по ретаргетингу
sub calc_retargeting_values :Monitor
{
    my $res = {};

    my @objects = (
        {desc => "клиентов", field => "c.uid", name => "clients"},
        {desc => "кампаний", field => "c.cid", name => "campaigns"},
        {desc => "баннеров", field => "b.bid", name => "banners"},
    );

    my @serv_types = (
        {desc => "всего",           cond => "1", name => "all"},
        {desc => "менеджерских",    cond => "c.ManagerUID > 0", name => "manager"},
        {desc => "агентских",       cond => "c.AgencyUID > 0", name => "agency"},
        {desc => "самостоятельных", cond => "c.ManagerUID is NULL and c.AgencyUID is NULL", name => "self"},
    );

    my @SQL_SELECT;
    my %overshard_opt;
    my %all_desc_by_name;

    for my $object (@objects) {
        for my $serv_type (@serv_types) {
            for my $segment_type ("", "_segment") {

                my $field_name = "$object->{name}_$serv_type->{name}$segment_type";
                my $desc = sprintf("условие подбора аудитории%s: $serv_type->{desc} $object->{desc}", $segment_type ? " с сегментами" : "");
                my $cond = $serv_type->{cond};
                if ($segment_type) {
                    $cond .= " and rg.goal_type = 'segment'"
                }

                push @SQL_SELECT, sprintf("COUNT(DISTINCT IF(%s, %s, NULL)) as %s",
                                          $cond, $object->{field},
                                          $field_name
                                         );
                push @{$overshard_opt{sum}}, $field_name;

                $all_desc_by_name{$field_name} = $desc;
            }
        }
    }
    my $SQL = join(",\n", @SQL_SELECT);

    my $values = overshard_get_one_line({ group => '', %overshard_opt },
        PPC(shard => 'all'),
        "select $SQL
         from campaigns c
            left join campaigns wc on wc.cid = c.wallet_cid
            join phrases p ON c.cid = p.cid
            join banners b ON b.pid = p.pid
            join bids_retargeting br ON br.pid = p.pid
            join retargeting_goals rg on rg.ret_cond_id = br.ret_cond_id
         where c.sum - c.sum_spent + IF(c.wallet_cid, wc.sum - wc.sum_spent, 0) >= 0.01
           and b.BannerID > 0
           and b.statusArch = 'No'
        "
    );

    for my $key (keys %$values) {
        $res->{"retargeting_$key"} = {
            desc => $all_desc_by_name{$key},
            value => $values->{$key} || 0,
        };
    }

    return $res;
}

# --------------------------------------------------------------------
# статистика по кампаниям с общим счетом
# https://jira.yandex-team.ru/browse/DIRECT-21279
# тестирование:
#    perl -ME -MDirect::Monitor::Daily -e 'local $Yandex::DBShards::STRICT_SHARD_DBNAMES = 0; p(Direct::Monitor::Daily::calc_wallet_values())'
#    perl protected/ppcMonitorDaily.pl --only-sub=calc_wallet_values
sub calc_wallet_values :Monitor
{
    my $res = {};

    my @serv_types = (
        {desc => "всего",
         cond_cl => "%s",
         cond_sum => "%s",
         name => "all"
        },
        {desc => "агентских",
         cond_cl => "IF(c.AgencyUID > 0, %s, NULL)",
         cond_sum => "IF(wc.AgencyUID > 0, %s, 0)",
         name => "agency"
        },
        {desc => "самостоятельных+менеджерских",
         cond_cl => "IF(IFNULL(c.AgencyUID, 0) = 0, %s, NULL)",
         cond_sum => "IF(IFNULL(wc.AgencyUID, 0) = 0, %s, 0)",
         name => "self"
        },
    );

    my @SQL_SELECT;

    # кол-во клиентов ...............
    my %overshard_opt;
    for my $serv_type (@serv_types) {
        push @SQL_SELECT, sprintf("COUNT(DISTINCT %s) as %s",
                                  sprintf($serv_type->{cond_cl}, "c.uid"),
                                  "clients_$serv_type->{name}",
                                 );
        push @{$overshard_opt{sum}}, "clients_$serv_type->{name}";
    }
    my $SQL = join(",\n", @SQL_SELECT);

    my $clients_values = overshard_get_one_line({ group => '', %overshard_opt },
        PPC(shard => 'all'),
        "select $SQL
         from wallet_campaigns
           join campaigns c on wallet_campaigns.wallet_cid = c.wallet_cid
         where c.wallet_cid > 0
        "
    );

    # суммы на общих счетах .........
    @SQL_SELECT = ();
    %overshard_opt = ();

    # конвертация валют
    my $rates = {map {$_ => convert_currency(1, $_, "YND_FIXED", with_nds => 1)} keys %Currencies::_CURRENCY_DESCRIPTION};
    my $currency_sql_case = sql_case("wc.currency", $rates, default => 1);

    for my $serv_type (@serv_types) {
        push @SQL_SELECT, sprintf("SUM(%s) as %s",
                                  sprintf($serv_type->{cond_sum}, "(wc.sum - wc.sum_spent) * $currency_sql_case"),
                                  "sums_$serv_type->{name}",
                                 );
        push @{$overshard_opt{sum}}, "sums_$serv_type->{name}";
    }
    $SQL = join(",\n", @SQL_SELECT);

    my $sums_values = overshard_get_one_line({ group => '', %overshard_opt },
        PPC(shard => 'all'),
        "select $SQL
         from wallet_campaigns
           join campaigns wc on wallet_campaigns.wallet_cid = wc.cid
         where wc.type = 'wallet'
        "
    );

    for my $serv_type (@serv_types) {
        my $key_clients = "clients_$serv_type->{name}";
        $res->{"wallet_$key_clients"} = {
            desc => "клиентов с общим счетом: $serv_type->{desc}",
            value => $clients_values->{$key_clients} || 0,
        };

        my $key_sums = "sums_$serv_type->{name}";
        $res->{"wallet_$key_sums"} = {
            desc => "остатков на общих счетах: $serv_type->{desc}",
            value => $sums_values->{$key_sums} || 0,
        };
    }

    return $res;
}

sub multicurrency_values :Monitor
{
    my $currency2clients_cnt = {
        map { $_->{currency} => $_->{clients_cnt} }
        @{
            overshard group => 'currency', sum => 'clients_cnt', 
            get_all_sql(PPC(shard => 'all'), q/
            SELECT
                IFNULL(cl.work_currency, "YND_FIXED") AS currency,
                COUNT( DISTINCT u.ClientID ) AS clients_cnt
            FROM users u
            LEFT JOIN clients cl ON u.ClientID = cl.ClientID
            GROUP BY 1
        /)
        }
    };
    my $ynd_fixed_clients_without_agency_cnt = overshard_sum(PPC(shard => 'all'), q/
        SELECT COUNT(DISTINCT u.ClientID)
        FROM users u
        LEFT JOIN clients cl ON u.ClientID = cl.ClientID
        INNER JOIN campaigns c ON u.uid = c.uid
        WHERE
            IFNULL(cl.work_currency, "YND_FIXED") = "YND_FIXED"
            AND IFNULL(c.AgencyUID, 0) = 0
    /);
    my $active_multicurrency_clients_cnt = overshard_sum(PPC(shard => 'all'), q/
        SELECT COUNT(DISTINCT cl.ClientID)
        FROM clients cl
        INNER JOIN users u ON cl.ClientID = u.ClientID
        INNER JOIN campaigns c ON u.uid = c.uid
        LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid
        WHERE
            cl.work_currency <> "YND_FIXED"
            AND c.sum + IF(c.wallet_cid, wc.sum, 0) > c.sum_spent + IF(c.wallet_cid, wc.sum_spent, 0)
            AND c.statusShow = 'Yes'
            AND c.statusModerate = 'Yes'
            AND c.archived = 'No'
            AND c.statusEmpty = 'No'
    /);
    my $multicurrency_currency2clients_cnt = hash_kgrep { $_ ne 'YND_FIXED' } $currency2clients_cnt;
    my $multicurrency_clients_cnt = sum 0, values %$multicurrency_currency2clients_cnt;

    my $PREFIX = 'direct.multicurrency';
    my %values = (
        "$PREFIX.clients_cnt.multicurrency_total" => {
            value => $multicurrency_clients_cnt,
            desc => 'Количество клиентов в валютах (всего)',
            units => 'num',
        },
        "$PREFIX.clients_cnt.multicurrency_active" => {
            value => $active_multicurrency_clients_cnt,
            desc => 'Количество активных клиентов в валютах',
            units => 'num',
        },
        "$PREFIX.clients_cnt.YND_FIXED_without_agency" => {
            value => $ynd_fixed_clients_without_agency_cnt,
            desc => 'Количество клиентов в у.е. (без субклиентов агентств)',
            units => 'num',
        },
    );
    while ( my($currency, $clients_cnt) = each %$currency2clients_cnt ) {
        $values{"$PREFIX.clients_cnt.$currency"} = {
            value => $clients_cnt,
            desc => 'Количество клиентов в ' . format_currency($currency),
            units => 'num',
        };
    }

    return \%values;
}

# --------------------------------------------------------------------

=head2 calc_timetargeting_values

200% во временном таргетинге
https://st.yandex-team.ru/DIRECT-39999

добавляем раздел "timetargeting". Делаем в разрезе по каналам всего/агентских/менеджерских/самостоятельных
Считаем количество активных кампаний с понижающими коэффициентами (если хотя бы для одного часа задан процент 90 и меньше, но не равно нулю)
Считаем количество активных кампаний с повышающими коэффициентами (если хотя бы для одного часа задан процент 110 и больше)
Распределение коэффициентов (10%, 20%, 30% и так далее) в активных кампаниях от всех активных кампаний.#

тестирование:
   perl -ME -MDirect::Monitor::Daily -e 'p(Direct::Monitor::Daily::calc_timetargeting_values())'
   perl protected/ppcMonitorDaily.pl --only-sub=calc_timetargeting_values

=cut

sub calc_timetargeting_values :Monitor
{
    # типы сервисирования
    my @serv_type = (
        {
            name => "all",
            cond => undef,
            desc => "все",
        },
        {
            name => "manager",
            cond => 'c.ManagerUID > 0',
            desc => "менеджерские"
        },
        {
            name => "agency",
            cond => 'c.AgencyUID > 0',
            desc => "агентские"
        },
        {
            name => "self",
            cond => 'IFNULL(c.AgencyUID, 0) = 0 AND IFNULL(c.ManagerUID, 0) = 0',
            desc => "самоходные"
        },
    );

    my @types = (
        {desc => "всего активных кампаний",
         cond => undef,
         name => "all_camps",
        },
        {desc => "количество активных кампаний с понижающими коэффициентами",
         cond => "CAST(c.timeTarget AS BINARY) REGEXP '^[^;]*[b-j]'",
         name => "negative_camps",
        },
        {desc => "количество активных кампаний с повышающими коэффициентами",
         cond => "CAST(c.timeTarget AS BINARY) REGEXP '^[^;]*[l-u]'",
         name => "positive_camps",
        },
    );

    my @all_coef = (
        map {
            {letter => TimeTarget::coef2letter($_), coef => $_}
        }
        map {
            $_ * 10
        }
        1 .. 9, 11 .. 20
    );

    my $num = 0;
    for my $coef_row (@all_coef) {
        push @types, {
            desc => "кампаний с коэффициентом: $coef_row->{coef}%",
            cond => "CAST(c.timeTarget AS BINARY) REGEXP '^[^;]*$coef_row->{letter}'",
            name => "coef_$coef_row->{coef}",
        };
    }

    my @SQL_SELECT;
    my %overshard_opt;
    my %desc_by_name;

    for my $row (@types) {
        for my $serv_type (@serv_type) {
            my $field_name = "$row->{name}_$serv_type->{name}";
            push @{$overshard_opt{sum}}, $field_name;
            my $select;

            if (! $row->{cond} && ! $serv_type->{cond}) {
                $select = 'c.cid';
            } else {
                my $cond = join " AND ", grep {defined $_} $row->{cond}, $serv_type->{cond};
                $select = sprintf("IF(%s, c.cid, NULL)", $cond);
            }

            push @SQL_SELECT, sprintf("COUNT(%s) as %s",
                                      $select,
                                      $field_name,
                                     );
    
            $desc_by_name{$field_name} = "$row->{desc} ($serv_type->{desc})";
        }
    }
    my $SQL = join(",\n", @SQL_SELECT);

    my $values = overshard_get_one_line({group => '', %overshard_opt},
        PPC(shard => 'all'),
        "select $SQL
         from campaigns c
           left join campaigns wc on wc.cid = c.wallet_cid
         where
           c.sum + IF(c.wallet_cid, wc.sum, 0) > c.sum_spent + IF(c.wallet_cid, wc.sum_spent, 0)
           and c.statusShow = 'Yes'
           and c.statusModerate = 'Yes'
           and c.archived = 'No'
           and c.statusEmpty = 'No'
         "
    );

    my $res = {};
    for my $name (keys %desc_by_name) {
        $res->{"timetargeting.$name"} = {
            desc => $desc_by_name{$name},
            value => $values->{$name}
        };
    }

    return $res;
}

1;
