package Direct::Organizations;

use Direct::Modern;
use Yandex::HashUtils qw/hash_cut/;

=head2 to_hash

    Формирует из ответа JavaIntapi::GetOrganizationsInfo хеш, пригодный
    к отправке в старый интерфейс

=cut
sub to_hash {
    my ($organization_raw) = @_;

    my $result = hash_cut($organization_raw, qw/companyName statusPublish profileUrl isOnline/);
    $result->{phone} = $organization_raw->{phone}{formatted};
    $result->{address} = _get_organization_city($organization_raw->{address}{components});
    $result->{workTimes} = _get_work_times($organization_raw->{workIntervals});
    return $result;
}

=head3 _get_organization_city

    Вытаскивает из структуры организации locality и склеивает их через запятую в строку.
    В этом поле должен лежать город либо населенный пункт.

=cut
sub _get_organization_city {
    my ($components) = @_;

    my @values = map { $_->{name}{value} } grep { $_->{kind} eq 'locality' } @{$components};
    return join(", ", @values);
}

=head3 _get_work_times

    Преобразует формат даты-времени из отдаваемого API Справочника 
    к принимаемому фронтом Директа

=cut
sub _get_work_times {
    my ($work_intervals) = @_;

    my $work_times = [];
    for my $work_interval (@$work_intervals) {
        my $converted = {
            daysOfWeek => _get_work_time_day($work_interval->{day}),
            startTime  => _convert_time($work_interval->{time_minutes_begin}),
            endTime    => _convert_time($work_interval->{time_minutes_end}),
        };
        push @{$work_times}, $converted;
    }
    return $work_times;
}

=head3 _get_work_time_day

    Преобразует дни недели, передаваемые API Справочника, в их номера для
    интерфейса Директа.

=cut
sub _get_work_time_day {
    my ($day) = @_;

    my %DAYS_TO_NUM = (
        "monday"    => [ 0 ],
        "tuesday"   => [ 1 ],
        "wednesday" => [ 2 ],
        "thursday"  => [ 3 ],
        "friday"    => [ 4 ],
        "saturday"  => [ 5 ],
        "sunday"    => [ 6 ],
        "weekdays"  => [ 0, 1, 2, 3, 4 ],
        "weekend"   => [ 5, 6 ],
        "everyday"  => [ 0, 1, 2, 3, 4, 5, 6 ],
    );

    return $DAYS_TO_NUM{$day};
}

=head3 _convert_time

    Преобразует отдаваемое API Справочника время в минутах от начала суток
    к хешу "часы-минуты"

=cut
sub _convert_time {
    my ($minutes) = @_;

    return {
        hour    => int($minutes / 60),
        minutes => $minutes % 60,
    };
}

1;
