package Direct::PerformanceFilters;

use Mouse;
with qw/Direct::Role::Copyable/;

use Direct::Modern;

use Settings;

use Direct::Model::PerformanceFilter;
use Direct::Model::PerformanceFilter::Manager;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::HashUtils qw/hash_cut/;
use List::MoreUtils qw/any pairwise uniq/;

use LogTools;
use ShardingTools qw/choose_shard_param/;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::PerformanceFilter]');
has 'data'  => (is => 'ro', isa => 'HashRef', init_arg => undef, lazy => 1, default => sub { +{}; });

around BUILDARGS => sub { my ($orig, $class) = (shift, shift); $class->$orig(@_ == 1 ? (items => $_[0]) : @_) };

=head2 manager_class
=cut

sub manager_class { 'Direct::Model::PerformanceFilter::Manager' }

=head2 WEB_FIELD_NAMES

Название полей в web интерфейсе для расшифровки результатов валидации.

=cut

sub WEB_FIELD_NAMES {(
    filter_name         => {field => sprintf('"%s"', iget('Название фильтра'))},
    price_cpc           => {field => iget('CPC')},
    price_cpa           => {field => iget('CPA')},
    autobudget_priority => {field => sprintf('"%s"', iget('Приоритет автобюджета'))},
    target_funnel       => {field => sprintf('"%s"', iget('Воронка целей'))},
)}

my %KEYS_MAP = (cid => 'campaign_id', gid => 'adgroup_id', pid => 'adgroup_id');

=head2 get_by($key, $vals, %options)

По заданному критерию возвращает instance с выбранными перфоманс фильтрами.

Параметры:
    $key      -> по какому ключу выбирать: campaign_id/adgroup_id/perf_filter_id
    $vals     -> scalar|arrayref; значение ключа
    %options:
        'shard_key'     -> (если $key равен 'perf_filter_id'); идентификатор для выбора номера шарда,
                            должен быть одним из @{$ShardingTools::DEFAULT_MAPPING_SHARD_KEYS}
        limit/offset    -> параметры для постраничной выборки
        with_additional -> выбирать также дополнительные данные (`from_tab`)
        with_deleted    -> выбирать также удаленные фильтры
        filter          -> HashRef; дополнительный фильтр

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;

    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};
    croak "only `campaign_id/adgroup_id/perf_filter_id` keys are supported" unless $key =~ /^(?:campaign|adgroup|perf_filter)_id$/;

    my $self = $class->new(items => []);
    return $self if !defined $vals || (ref($vals) eq 'ARRAY' && !@$vals);

    my @shard;
    my %where = %{$options{filter} // {}};
    if ($key eq 'adgroup_id') {
        @shard = (pid => $vals);
        $where{'bpf.pid'} = SHARD_IDS;
    } elsif ($key eq 'campaign_id') {
        @shard = (cid => $vals);
        $where{'g.cid'} = SHARD_IDS;
    } else {
        # perf_filter_id
        @shard = $options{shard} ? (shard => $options{shard}) : choose_shard_param(\%options);
        $where{'bpf.perf_filter_id'} = $vals;
    }

    my (@select_columns, @from_tables);

    push @select_columns,
        Direct::Model::PerformanceFilter->get_db_columns(bids_performance => 'bpf', prefix => ''),
        'g.cid AS campaign_id';

    push @from_tables,
        'bids_performance bpf',
        'JOIN phrases g ON (g.pid = bpf.pid)',
        'LEFT JOIN adgroups_performance gp on gp.pid = g.pid',
        'LEFT JOIN feeds on feeds.feed_id = gp.feed_id';

    push @select_columns, 'feeds.feed_id, feeds.business_type, feeds.feed_type';

    $where{'bpf.is_deleted'} = 0 if !$options{with_deleted};

    my $perf_filter_rows = get_all_sql(PPC(@shard), [
        sprintf("SELECT %s FROM %s", join(', ', @select_columns), join(' ', @from_tables)),
        where => \%where,
        'ORDER BY bpf.perf_filter_id',
        $options{limit} ? (
            limit => $options{limit}, $options{offset} ? (offset => $options{offset}) : (),
        ) : (),
    ]);

    return $self unless @$perf_filter_rows;

    if (!$options{with_additional}) {
        # поле from_tab должно отдаваться только при установленной опции with_additional
        delete @$_{qw/from_tab/} for grep { !$_->{perf_filter_id} } @$perf_filter_rows;
    }

    for my $pf_row (@$perf_filter_rows) {
        if ($pf_row->{feed_id} && $pf_row->{feed_type}) {
            $pf_row->{filter_type} = $pf_row->{business_type} . '_' . $pf_row->{feed_type};
        }
        $pf_row->{filter_type} //= 'performance';
    }

    push @{$self->items}, @{Direct::Model::PerformanceFilter->from_db_hash_multi($perf_filter_rows)};

    return $self;
}

=head2 items_by($key)

Возвращает структуру с перфоманс фильтрами, вида:
    $key //eq 'id' => {$perf_filter1->id => $perf_filter1, $perf_filter2->id => $perf_filter2, ...};
    $key eq 'gid'  => {$adgroup_id1 => [$perf_filter1, $perf_filter2, ...], adgroup_id2 => [$perf_filter3, ...], ...};

=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';
    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};
    croak "by `id`/`adgroup_id` only supported" unless $key =~ /^(?:id|adgroup_id)$/;

    my %result;
    if ($key eq 'id') {
        $result{$_->id} = $_ for @{$self->items};
    } else {
        push @{$result{ $_->adgroup_id }}, $_ for @{$self->items};
    }

    return \%result;
}

=head2 prepare_create

Подготовка списка перфоманс фильтров к созданию.

=cut

sub prepare_create {
    my ($self) = @_;

    for my $perf_filter (@{$self->items}) {
        $perf_filter->last_change('now');
        $perf_filter->status_bs_synced('No');

        # Если фильтр добавляется в группу не-черновик, то сбросим статус синхронизации на группе
        # Также, сбросим статус синхронизации всех баннеров группы
        if ($perf_filter->adgroup->status_moderate ne 'New') {
            $perf_filter->do_bs_sync_banners(1);
            $perf_filter->do_bs_sync_adgroup(1);

            if ( !($perf_filter->has_is_suspended && $perf_filter->is_suspended)) {
                $perf_filter->do_set_adgroup_bl_status('Processing');
            }
            $perf_filter->do_freeze_autobudget_alert(1);
        }

        $perf_filter->do_update_adgroup_last_change(1);
    }

    return $self;
}

=head2 create

Создание списка перфоманс фильтров в БД.

=cut

sub create {
    my ($self) = @_;

    # Выдача идентификаторов
    my $ids = get_new_id_multi(perf_filter_id => scalar(@{$self->items}));
    $_->id(shift @$ids) for @{$self->items};

    $self->prepare_create();
    $self->prepare_logging('create');
    Direct::Model::PerformanceFilter::Manager->new(items => $self->items)->create();
    $self->do_logging();

    return;
}

=head2 _numerize_categoryId_rule_values

Сделать значения чисел в правиле по полю categoryId настоящими числами

=cut

sub _numerize_categoryId_rule_values {
    my ($self) = @_;
    for my $perf_filter (@{$self->items}) {
        for my $rule (@{$perf_filter->condition}) {
            if (($rule->{field} eq 'categoryId') && (ref $rule->{value} eq 'ARRAY')) {
                for my $item (@{$rule->{value}}) {
                    $item += 0;
                }
            }
        }
    }
}

=head2 prepare_update

Подготовка списка перфоманс фильтров к обновлению.

=cut

sub prepare_update {
    my ($self) = @_;

    $self->_numerize_categoryId_rule_values();

    for my $perf_filter (@{$self->items}) {
        # При изменении условия фильтра - больше не помечаем текущий как удалёный и не создаем новый,
        # а только обновляем нужные статусы
        if ($perf_filter->is_condition_changed) {
            if ($perf_filter->adgroup->status_moderate ne 'New') {
                $perf_filter->do_set_adgroup_bl_status('Processing');
            }
        }

        # Время последнего обновления фильтра
        if (any { $perf_filter->$_ } map { "is_${_}_changed" } qw/
            filter_name price_cpc price_cpa autobudget_priority target_funnel is_suspended is_deleted condition
        /) {
            $perf_filter->last_change('now');
            }

        # Статусы синхронизации группы/баннеров (сбрасываем при условии, что группа не черновик)
        if (
            (any { $perf_filter->$_ } map { "is_${_}_changed" } qw/target_funnel is_suspended is_deleted condition/) &&
            $perf_filter->adgroup->status_moderate ne 'New'
        ) {
            $perf_filter->do_bs_sync_banners(1);
            $perf_filter->do_bs_sync_adgroup(1);
        }

        # Время последнего изменения группы
        if (any { $perf_filter->$_ } map { "is_${_}_changed" } qw/filter_name target_funnel is_suspended is_deleted condition/) {
            $perf_filter->do_update_adgroup_last_change(1);
        }

        # При изменении цен/приоритета_автобюджета фильтры отправляются отдельным транспортом
        if (any { $perf_filter->$_ } map { "is_${_}_changed" } qw/price_cpc price_cpa autobudget_priority condition/) {
            $perf_filter->status_bs_synced('No');
        }
    }

    return $self;
}

=head2 update(%options)

Обновление списка перфоманс фильтров в БД.

Параметры:
    %options:
        log_price__type -> тип логирования для цен

=cut

sub update {
    my ($self, %options) = @_;

    $self->prepare_update();
    $self->prepare_logging('update', %{hash_cut \%options, qw/log_price__type/});
    Direct::Model::PerformanceFilter::Manager->new(items => $self->items)->update();
    $self->do_logging();

    return;
}

=head2 prepare_delete

Подготовка списка перфоманс фильтров к удалению.

=cut

sub prepare_delete {
    my ($self) = @_;

    for (@{$self->items}) {
        $_->reset_state(); # На всякий случай
        $_->is_deleted(1);
    }

    $self->prepare_update();

    return $self;
}

=head2 delete

Удаление списка перфоманс фильтров (помечаем как удаленные в БД).

=cut

sub delete {
    my ($self) = @_;

    $self->prepare_delete();
    $self->prepare_logging('delete');
    Direct::Model::PerformanceFilter::Manager->new(items => $self->items)->update();
    $self->do_logging();

    return;
}

=head2 prepare_logging($action, %params)
=head2 do_logging

Методы для логирования событий (действий).

Параметры:
    $action -> выполненное действие: create/update
    %params:
        log_price__type -> тип логирования для цен

=cut

sub prepare_logging {
    my ($self, $action, %params) = @_;

    my $adgroup_id2campaign = get_hashes_hash_sql(PPC(pid => [map { $_->adgroup_id } @{$self->items}]), [
        'SELECT g.pid, g.cid, c.currency FROM phrases g JOIN campaigns c ON (g.cid = c.cid)',
        where => {'g.pid' => SHARD_IDS},
    ]);

    for my $perf_filter (@{$self->items}) {
        if (
            $action eq 'create' ||
            $action eq 'copy' ||
            ($action eq 'update' && ($perf_filter->is_price_cpc_changed || $perf_filter->is_price_cpa_changed))
        ) {
            # Внимание! Поддерживаются действия: create/update/copy
            my $log_price_type = $params{log_price__type} // "perf_filter_${action}";

            push @{$self->data->{log_price}}, {
                cid => $adgroup_id2campaign->{$perf_filter->adgroup_id}->{cid},
                pid => $perf_filter->adgroup_id,
                id => $perf_filter->id,
                type => $log_price_type,
                price => $perf_filter->has_price_cpc ? $perf_filter->price_cpc : 0,
                price_ctx => $perf_filter->has_price_cpa ? $perf_filter->price_cpa : 0,
                currency => $adgroup_id2campaign->{$perf_filter->adgroup_id}->{currency},
            };
        }
    }

    return;
}

sub do_logging {
    my ($self) = @_;

    LogTools::log_price($self->data->{log_price}) if $self->data->{log_price};

    return;
}

=head2 copy_extra

Копирование фильтров перфоманс группы.

Параметры:
    см. Direct::Role::Copyable

Результат:
    $src_perf_filter_id2dst_perf_filter_id - отображение исходного perf_filter_id в новый perf_filter_id
    $src_pid2src_perf_filter_ids - {pid => [perf_filter_id1, perf_filter_id2 ...]} отображение исходной группы в исходные id фильтра

=cut

sub copy_extra {
    my ($self, $from_client_id, $source, $to_client_id, $destination, $src_id2dst_id, %options) = @_;
    my %src_pid2src_perf_filter_ids;
    for my $filter (@{$source->items}) {
        push @{$src_pid2src_perf_filter_ids{$filter->adgroup_id}}, $filter->id; 
    }
    return ($src_id2dst_id, \%src_pid2src_perf_filter_ids, $destination);
}

1;
