package Direct::PredefineVars;

use strict;
use warnings;
use utf8;

use Carp;
use List::MoreUtils qw/uniq none/;

use Yandex::Validate;

use Agency;
use Client;
use DoCmd::Base;
use PrimitivesIds qw/get_clientid/;
use Rbac;
use RBACDirect;
use User qw/get_user_options/;

use feature 'state';
use Property;

=head2 @COMMON_VARS

Список переменных, которые рассчитываем для всех контроллеров

=cut

our @COMMON_VARS = qw/
    campaign_agency_contacts features_enabled_for_client
/;


=head2

    Определяет значения переменных, заданных в атрибуте :PredefineVars
    Возвращяет хеш с заданными ключами и вычесленными значениями

=cut

sub make_vars {
    my ($cmd, $c, $FORM) = @_;

    my $vars = {};
    my @var_names = uniq (@COMMON_VARS, @{$DoCmd::Base::predefine_vars{$cmd} || []});
    return $vars if !@var_names;

    for my $name (@var_names) {
        eval {
            no strict 'refs';
            $vars->{$name} = &{"_$name"}($c, $FORM);
            1;
        } or do {
            carp "Unknown or invalid variable '$name' wants to be predefined: $@";
        };
    }

    return $vars;
}

=head2

    Internal use only!
    Возвращяет контакты агентства для отображения в шаблоне

=cut

sub _campaign_agency_contacts {
    my ($c, $FORM) = @_;

    my $campaign_agency_contacts;
    if (
        is_valid_id($FORM->{cid}) && rbac_is_agencycampaign($c->rbac, $FORM->{cid}) &&
        ($campaign_agency_contacts = get_agency_contacts_by_cid($c->rbac, $FORM->{cid}))
    ) {
        # Если кампанию ведёт главный представитель, то fio не выводим
        delete $campaign_agency_contacts->{fio} if $campaign_agency_contacts->{is_chief};
    }

    return $campaign_agency_contacts;
}

sub _show_account_score {
    my $c = shift;
    # DIRECT-33517 - Открыть показатель качества аккаунта на всех
    # Однако, турецкому интерфейсу пока остается закрытым. Проверка на интерфейс в DoCmd.pm (локальная)
    return 1;
}

sub _visible_futures {

    my $c = shift;
    return get_user_options($c->login_rights->{role} =~ /^super(reader)?$/ ? $c->UID : $c->uid)->{visible_futures} || [];
}

sub _enable_cpm_banner_campaigns { # флаг указывает, доступны ли клиенту кампании охватного продукта
    my $c = shift;

    return 1; # DIRECT-76686: Открыть охватный продукт на всех
}

sub _enable_cpm_deals_campaigns { # флаг указывает, доступны ли клиенту кампании частных сделок
    my $c = shift;
    return (
            (($c->login_rights->{role} eq 'super') || Client::ClientFeatures::has_cpm_deals_allowed_feature($c->login_rights->{ClientID}))
            && $c->client_client_id && (get_client_currencies($c->client_client_id)->{work_currency} ne 'YND_FIXED')
        ) ? 1 : 0; # пока доступны только суперам и клиентам с включенной фичей. Недоступны в фишках
}


sub _enable_cpm_yndx_frontpage_campaigns { # флаг указывает, доступны ли клиенту кампании охватного продукта на главной
    my $c = shift;
    return Client::ClientFeatures::has_cpm_yndx_frontpage_allowed_feature($c->client_client_id);
}

sub get_collecting_verified_phones_disabled_for_countries {
    state $collecting_verified_phones_disabled_for_countries_property //= Property->new('collecting_verified_phones_disabled_for_countries');
    my $collecting_verified_phones_disabled_for_countries_raw = $collecting_verified_phones_disabled_for_countries_property->get(300) // "";
    my @collecting_verified_phones_disabled_for_countries = split /,/, $collecting_verified_phones_disabled_for_countries_raw;

    return \@collecting_verified_phones_disabled_for_countries;
}

sub _enable_collecting_verified_phones { # флаг указывает, включен ли сбор подтвержденных телефонов
    my ($r, $c) = @_;

    my @collecting_verified_phones_disabled_for_countries = get_collecting_verified_phones_disabled_for_countries();
    my $country_by_location = HttpTools::http_country_from_geobase($r);

    return (none { $_ == $c->{country_region_id} || $_ == $country_by_location } @collecting_verified_phones_disabled_for_countries) &&
        $c->login_rights->{client_role} eq 'client' && !$c->login_rights->{client_have_agency} &&
        Client::ClientFeatures::collecting_verified_phones_feature($c->client_client_id)
        && !Client::ClientFeatures::facelift_disabled_for_dna_feature($c->client_client_id);
}

sub _show_new_dynamic_edit { # флаг показыват кнопку добавление/редактирование в новом интерфейсе
    my $c = shift;
    return Client::ClientFeatures::dynamic_only_for_dna_feature($c->client_client_id)
        && Client::ClientFeatures::dynamic_groups_edit_allowed_for_dna_feature($c->client_client_id)
        && Client::ClientFeatures::dynamic_ads_edit_allowed_for_dna_feature($c->client_client_id);
}

sub _show_new_mobile_content_edit { # флаг показыват кнопку добавление/редактирование в новом интерфейсе
    my $c = shift;
    return Client::ClientFeatures::mobile_content_only_for_dna_feature($c->client_client_id)
        && Client::ClientFeatures::mobile_content_groups_edit_allowed_for_dna_feature($c->client_client_id)
        && Client::ClientFeatures::mobile_content_ads_edit_allowed_for_dna_feature($c->client_client_id);
}

sub _show_new_cpm_yndx_frontpage_edit { # флаг показыват кнопку добавление/редактирование в новом интерфейсе
    my $c = shift;
    return Client::ClientFeatures::cpm_yndx_frontpage_in_dna_feature($c->client_client_id);
}

sub _redirect_cpm_yndx_frontpage_campaign_edit { # флаг редиректа в новый интерфейс со страниц параметров кампании
    my $c = shift;
    return Client::ClientFeatures::cpm_yndx_frontpage_in_dna_feature($c->client_client_id);
}

sub _enable_cpm_price_campaigns { # флаг указывает, доступны ли клиенту кампании прайсового охватного продукта на главной
    my $c = shift;
    return Client::ClientFeatures::has_cpm_price_allowed_feature($c->client_client_id);
}

sub _is_feature_cpm_yndx_frontpage_profile_enabled {
    my $c = shift;
    return Client::ClientFeatures::has_cpm_yndx_frontpage_profile_allowed_feature($c->client_client_id);
}

sub _is_new_campaign_strategy_enabled {
    my $c = shift;
    return Client::ClientFeatures::has_new_campaign_strategy_enabled_feature($c->client_client_id);
}

sub _enable_internal_campaigns { # флаг указывает, доступны ли клиенту кампании внутренней рекламы
    my $c = shift;
    my $perminfo = Rbac::get_perminfo( ClientID => $c->client_client_id );
    return Rbac::has_perm( $perminfo, 'internal_ad_product' ) ? 1 : 0;
}

sub _enable_content_promotion_video_campaigns { # флаг указывает, доступны ли клиенту кампании типа "Продвижение контента"
    my $c = shift;
    return Client::ClientFeatures::has_content_promotion_video_allowed_feature($c->client_client_id);
}

sub _enable_content_promotion_collections { # флаг указывает, доступно ли клиенту продвижение коллекций
    my $c = shift;
    return Client::ClientFeatures::is_feature_content_promotion_collection_enabled($c->client_client_id);
}

sub _enable_content_promotion_service_campaigns {
    my $c = shift;
    return Client::ClientFeatures::has_content_promotion_services_allowed_feature($c->client_client_id);
}

sub _enable_any_content_promotion { # флаг указывает, доступно ли продвижение какого-либо из типов
    my $c = shift;
    return Client::ClientFeatures::has_any_content_promotion_feature_enabled($c->client_client_id);
}

sub _is_featureTurboLandingEnabled {
    my $c = shift;
    return Client::ClientFeatures::get_is_featureTurboLandingEnabled(client_id => $c->client_client_id, rights => $c->login_rights);
}

sub _is_feature_brand_lift_enabled {
    my $c = shift;
    return Client::ClientFeatures::get_is_feature_brand_lift_enabled(client_id => $c->client_client_id, rights => $c->login_rights);
}

sub _is_cpm_video_disabled {
    my $c = shift;
    return Client::ClientFeatures::is_cpm_video_disabled($c->client_client_id);
}

sub _is_feature_autooverdraft_enabled {
    my $c = shift;
    return Client::ClientFeatures::get_is_feature_autooverdraft_enabled(client_id => $c->client_client_id, rights => $c->login_rights);
}

sub _is_feature_cpc_video_banner_enabled {
    my $c = shift;
    return Client::ClientFeatures::get_is_feature_cpc_video_banner_enabled(client_id => $c->client_client_id, region_id => $c->country_region_id, domain => $c->site_host);
}

sub _is_feature_increase_ad_text_limits_enabled {
    my $c = shift;
    return Client::ClientFeatures::has_increase_ad_text_limits_allowed_feature($c->client_client_id);
}

sub _is_feature_virtual_campaigns_enabled {
    my $c = shift;
    return Client::ClientFeatures::has_virtual_campaigns_feature($c->client_client_id);
}

sub _features_enabled_for_operator {
    my $c = shift;
    my $operator_client_id = get_clientid(uid => $c->UID);

    return {map {$_ => 1} @{Client::ClientFeatures::allowed_features($operator_client_id)}};
}

sub _features_enabled_for_client {
    my $c = shift;

    return {map {$_ => 1} @{Client::ClientFeatures::allowed_features($c->client_client_id)}};
}

sub _save_text_group_in_java {
    my ($c) = @_;
    return Client::ClientFeatures::has_save_text_group_in_java_allowed_feature($c->client_client_id);
}

sub _enable_recommendations {
    my $c = shift;
    return Client::ClientFeatures::has_grid_allowed_feature($c->login_rights->{ClientID})
        && ($c->login_rights->{role} =~ /^(super(reader)?|manager|placer)$/ ||
            Client::ClientFeatures::has_recommendation_availability_feature($c->client_client_id));
}

sub _is_feature_smart_at_search_enabled {
    my $c = shift;
    return Client::ClientFeatures::has_smart_at_search_enabled_feature($c->client_client_id);
}

sub _is_feature_cpm_yndx_frontpage_frequency_cap_enabled {
    my $c = shift;
    return Client::ClientFeatures::has_cpm_yndx_frontpage_frequency_cap_feature($c->client_client_id);
}

sub _is_feature_cpm_video_several_pixels_enabled {
    my $c = shift;
    return Client::ClientFeatures::has_cpm_video_several_pixels_enabled_feature($c->client_client_id);
}

sub _has_turbo_smarts {
    my $c = shift;
    return Client::ClientFeatures::has_turbo_smarts($c->client_client_id);
}

sub _is_feature_uniq_completed_viewers_stat_enabled {
    my $c = shift;
    return Client::ClientFeatures::has_uniq_completed_viewers_stat_enabled_feature($c->client_client_id);
}

1;
