use strict;
use warnings;

package Direct::ReShard;

=head1 NAME

    Direct::ReShard - перенос всех данных клиента в другой шард

=head1 DESCRIPTION

=cut

use Carp qw/croak/;
use List::MoreUtils qw/uniq/;
use Time::HiRes;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils;

use Settings;

use Direct::ReShard::Rules qw/%RULES_IDS %IGNORED_TABLES @RULES/;
use Direct::ReShard::Table;

use Mouse;

our $LOG_NAME ||= 'reshard';
our $DATA_LOG_NAME ||= 'reshard.data';

# логер
has logger => (
    is => 'rw', 
    default => sub {
        Yandex::Log->new(date_suf => "%Y%m%d", log_file_name => $LOG_NAME);
    },
    lazy => 1,
    );

# логер данных
has data_logger => (
    is => 'rw', 
    default => sub {
        Yandex::Log->new(date_suf => "%Y%m%d", log_file_name => $DATA_LOG_NAME, use_syslog => 0);
    },
    lazy => 1,
    );

# readonly режим - данные выбираются, но не инсёртятся и не удаляются
# может быть использовано для прогрева базы
has simulate => (
    is => 'rw', 
    isa => 'Bool',
    default => 0,
    );

# выполнять ли всё в одной транзакции, или делать транзакцию на каждую таблицу
has transaction_per_table => (
    is => 'rw', 
    isa => 'Bool',
    default => 0,
    );

# нужно ли валидировать db_schema
has validate_db_schema => (
    is => 'rw', 
    isa => 'Bool',
    default => 1,
    );

# нужно ли сравнивать столбцы таблицы в шардах
has validate_db_cols => (
    is => 'rw',
    isa => 'Bool',
    default => 1,
    );

# шард, из которого получаем схему данных
# если не определён - получаем схему данных из db_schema
has orig_shard => (
    is => 'rw',
    isa => 'Maybe[Int]',
    default => undef,
    );

# нужно ли в лог писать строки с данными
has log_data => (
    is => 'rw', 
    isa => 'Bool',
    default => 1,
    );


__PACKAGE__->meta->make_immutable();


=head2 my $resharder = Direct::ReShard->create(simulate => 1, ...)

    создание/получение объекта для выполнения переноса клиентов между шардами

=cut
sub create {
    shift;
    my (%O) = @_;

    my $self = __PACKAGE__->new(%O);

    # таблицы, которые нужны для получения ключей
    my %ids_tables = (ClientID => []);
    for my $id_name (keys %RULES_IDS) {
        $ids_tables{$id_name} = [$RULES_IDS{$id_name} =~ /\s+(?:FROM|JOIN)\s+(\w+)/g];
    }

    my %seen_tables;
    my %postponed_deletion_tables;

    for my $rule (@RULES) {

        my @froms = ref($rule->{from}) eq 'ARRAY' ? @{$rule->{from}} : $rule->{from} || ();
        my @joins = uniq map {/(?:^\s*|join\s+|,\s*)(\w+)/gi;} @froms;

        for my $table_name (@{$rule->{tables}}) {
            if (my @cleared_tables = grep {exists $seen_tables{$_} && !exists $postponed_deletion_tables{$_}} @joins) {
                croak "Can't process rule for $table_name: tables from joins already processed: ".join(', ', @cleared_tables);
            }

            my $tbl = Direct::ReShard::Table->new_from_rule($table_name, $rule, $self->orig_shard);

            # проверка существования таблиц, из которых получаем ключи
            if (my @cleared_tables = grep {exists $seen_tables{$_} && !exists $postponed_deletion_tables{$_}} @{$ids_tables{$tbl->key_type}}) {
                croak "Can't process rule for $table_name: tables for key ".$tbl->key_type." already processed: ".join(', ', @cleared_tables);
            }

            if ($rule->{dont_delete} && $rule->{only_delete}) {
                croak "Can't process rule for $table_name: dont_delete and only_delete exclude each other";
            }
            if ($rule->{dont_delete}) {
                $postponed_deletion_tables{$table_name} = undef;
            } else {
                # данные могут быть уже удалены, но запрещать удалять пустые данные большого смысла нет
                if ($rule->{only_delete} && !exists $seen_tables{$table_name}) {
                    croak "Can't process rule for $table_name: for rule with only_delete data must be copied in an earlier rule";
                }
                delete $postponed_deletion_tables{$table_name};
            }
            $seen_tables{$table_name} = undef;
            push @{$self->{_tables}}, $tbl;
        }
    }
    if (%postponed_deletion_tables) {
        croak "Invalid rules: data in tables ".join(", ", sort keys %postponed_deletion_tables)." need to be deleted in the end";
    }
    return $self;
}


=pod2 my @table_objs = $r->tables_list()

    Список объектов Direct::ReShard::Table в порядке выполнения переноса

=cut
sub tables_list {
    my $self = shift;
    return @{$self->{_tables}};
}


=pod2 my @rrors = $r->validate($shard1, $shard2)

    Проверка возможности выполнения переноса между двумя шардами
    + проверка того, что в ::Rules описаны все таблицы из db_schema

=cut
sub validate {
    my ($self, $shard_old, $shard_new) = @_;

    my @errors;
    for my $table ($self->tables_list) {
        my @cols_schema = $table->schema_cols();
        my @cols_old = $table->shard_cols($shard_old);
        my @cols_new = $table->shard_cols($shard_new);
        if ($self->validate_db_schema && "@cols_schema" ne "@cols_old") {
            push @errors, "Incorrect columns set in table $table->{name}:\nold shard $shard_old: @cols_old\ndb_schema: @cols_schema";
        }
        if ($self->validate_db_cols && "@cols_old" ne "@cols_new") {
            push @errors, "Incorrect columns set in table $table->{name}:\nold shard $shard_old: @cols_old\nnew shard $shard_new: @cols_new";
        }
    }
        
    return @errors; 
}


=pod2 $r->move_client_data($ClientID, $old_shard, $new_shard)

    перенос данных клиента по всем таблицам

=cut
sub move_client_data {
    my ($self, $ClientID, $old_shard, $new_shard) = @_;
    
    return 0 if $old_shard == $new_shard;

    my $cur_shard = get_shard(ClientID => $ClientID);
    if (!defined $cur_shard || $cur_shard > 0 && $cur_shard != $old_shard) {
        croak "Incorrect shard info for CLientID=$ClientID - old_shard: $old_shard, cur_shard: $cur_shard";
    }

    if (my @errors = $self->validate($old_shard, $new_shard)) {
        croak "Can't reshard $ClientID($old_shard -> $new_shard):\n".join('', map {"  $_\n"} @errors);
    }

    my $t1 = Time::HiRes::time();
    local $self->logger->{msg_prefix} = "[$ClientID:$old_shard-$new_shard:$t1]";
    local $self->data_logger->{msg_prefix} = "[$ClientID:$old_shard-$new_shard:$t1]";

    no warnings 'redefine';
    local *din = $self->transaction_per_table ? sub {$_[0]->()} : *do_in_transaction;
    din(sub {
        $self->_apply_rules($ClientID, $old_shard, $new_shard);
        save_shard(ClientID => $ClientID, shard => $new_shard) if !$self->simulate;
    });
    $self->logger->out(sprintf "TOTAL: %.3f sec", Time::HiRes::time - $t1);

    return 1;
}


# получение словаря всех id, необходимых для переноса клиента
sub _get_client_keys_dict {
    my ($self, $ClientID, $old_shard) = @_;
    my %ids = (ClientID => [$ClientID]);
    for my $id (keys %RULES_IDS) {
        $ids{$id} = get_one_column_sql(PPC(shard => $old_shard), $RULES_IDS{$id}, $ClientID) || [];
    }
    return \%ids;
}


# применение правил переноса для всех таблиц
sub _apply_rules {
    my ($self, $ClientID, $old_shard, $new_shard) = @_;

    my $log = $self->logger;
    $log->out("PROCESS: start ClientID=$ClientID, old_shard=$old_shard, new_shard=$new_shard");

    local get_dbh(PPC(shard => $old_shard))->{mysql_use_result} = 1;

    # получаем id, необходимые для запросов
    my $keys_dict = $self->_get_client_keys_dict($ClientID, $old_shard);

    my $tables_num = scalar($self->tables_list);
    my $iter = enumerate_iter([$self->tables_list]);
    while(my ($i, $table) = $iter->()) {
        $log->out("TABLE: $table->{name} #$i/$tables_num");
        no warnings 'redefine';
        local *din = $self->transaction_per_table ? *do_in_transaction : sub {$_[0]->()};
        din(sub {
            do_sql(PPC(shard => [$old_shard, $new_shard]), 'SET SESSION FOREIGN_KEY_CHECKS = 0');
            my $del_stats = $table->copy_data($keys_dict, $old_shard, $new_shard, $self);
            $table->delete_copied_data($del_stats, $old_shard, $self) unless $table->dont_delete;
            do_sql(PPC(shard => [$old_shard, $new_shard]), 'SET SESSION FOREIGN_KEY_CHECKS = 1');
        });
    }

    $log->out("PROCESS: finish");
}

sub get_client_lock_data {
    my ($self, $ClientID, $shard) = @_;

    my $data = {};
    $data->{blocked_uids} = get_one_column_sql(PPC(shard => $shard), ["SELECT uid FROM users", WHERE => {ClientID => $ClientID, statusBlocked => "No"}]) || [];
    my $bes_data = get_hash_sql(PPC(shard => $shard), "
                                        SELECT c.cid, bes.par_type
                                          FROM users u
                                               JOIN campaigns c using (uid)
                                               LEFT JOIN bs_export_specials bes ON bes.cid = c.cid
                                         WHERE u.ClientID = ?
                                               AND (bes.cid is null OR bes.par_type != 'nosend')
                                        ", $ClientID) || [];
    $data->{no_send_cids} = [keys %$bes_data];
    $data->{par_type_for_cid} = { map { $_ => $bes_data->{$_} } grep { $bes_data->{$_} } keys %$bes_data };
    return $data;
}


sub lock_client {
    my ($self, $ClientID, $shard, $data) = @_;

    do_update_table(PPC(shard => $shard), "users", {statusBlocked => "Yes"}, where => {uid => $data->{blocked_uids}});
    do_mass_insert_sql(PPC(shard => $shard), 
                       "REPLACE INTO bs_export_specials (cid, par_type) VALUES %s",
                       [map {[$_, 'nosend']} @{$data->{no_send_cids}}]);
}


sub unlock_client {
    my ($self, $ClientID, $shard, $data) = @_;
    do_update_table(PPC(shard => $shard), "users", {statusBlocked => "No"}, where => {uid => $data->{blocked_uids}});
    my @cids_to_delete_from_bes;
    my %old_par_type_for_cid = %{ $data->{par_type_for_cid} };
    for my $cid (@{ $data->{no_send_cids} }) {
        if (!$old_par_type_for_cid{$cid}) {
            push @cids_to_delete_from_bes, $cid;
        }
    }
    if (@cids_to_delete_from_bes) {
        do_delete_from_table(PPC(shard => $shard), "bs_export_specials", where => {cid => \@cids_to_delete_from_bes});
    }
    if (%old_par_type_for_cid) {
        do_mass_insert_sql(PPC(shard => $shard),
                           "REPLACE INTO bs_export_specials (cid, par_type) VALUES %s",
                           [map {[$_, $old_par_type_for_cid{$_}]} keys %old_par_type_for_cid]);
    }
}


1;
