use strict;
use warnings;

package Direct::ReShard::Exporter;

=head1 NAME
    
    Direct::ReShard::Exporter - выгрузка всех данных клиента из базы в текстовые файлы с sql insert-ами,
    пользуясь механизмами решардинга

=head1 DESCRIPTION

=cut

use Path::Tiny qw/path/;
use Carp qw/croak/;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils;

use Settings;
use Direct::ReShard;

use Mouse; 

# объект решардера
has resharder => (
    is => 'ro',
    isa => 'Direct::ReShard',
    default => sub {
        return Direct::ReShard->create(
            simulate => 1,
            log_data => 0,
        );
    }
    );

has dir => (
    is => 'ro',
    isa => 'Str',
    required => 1,
    );

has sortlist => (
    is => 'rw',
    isa => 'ArrayRef',
    default => sub { [] }, 
    required => 0,
    );

__PACKAGE__->meta->make_immutable();


=head2 $e->export_client_data(ClientID)

    основная функция - выгразка данных клиента

=cut
sub export_client_data {
    my ($self, $ClientID) = @_;

    my $data_file = path($self->dir."/ppc.$ClientID.sql");
    my $data_tmp_file = path($data_file.".tmp");
    $self->_write_client_ppcdata_inserts($ClientID, $data_tmp_file->openw_raw);
    $data_tmp_file->move($data_file);
    
    my $dict_file = path($self->dir."/ppcdict.$ClientID.sql");
    my $dict_tmp_file = path($dict_file.".tmp");
    $self->_write_client_ppcdict_inserts($ClientID, $dict_tmp_file->openw_raw);
    $dict_tmp_file->move($dict_file);
}


=head2 _write_client_ppcdata_inserts(...)

    выгрузда данных из базы ppcdata

=cut
sub _write_client_ppcdata_inserts {
    my ($self, $ClientID, $fh) = @_;

    # получаем id, необходимые для запросов
    my $shard = get_shard(ClientID => $ClientID);
    my $keys_dict = $self->resharder->_get_client_keys_dict($ClientID, $shard);

    local get_dbh(PPC(shard => $shard))->{mysql_use_result} = 1;


    my @tables;
    if (scalar @{$self->sortlist} == 0) {
        @tables = $self->resharder->tables_list;
    } else { 
        @tables = $self->sortlist;
    }

    my %hash;
    
    for my $table ($self->resharder->tables_list) {
        $table->iterate_data(
            $keys_dict, $shard, $self->resharder,
            sub {
                my ($buffer) = @_;

                if (defined $table->autoinc_key_idx) {
                    splice @$_, $table->autoinc_key_idx, 1 for @$buffer;
                }
                my $inserts = Yandex::DBTools::sql_mass_inserts(
                    $table->insert_sql, $buffer, {
                        desired_sql_length => 100_000,
                    });
             
               $hash{$table->name} = $inserts
            }
            );
    }
    # при пустом sortlist последовательно записываем все insert'ы
    if (scalar @{$self->sortlist} == 0) {
        foreach my $name (values %hash) {
            for my $sql (@{$name}) {
                print $fh "$sql;\n" or die "Can't write inserts: $!";
            }
        }
     
    } else {
        
        for my $name (@{$self->sortlist}) {
            next if !exists $hash{$name};
            for my $sql (@{$hash{$name}}) {
                print $fh "$sql;\n" or die "Can't write inserts: $!";
            }
        }
    }
}


=head2 _write_client_ppcdict_inserts(...)

    выгрузда данных из базы ppcdata

=cut
sub _write_client_ppcdict_inserts {
    my ($self, $ClientID, $fh) = @_;

    # получаем id, необходимые для запросов
    my $shard = get_shard(ClientID => $ClientID);
    my $keys_dict = $self->resharder->_get_client_keys_dict($ClientID, $shard);

    local get_dbh(PPCDICT)->{mysql_use_result} = 1;

    for my $key (sort keys %Yandex::DBShards::SHARD_KEYS) {
        my $key_info = $Yandex::DBShards::SHARD_KEYS{$key};

        my $data_col = $key_info->{shard_key} || $key_info->{chain_key};

        next unless defined $data_col;

        my ($key_col) = grep {defined $_ && exists $keys_dict->{$_}} $key, $key_info->{chain_key};
        croak "No suitable key for $key" if !defined $key_col;
        
        for my $chunk (chunks $keys_dict->{$key_col}, 10_000) {
            my $sth = exec_sql(PPCDICT, ["SELECT $key, $data_col FROM $key_info->{table}", WHERE => {$key_col => $chunk}]);
            while(my $rows = $sth->fetchall_arrayref([], 100_000)) {

                my $inserts = Yandex::DBTools::sql_mass_inserts(
                    "INSERT INTO $key_info->{table} ($key, $data_col) VALUES %s", $rows, {
                        desired_sql_length => 100_000,
                    });
                
                for my $sql (@$inserts) {
                    print $fh "$sql;\n" or die "Can't write inserts: $!";
                }
            }
        }
    }
  
    #DIRECT-114719 
    my $h1 = { "chain_key" => "ClientID",
           "table" => "cpm_price_packages",
           "columns" => ["package_id", "title", "tracker_url", "price", "currency", "order_volume_min",
                "order_volume_max", "targetings_fixed", "targetings_custom", "status_approve", 
                "last_update_time", "date_start", "date_end", "is_public", "is_archived"],
           "ref_table" => "cpm_price_package_clients",
           "join_key" => "package_id",
           "ref_columns" => ["ClientID", "package_id", "is_allowed"],
    };
    for my $chunk (chunks $keys_dict->{$h1->{chain_key}}, 10_000) {
        my $c1 = "t1." . join ", t1.", @{$h1->{columns}};
        my $sth = exec_sql(PPCDICT, ["SELECT $c1 FROM $h1->{table} as t1 
		INNER JOIN $h1->{ref_table} as t2 ON t1.$h1->{join_key} = t2.$h1->{join_key}", 
		WHERE => {"t2.$h1->{chain_key}" => $chunk}]
	);
        my $c2 = join ", ", @{$h1->{columns}};
        while(my $rows = $sth->fetchall_arrayref([], 100_000)) {
             my $inserts = Yandex::DBTools::sql_mass_inserts(
               "INSERT INTO $h1->{table}($c2) VALUES %s", $rows, {
                        desired_sql_length => 100_000,
                });
            for my $sql (@$inserts) {
                print $fh "$sql;\n" or die "Can't write inserts: $!";
            }
        }

        my $c0 = "". join ", ", @{$h1->{ref_columns}};
        $sth = exec_sql(PPCDICT, ["SELECT $c0 FROM $h1->{ref_table}", WHERE => {"$h1->{chain_key}" => $chunk}]);
        while(my $rows = $sth->fetchall_arrayref([], 100_000)) {
            my $inserts = Yandex::DBTools::sql_mass_inserts(
                "INSERT INTO $h1->{ref_table}($c0) VALUES %s", $rows, {
                        desired_sql_length => 100_000,
                });
            for my $sql (@$inserts) {
                print $fh "$sql;\n" or die "Can't write inserts: $!";
            }
        }
    }
    #DIRECT-76292. deals_adfox
    my $h2 = { "chain_key" => "ClientID",
           "table" => "deals_adfox",
           "columns" => ["deal_id", "ClientID", "create_time_utc", "adfox_deal_status", "deal_type", "publisher_name",
           	"adfox_name", "adfox_description", "date_start_utc", "date_end_utc", "contacts", "targetings", "currency_iso_code",
		"expected_impressions_per_week", "expected_money_per_week", "cpm", "margin_ratio", "agency_revenue_ratio",
		"agency_revenue_type", "placements", "misc_attrs_json"],
    };
    for my $chunk (chunks $keys_dict->{$h2->{chain_key}}, 10_000) {
    	my $c4 = "". join ", ", @{$h2->{columns}};
	my $sth = exec_sql(PPCDICT, ["SELECT $c4 FROM $h2->{table}", WHERE => {"$h2->{chain_key}" => $chunk}]);
        while(my $rows = $sth->fetchall_arrayref([], 100_000)) {
            my $inserts = Yandex::DBTools::sql_mass_inserts(
                "INSERT INTO $h2->{table}($c4) VALUES %s", $rows, {
                        desired_sql_length => 100_000,
                });
            for my $sql (@$inserts) {
                print $fh "$sql;\n" or die "Can't write inserts: $!";
            }
        }
    }
}

1;
