use strict;
use warnings;

package Direct::ReShard::Rules;

=head1 NAME

Direct::ReShard::Rules - данные для переноса данных в другой шард

=head1 DESCRIPTION

=cut

use base qw/Exporter/;
our @EXPORT_OK = qw/%RULES_IDS %IGNORED_TABLES @RULES/;

=head2 %RULES_IDS

    Запросы, которыми можно по ClientID получить прочие id

=cut

our %RULES_IDS = (
    uid => "SELECT uid FROM users WHERE ClientID = ?",
    cid => "SELECT c.cid FROM campaigns c JOIN users u USING (uid) WHERE u.ClientID = ?",
    OrderID => "SELECT c.OrderID FROM campaigns c JOIN users u USING (uid) WHERE u.ClientID = ? AND c.OrderID > 0",
    strategy_id => "SELECT s.strategy_id FROM strategies s WHERE s.ClientID = ?",
    );

# m devtestsharded:ppc:1 --dump campaigns --no-data | m devtestsharded:ppc:2
# m devtestsharded:ppc:1 --dump geo_regions geo_translation | m devtestsharded:ppc:2

=head2 %IGNORED_TABLES

    Таблицы, которые по разным причинам исключаем из рассмотрения

=cut

our %IGNORED_TABLES = map {$_=>1} (
            'dbqueue_job_types', 'geo_regions', 'geo_translation', # словарные таблицы

            'checksum', 'checksum_result', # системные
            'yt_checksum',  # для хранения чексумм между mysql и yt

            'warnings', 'warnings_60_n', 'warnings_nn', # временные

            'currency_rates',

            # записи добавляются в таблицу вручную редко
            'ess_additional_objects',

            # решардинг - редкий процесс, не боимся потерять чуть-чуть пушей
            # это лучше, чем делать уникальные id при записи eventlog
            'push_notifications_process',

            # одинаковые во всех шардах
            'agency_nds',
            'bs_dead_domains',

            'aggr_statuses_resync_queue',

            # очередь почти всегда пустая, поэтому при решардинге не будем ничего делать
            'banners_to_fill_language_queue',

            'clients_to_clean_conversion_multipliers'
    );

=head2 @RULES

    Правила для потабличного переноса данных

=cut

our @RULES = (
    {
        # эти таблицы копируем в первую очередь, но удаляем позже, чтобы корректно работала вставка данных в репликаторе
        tables => [qw/campaigns phrases banners bids bids_retargeting/],
        key => 'cid',
        replace => 1,   # c INSERT решардинг будет падать, если прервётся посередине и начнёт обрабатывать правила в начале
        dont_delete => 1,
    },
    {
        tables => [qw/tag_group/],
        from => "phrases p join %s t on p.pid = t.pid",
        key => 'p.cid',
        delete_key => 'tag_id',
        replace => 1,   # c INSERT решардинг будет падать, если прервётся посередине и начнёт обрабатывать правила в начале
        dont_delete => 1,
    },
    {
        tables => [qw/
                agency_currencies agency_total_sums api_special_user_options
                client_currency_changes client_discounts client_firm_country_currency client_limits clients
                clients_autoban
                clients_to_fetch_nds clients_to_force_multicurrency_teaser
                currency_convert_money_correspondence
                clients_custom_options
                clients_options
                client_managers
                clients_api_options
                clients_stat
                client_reminder_letters
                client_dialogs
                currency_convert_queue
                dbqueue_jobs dbqueue_job_archive
                force_currency_convert
                infoblock_teasers_factors
                client_brands
                client_teaser_data_lastupdate
                experiments
                mod_mail_candidates
                yandex_agency_orders
                agency_options
                clients_features
                deals deal_notifications
                mobile_app_trackers
                recommendations_online recommendations_status
                recommendations_status_ess
                clients_avatars
                internal_ad_products
                organizations
                idm_groups_members
                client_measurers_settings
                turbo_apps_info
                client_phones
                brand_survey
		        xiva_pushes_queue
                client_additional_targetings
                client_erir_attributes
                /],
        key => 'ClientID',
    },
    {
        tables => [qw/
                idm_group_roles
                /],
        key => 'subject_ClientID',
        key_type => 'ClientID',
    },
    {
        tables => [qw/agency_offline_reports/],
        key => 'agency_client_id',
        key_type => 'ClientID'
    },
    {
        tables => [qw/
                clients_geo_ip
                client_nds
                account_score
                agency_lim_rep_clients
                /],
        replace => 1,
        key => 'ClientID',
    },
    {
        tables => [qw/
                manager_hierarchy
                /],
        replace => 1,
        key => 'manager_client_id',
        key_type => 'ClientID',
    },
    {
        tables => [qw/
                    api_user_domain_stat bm_reports
                    internal_users
                    mobile_teaser mod_sent_user_time
                    infoblock_state
                    org_details pdf_reports
                    stat_reports
                    offline_reports
                    /],
        key => 'uid',
    },
    {
        tables => [qw/mediaplan_accept_stats/],
        from => 'mediaplan_stats ms JOIN %s t using(mpid)',
        delete_key => 'mpid',
        key => 'ms.cid',
    },
    {
        tables => [qw/
                    aggr_statuses_campaigns
                    autobudget_alerts autobudget_cpa_alerts autobudget_forecast
                    bids_manual_prices bids_phraseid_associate bids_phraseid_history bids_href_params
                    camp_activization campaigns_multicurrency_sums camp_domains_count camp_metrika_counters camp_metrika_goals camp_operations_queue
                    subcampaigns camp_options camp_secondary_options camp_unsync_media cpm_limit camp_payments_info
                    camp_additional_data
                    camp_additional_targetings
                    camp_autobudget_restart
                    camp_turbolanding_metrika_counters
                    camp_order_types
                    camp_promocodes
                    campaign_promoactions
                    mediaplan_banners_original mediaplan_bids mediaplan_stats
                    metrika_counters
                    need_license_camps
                    pre_moderate
                    tag_campaign_list
                    user_campaigns_favorite
                    when_money_on_camp_was
                    vcards
                    daas_campaigns
                    warnplace warn_pay
                    campaigns_mobile_content
                    campaigns_performance
                    campaigns_cpm_yndx_frontpage
                    campaigns_experiments
                    added_phrases_cache
                    campaigns_deals
                    camps_for_servicing
                    camp_dialogs
                    campaigns_minus_words
                    campaigns_internal
                    campaigns_cpm_price
                    campaign_permalinks
                    campaign_phones
                    camp_calltracking_settings
                    camp_calltracking_phones
                    camp_measurers
                    camp_payment_data
                    campaign_moderation_versions
                    campaigns_promotions
                    widget_partner_campaigns
                    /],
        key => 'cid',
    },
    {
        tables => [qw/
            demography_multiplier_values
            geo_multiplier_values
            retargeting_multiplier_values
            ab_segment_multiplier_values
            banner_type_multiplier_values
            inventory_multiplier_values
            mobile_multiplier_values
            weather_multiplier_values
            expression_multiplier_values
            trafaret_position_multiplier_values
            tablet_multiplier_values
        /],
        from => 'hierarchical_multipliers h JOIN %s t using(hierarchical_multiplier_id)',
        delete_key => 'hierarchical_multiplier_id',
        key => 'h.cid',
    },
    {
        tables => [ 'hierarchical_multipliers' ],
        key_type => 'cid',
        key => 'cid',
    },
    {
        tables => [ 'campaigns_budget_reach_daily' ],
        key => 'cid',
    },
    {
        tables  => [ 'camp_operations_queue_copy', 'camp_day_budget_stop_history' ],
        key     => 'cid',
        autoinc => 1,
    },
    {
        tables  => [ 'events' ],
        key_type => 'uid',
        key     => 'objectuid',
        autoinc => 1,
    },
    {
        tables  => [ 'stat_rollback_data' ],
        key_type => 'uid',
        key     => 'uid',
        autoinc => 1,
    },
    {
        tables => [qw/
                    wallet_campaigns
                    autopay_settings
                    /],
        key => 'wallet_cid',
        key_type => 'cid',
    },
    {
        tables => [ 'wallet_payment_transactions' ],
        key => 'wallet_cid',
        key_type => 'cid',
        autoinc => 1,
    },
    {
        tables => [qw/aggr_statuses_adgroups/ ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        delete_key => 'pid',
        key => 'p.cid',
    },
    {
        tables => [qw/ adgroups_mobile_content group_params adgroups_minus_words adgroups_moderation_versions/ ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        delete_key => 'pid',
        key => 'p.cid',
    },
    {
        tables => [qw/bs_auction_stat/ ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        replace => 1,
        delete_key => 'pid',
        key => 'p.cid',
    },
    {
        tables => [qw/ adgroups_performance bids_performance / ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        delete_key => 'pid',
        key => 'p.cid',
    },
    {
        tables => [qw/ adgroups_text / ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        delete_key => 'pid',
        key => 'p.cid',
    },
    {
        tables => [qw/ adgroups_content_promotion adgroups_cpm_banner adgroups_cpm_video adgroup_project_params adgroups_hypergeo_retargetings/ ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        delete_key => 'pid',
        key => 'p.cid',
    },
    {
        tables => [qw/adgroups_internal/ ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        delete_key => 'pid',
        key => 'p.cid',
    },
    {
        tables => [qw/adgroup_page_targets/ ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        delete_key => 'pid',
        key => 'p.cid',
    },
    {
        tables => [qw/adgroup_bs_tags/ ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        delete_key => 'pid',
        key => 'p.cid',
    },
    {
        tables => [qw/video_segment_goals/ ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        delete_key => 'pid',
        key => 'p.cid',
    },
    {
        tables => [qw/adgroup_additional_targetings/ ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        delete_key => 'pid',
        key => 'p.cid',
    },
    {
        tables => [qw/adgroup_priority adgroup_promoactions/ ],
        from => "phrases p JOIN %s t on p.pid = t.pid",
        delete_key => 'pid',
        key => 'p.cid',
    },
    {   
        tables => [qw/aggr_statuses_keywords/],
        from => "bids bi JOIN %s t on bi.id = t.id",
        delete_key => 'id',
        key => 'bi.cid',
    },
    {   
        tables => [qw/aggr_statuses_retargetings/],
        from => "bids_retargeting br JOIN %s t on br.ret_id = t.ret_id",
        delete_key => 'ret_id',
        key => 'br.cid',
    },
    {
        tables => [qw/
            aggr_statuses_banners
            auto_moderate banner_images_process_queue_bid banners_mobile_content
            catalogia_banners_rubrics deleted_banners mediaplan_mod_reasons
            post_moderate pre_moderate_banners
            banners_additions
            banner_display_hrefs
            banners_minus_geo
            banner_pixels
            aggregator_domains
            banners_internal
            banner_prices
            banners_content_promotion_video
            banners_content_promotion
            banner_measurers
            banners_tns
            banner_turbo_galleries
            banner_turbo_apps
            banner_additional_hrefs
            banner_logos
            banner_buttons
            banner_phones
            banner_names
            banner_image_asset
            banner_display_href_texts
            banner_leadform_attributes
            banner_multicards
            banner_multicard_sets
            banner_user_flags_updates
            banner_publisher
        /],
        from => "banners b JOIN %s t on b.bid = t.bid",
        delete_key => 'bid',
        key => 'b.cid',
    },
    {
        tables => [qw/ sitelinks_links /],
        from => "sitelinks_sets s JOIN sitelinks_set_to_link l on l.sitelinks_set_id = s.sitelinks_set_id JOIN %s t on l.sl_id = t.sl_id",
        copy => 1,
        key => 's.ClientID',
        distinct => 1,
    },
    {
        tables => [qw/ sitelinks_set_to_link /],
        from => "sitelinks_sets s JOIN %s t on t.sitelinks_set_id = s.sitelinks_set_id",
        key => 's.ClientID',
        delete_key => 'sitelinks_set_id',
        distinct => 1,
    },
    {
        tables => [qw/ agency_client_relations /],
        key => 'client_client_id',
        key_type => 'ClientID',
    },
    {
        tables => [qw/ agency_client_prove  agency_managers /],
        key => 'agency_client_id',
        key_type => 'ClientID',
    },
    {
        tables => [qw/ bs_media_stat /],
        from => 'media_groups mg JOIN media_banners mb on mb.mgid = mg.mgid JOIN %s t ON mb.BannerID = t.BannerID and mb.BannerID > 0',
        key => 'mg.cid',
        delete_key => 'BannerID',
    },
    {
        tables => [qw/ retargeting_goals mediaplan_bids_retargeting /],
        from => 'retargeting_conditions c JOIN %s t on c.ret_cond_id = t.ret_cond_id',
        delete_key => 'ret_cond_id',
        key => 'c.ClientID',
    },
    {
        tables => [qw/filtered_feeds/],
        from => 'feeds f JOIN %s t using(feed_id)',
        delete_key => 'feed_id',
        key => 'f.ClientID',
    },
    {
        tables => [qw/ perf_feed_history perf_feed_vendors perf_feed_categories /],
        from => 'feeds f JOIN %s t on f.feed_id = t.feed_id',
        delete_key => 'id',
        autoinc => 1,
        key => 'f.ClientID',
    },
    {
        tables => [qw/ optimizing_campaign_stat optimizing_campaign_postpone /],
        from => 'optimizing_campaign_requests r JOIN %s t on r.request_id = t.request_id',
        delete_key => 'request_id',
        key => 'r.cid',
    },
    {
        tables => [qw/ mod_edit /],
        from => "banners b join %s t on b.bid = t.id and t.type = 'banner'",
        delete_key => 'id',
        key => 'b.cid',
    },
    {
        tables => [qw/ balance_info_queue /],
        from => [
            "users u JOIN %s t on u.uid = t.cid_or_uid and t.obj_type = 'uid'",
            "users u JOIN campaigns c on u.uid = c.uid JOIN %s t on c.cid = t.cid_or_uid and t.obj_type = 'cid'",
            ],
        key => 'u.ClientID',
        autoinc => 1,
        delete_key => 'id',
    },
    {
        tables => [qw/ redirect_check_queue /],
        from => "banners b JOIN %s t ON b.bid = t.object_id and t.object_type = 'banner'",
        delete_key => 'id',
        autoinc => 1,
        key => 'b.cid',
    },
    {
        tables => [qw/ mod_reasons /],
        from => [
            "banners o JOIN %s t on t.id = o.bid and t.type in ('banner', 'contactinfo', 'sitelinks_set', 'image', 'display_href')",
            "phrases o JOIN %s t on t.id = o.pid and t.type in ('phrases')",
            "media_groups o JOIN %s t on t.id = o.mgid and t.type in ('media_group')",
            "media_groups o JOIN media_banners b on b.mgid = o.mgid JOIN %s t on t.id = b.mbid and t.type in ('media_banner')",
            "banners_performance o JOIN %s t on t.id = o.banner_creative_id and t.type in ('canvas', 'video_addition')",
            "banners_performance o JOIN %s t on t.id = o.creative_id and t.type in ('perf_creative')",
            "campaigns o JOIN %s t on t.id = o.cid and t.type in ('campaign')",
            ],
        key => 'o.cid',
        autoinc => 1,
        delete_key => 'rid',
    },
    {
        tables => [qw/ mod_reasons /],
        from => [
            "images im JOIN %s t on t.id = im.image_id and t.type = 'image_ad'",
            ],
        key => 'im.cid',
        autoinc => 1,
        delete_key => 'rid',
    },
    {
        tables => [qw/ mod_reasons /],
        from => [
            "mobile_content o JOIN %s t on t.id = o.mobile_content_id and t.type = 'mobile_content'",
            ],
        key => 'o.ClientID',
        autoinc => 1,
        delete_key => 'rid',
    },
    {
        tables => [qw/ mod_reasons /],
        from => [
            "additions_item_callouts aic JOIN %s t on t.id = aic.additions_item_id and t.type = 'callout'",
            ],
        key => 'aic.ClientID',
        autoinc => 1,
        delete_key => 'rid',
    },
    {
        tables => [qw/ mod_object_version /],
        from => [
            "banners o JOIN %s t on t.obj_id = o.bid and t.obj_type in ('banner', 'contactinfo', 'sitelinks_set', 'image', 'display_href')",
            "phrases o JOIN %s t on t.obj_id = o.pid and t.obj_type in ('phrases')",
            ],
        key => 'o.cid',
        delete_key => [qw/obj_type obj_id/],
    },
    {
        tables => [qw/ mod_object_version /],
        from => [
            "images im JOIN %s t on t.obj_id = im.image_id and t.obj_type = 'image_ad'",
            ],
        key => 'im.cid',
        delete_key => [qw/obj_type obj_id/],
    },
    {
        tables => [qw/ mod_object_version /],
        from => [
            "mobile_content o JOIN %s t on t.obj_id = o.mobile_content_id and t.obj_type = 'mobile_content'",
            ],
        key => 'o.ClientID',
        delete_key => [qw/obj_type obj_id/],
    },
    {
        tables => [qw/ mod_resync_queue /],
        from => [
            "banners o JOIN %s t on t.object_id = o.bid and t.object_type in ('banner', 'contactinfo', 'sitelinks_set', 'image', 'display_href', 'image_ad')",
            "phrases o JOIN %s t on t.object_id = o.pid and t.object_type in ('phrases')",
            ],
        key => 'o.cid',
        autoinc => 1,
        delete_key => 'id',
    },
    {
        tables => [qw/ mod_resync_queue /],
        from => [
            "mobile_content o JOIN %s t on t.object_id = o.mobile_content_id and t.object_type = 'mobile_content'",
            ],
        key => 'o.ClientID',
        autoinc => 1,
        delete_key => 'id',
    },
    {
        tables => [qw/ media_banners mcb_phrases media_auto_moderate /],
        from => 'media_groups mg JOIN %s t on mg.mgid = t.mgid',
        key => 'mg.cid',
        delete_key => 'mgid',
    },
    {
        tables => [qw/ users_notifications_details /],
        from => "users_notifications n JOIN %s t on n.notification_id = t.notification_id",
        key => 'n.uid',
        delete_key => 'notification_id',
    },
    {
        tables => [qw/ manager_got_servicing /],
        key => 'manager_uid',
        key_type => 'uid',
        autoinc => 1,
    },
    {
        tables => [qw/ internal_ad_manager_place_access / ],
        from => "clients_relations cr JOIN %s t on cr.relation_id = t.relation_id",
        delete_key => 'relation_id',
        key_type => 'ClientID',
        key => 'cr.client_id_to',
    },
    # copy
    {
        tables => [qw/ domains /],
        from => [
            "banners pt JOIN %s t ON t.domain_id = pt.domain_id",
            "phrases pt JOIN adgroups_dynamic ad ON ad.pid = pt.pid JOIN %s t ON t.domain_id = ad.main_domain_id"
            ],
        key => 'pt.cid',
        copy => 1,
        distinct => 1,
    },
    {
        tables => [qw/ domains /],
        from => [
            "mobile_content pt JOIN %s t ON t.domain_id = pt.publisher_domain_id",
            "client_domains pt JOIN %s t ON t.domain_id = pt.domain_id",
            "client_domains_stripped pt JOIN %s t ON t.domain_id = pt.domain_id",
            ],
        key => 'pt.ClientID',
        copy => 1,
        distinct => 1,
    },
    {
        tables => [qw/ banner_images_formats /],
        distinct => 1,
        from => 'banner_images_pool p JOIN %s t on p.image_hash = t.image_hash',
        copy => 1,
        key => 'p.ClientID',
    },
    {
        tables => [qw/ filter_domain /],
        from => 'banners b JOIN %s t on b.domain = t.domain',
        copy => 1,
        key => 'b.cid',
        distinct => 1,
    },
    {
        tables => [qw/ domain_vcards /],
        from => 'banners b JOIN %s t on b.domain = t.domain',
        copy => 1,
        key => 'b.cid',
        distinct => 1,
    },

    {
        tables => [qw/ minus_words /],
        replace => 1,
        key => 'ClientID',
    },
    {
        tables => [qw/ maps /],
        copy => 1,
        from => [
            "addresses o JOIN %s t on t.mid = o.map_id",
            "addresses o JOIN %s t on t.mid = o.map_id_auto",
            ],
        key => 'o.ClientID',
    },
    {
        tables => [qw/banner_images banner_permalinks/],
        from => "banners b JOIN %s t on b.bid = t.bid",
        delete_key => 'bid',
        key => 'b.cid',
    },
    {
        tables => [qw/banner_moderation_versions/],
        from => "banners b JOIN %s t on b.bid = t.bid",
        delete_key => 'bid',
        key => 'b.cid',
    },
    {
        tables => [qw/banner_turbo_moderation_versions banner_displayhrefs_moderation_versions
            banner_sitelinks_moderation_versions banner_images_moderation_versions  banner_video_additions_moderation_versions
            banner_logos_moderation_versions banner_buttons_moderation_versions bannerstorage_creatives_moderation_versions
            banner_names_moderation_versions banner_creatives_moderation_versions banner_vcards_moderation_versions
            banner_multicard_sets_moderation_versions
            /],
        from => "banners b JOIN %s t on b.bid = t.bid",
        delete_key => 'bid',
        key => 'b.cid',
    },
    {
        tables => [qw/moderate_banner_pages/],
        from => "banners b JOIN %s t on b.bid = t.bid",
        delete_key => 'bid',
        key => 'b.cid',
    },
    {
        tables => [qw/banner_resources banner_turbolanding_params/],
        from => "banners b JOIN %s t on b.bid = t.bid",
        delete_key => 'bid',
        key => 'b.cid',
    },
    {
        tables => [qw/turbolanding_metrika_counters turbolanding_metrika_goals/],
        from => "turbolandings tbl JOIN %s t on tbl.tl_id = t.tl_id",
        delete_key => 'tl_id',
        key => 'tbl.ClientID',
    },
    {
        tables => [qw/
                    eventlog
                    camp_copy_reports_queue
                    api_reports_offline
                    recommendations_queue
                    /],
        autoinc => 1,
        key => 'ClientID',
    },
    {
        tables => [qw/
                    phrases_doubles_queue
                    users_notifications_register_queue
                    user_payment_transactions
                    /],
        autoinc => 1,
        key => 'uid',
    },
    {
        tables => [qw/
                    auto_price_camp_queue
                    sms_queue
                    moderation_cmd_queue
                    bs_resync_queue
                    banner_images_uploads
                    xls_history
                    /],
        autoinc => 1,
        key => 'cid',
    },
    {
        tables => [qw/
                    bids_dynamic
                    adgroups_dynamic
                    dynamic_conditions
                    /],
        from => [
            "phrases p JOIN %s t ON t.pid = p.pid AND p.adgroup_type = 'dynamic'",
            ],
        key => 'p.cid',
        delete_key => 'pid',
    },
    {
        tables => [qw/
                    subclient_convert_requests
                    /],
        key => 'AgencyID',
        key_type => 'ClientID',
    },
    {
        tables => [qw/creative_banner_storage_sync/],
        from => "perf_creatives p JOIN %s t on p.creative_id = t.creative_id",
        key => 'p.ClientID',
        delete_key => 'creative_id',
    },
    {
        tables => [qw/mds_custom_names/],
        from => 'mds_metadata mm join %s t on t.mds_id = mm.id',
        key => 'mm.ClientID',
        delete_key => 'mds_id',
        autoinc => 1,
    },
    {
        tables => [qw/stat_rollbacks/],
        key => 'OrderID',
        autoinc => 1,
    },

    {
        tables => [qw/clients_cashback_history/],
        from => "clients_cashback_programs ccp JOIN %s t on ccp.client_cashback_program_id = t.client_cashback_program_id",
        delete_key => 'client_cashback_program_id',
        key => 'ccp.ClientID',
    },

    # порядок выполнения важен
    {
        tables => [qw/
            banners_performance
            banners_performance_main
        /],
        from => "banners b JOIN %s t on b.bid = t.bid",
        delete_key => 'bid',
        key => 'b.cid',
    },
    {
        tables => [qw/tag_group/],
        from => "phrases p join %s t on p.pid = t.pid",
        key => 'p.cid',
        delete_key => 'tag_id',
        replace => 1,
    },
    {
        tables => [qw/banners phrases bids bids_retargeting/],
        key => 'cid',
        replace => 1,
    },
    {
        tables => [qw/
                    mediaplan_banners
                    bids_arc bids_base
                    media_groups
                    optimizing_campaign_requests
                    images
                    banner_turbolandings
                    /],
        key => 'cid',
    },
    {
        tables => [qw/mobile_app_goals_external_tracker/],
        from => 'mobile_apps ma JOIN %s t using(mobile_app_id)',
        delete_key => 'goal_id',
        key => 'ma.ClientID',
    },
    {
        tables => [qw/mobile_content_icon_moderation_versions/],
        from => "mobile_content mc join %s t on t.mobile_content_id = mc.mobile_content_id",
        replace => 1,
        delete_key => 'mobile_content_id',
        key => 'mc.ClientID',
    },
    {
        tables => [qw/
                    addresses
                    retargeting_conditions
                    sitelinks_sets
                    mobile_content
                    client_domains
                    client_domains_stripped
                    feeds
                    daas_briefs
                    perf_creatives
                    mds_metadata
                    banner_images_pool
                    mobile_apps
                    turbolandings
                    client_pixel_providers
                    content_promotion_video
                    content_promotion
                    promoactions
                    hypergeo_segments
                    client_overdraft_limit_changes
                    calltracking_settings
                    clients_cashback_programs
                    clients_cashback_details
                    /],
        key => 'ClientID',
    },
    {
        tables => [qw/
                    users_notifications
                    users_agency users_captcha
                    users_options users_api_options users_units api_users_units_consumption
                    users_surveys users_typed_options
                    users_messengers
                    /],
        key => 'uid',
    },
    {
        tables => [qw/bs_export_specials bs_export_candidates bs_export_queue
                      mod_export_candidates mod_export_queue
                      camp_aggregated_lastchange
                      /],
        replace => 1,
        key => 'cid',
    },
    {
        tables => [qw/
                    strategy_options
                    strategy_metrika_counters
                    strategy_autobudget_restart
                    strategy_inventori_data
                    /],
        key => 'strategy_id',
    },
    {
        tables => [qw/strategies/],
        key => 'strategy_id'
    },
    {
        tables => [qw/campaigns/],
        key => 'cid',
        only_delete => 1,   # кажется, что после replace в репликаторной campaigns затираются данные из camp_metrika_counters, camp_options и т. п., так что вообще эти данные не трогаем.
                            # TODO разобраться, что происходит при replace
    },
    {
        tables => [qw/users/],
        key => 'ClientID',
    },
    {
        tables => [qw/pre_moderate_additions moderate_additions/],
        from => "additions_item_callouts aic join %s t on t.additions_item_id = aic.additions_item_id",
        replace => 1,
        delete_key => 'additions_item_id',
        key => 'aic.ClientID',
    },
    {
        tables => [qw/callout_moderation_versions/],
        from => "additions_item_callouts aic join %s t on t.callout_id = aic.additions_item_id",
        replace => 1,
        delete_key => 'callout_id',
        key => 'aic.ClientID',
    },
    {
        tables => [qw/additions_item_callouts additions_item_disclaimers additions_item_experiments/],
        key => 'ClientID',
    },
    {
        tables => [qw/wallet_campaigns_lock/],
        key => 'ClientID',
    },
    {
        tables => [qw/clients_relations/],
        key => 'client_id_to',
        key_type => 'ClientID',
    },
    {
        tables => [qw/reverse_clients_relations/],
        key => 'client_id_from',
        key_type => 'ClientID',
    },
    {
        tables => [qw/client_mcc_requests/],
        key => 'client_id_from',
        key_type => 'ClientID',
    },

    {
        tables => [qw/freelancers freelancers_card freelancers_skills/],
        key => 'ClientID',
    },
    {
        tables => [qw/freelancers_projects/],
        key => 'customer_client_id',
        key_type => 'ClientID',
    }
);

1;
