use strict;
use warnings;
use feature 'state';

package Direct::ReShard::Table;

=head1 NAME
    
    Direct::ReShard::Table - перенос данных клиента для одной таблицы

=head1 DESCRIPTION

=cut

use JSON;
use Carp qw/croak/;
use List::MoreUtils qw/first_index/;
use Time::HiRes;

use Yandex::DBTools;
use Yandex::DBSchema;
use Yandex::ListUtils;
use Yandex::Retry;

use Settings;

use Mouse; 

our $SELECT_IDS_CHUNK_SIZE ||= 100;
our $INSERT_CHUNK_ROWS ||= 1_000;
our $DELETE_CHUNK_ROWS ||= 1_000;
our $SLEEP_COEF ||= 0.0;

# имя таблицы
has name => (is => 'ro', isa => 'Str', required => 1);

# ключ, по которому пойдёт выборка и его тип (ClientID/uid/cid)
has key => (is => 'ro', isa => 'Str', required => 1);
has key_type => (is => 'ro', isa => 'Str', required => 1);

# список select-запросов для выборки данных, вместо списка ключей - %s
has select_sqls => (is => 'ro', isa => 'ArrayRef[Str]', required => 1);

# запрос для вставки данных, вместо values - %s
has insert_sql => (is => 'ro', isa => 'Str', required => 1);

# индексы столбцов, значения из которых нужно сохранить для последующего удаления
has delete_keys_idx => (is => 'ro', isa => 'ArrayRef[Int]');

# индекс столбца, с автоинкрементным полем - его нужно пропускать при инсёрте
has autoinc_key_idx => (is => 'ro', isa => 'Int');

# список столбцов таблицы во всех шардах
has '_cols_by_shards' => (is => 'rw', isa => 'HashRef[ArrayRef[Str]]', default => sub {+{}});

# список столбцов таблицы в db_schema
has '_cols_schema' => (is => 'rw', isa => 'ArrayRef[Str]');

# не удалять таблицу (должна удалиться другим правилом позже)
has 'dont_delete' => (is => 'ro', isa => 'Bool');

# только удалить таблицу (должна скопироваться другим правилом ранее)
has 'only_delete' => (is => 'ro', isa => 'Bool');

__PACKAGE__->meta->make_immutable();


=head2 my $tbl = Direct::ReShard::Table->new_from_rule($table_name, $rule, $orig_shard)

    из мета-языка описания правила Direct::ReShard::Rules создаём объект, готовый для использования

=cut
sub new_from_rule {
    my (undef, $table_name, $rule, $orig_shard) = @_;

    my ($key_col) = $rule->{key} =~ /(\w+)$/;
    my $key_type = $rule->{key_type} || $key_col;

    my $create_table_sql = Yandex::DBSchema::get_create_table_sql(db=>'ppc', table => $table_name);
    # столбцы потом будут сверяться с реальной базой, поэтому не боимся корявой регулярки
    # вообще здесь полагаемся на то, что в dbschema определения таблиц в каноническом виде, 
    # т.к. на это есть отдельный тест db_schema/valid_schema.t
    my @cols;
    if (defined $orig_shard) {
        @cols = @{ get_one_column_sql(PPC(shard => $orig_shard), "SHOW COLUMNS FROM $table_name") };
    } else {
        @cols = $create_table_sql =~ /^\s*`(\w+)`/mg;
    }

    # todo check all indexes
    my $GROUP_BY = '';
    if ($rule->{distinct}) {
        if ($create_table_sql =~ /PRIMARY KEY \(([^\)]+)\)/) {
            my @fields = split /\s*,\s*/, $1;
            $GROUP_BY = "GROUP BY ".join(', ', map {"t.$_"} @fields)." ORDER BY NULL";
        } else {
            croak "Can't enforce distinct for $table_name";
        }
    }

    # столбцы, которые даже не надо выбирать
    # столбцы, которые не надо инсертить
    my ($column_skip_select, $column_skip_insert) = ({}, {});

    # ищем генерируемые колонки и отмечаем, что их надо пропустить
    my @generated_cols = ($create_table_sql =~ /^\s*`(\w+)`.*GENERATED ALWAYS AS/gm);
    for my $c ( @generated_cols ){
        $c = lc($c);
        $column_skip_select->{$c} = 1;
        $column_skip_insert->{$c} = 1;
    }

    my ($autoinc_key, $autoinc_key_idx);
    if ($rule->{autoinc}) {
        if ($create_table_sql =~ /^\s*`(\w+)`.*AUTO_INCREMENT/m) {
            $autoinc_key = $1;
            $autoinc_key_idx = first_index {lc($_) eq lc($autoinc_key)} @cols;
            # автоинкрементную колонку надо выбирать, но не надо инсертить
            $column_skip_insert->{lc($autoinc_key)} = 1;
        } else {
            croak "Can't find auto_increment column for $table_name";
        }
    }

    my @FROMS = 
        $rule->{from} && ref($rule->{from}) eq 'ARRAY' ? @{$rule->{from}} :
        $rule->{from} ? ($rule->{from}) :
        ("%s AS t");
    my @selects;
    for my $FROM (@FROMS) {
        push @selects, "SELECT STRAIGHT_JOIN "
            .sql_fields(map {"t.$_"} grep {!$column_skip_select->{lc($_)}} @cols)
            ." FROM ".sprintf($FROM, $table_name)
            ." WHERE $rule->{key} in (%s) "
            .$GROUP_BY;
    }

    my @insert_cols = grep {!$column_skip_insert->{lc($_)}} @cols;
    my $insert_sql = ($rule->{replace} ? "REPLACE" : "INSERT")
        ." ".($rule->{copy} ? 'IGNORE' : '')
        ." INTO $table_name "
        ."(".join(', ', map {sql_quote_identifier($_)} @insert_cols).")"
        ." VALUES %s";

    my (@delete_keys_idx);
    if (!$rule->{copy}) {
        my @delete_keys_name = map { lc($_) } xflatten($rule->{delete_key} || $key_col);
        for my $delete_key (@delete_keys_name) {
            my $delete_key_idx = first_index {lc($_) eq $delete_key} @cols;
            croak "Can't find delete key '$delete_key' in table $table_name" if !defined $delete_key_idx || $delete_key_idx < 0;
            push @delete_keys_idx, $delete_key_idx;
        }
    }

    return Direct::ReShard::Table->new(
        name => $table_name,
        key => $rule->{key},
        key_type => $key_type,
        _cols_schema => \@cols,

        select_sqls => \@selects,
        insert_sql => $insert_sql,

        @delete_keys_idx ? (
            delete_keys_idx => \@delete_keys_idx,
        ) : (),

        defined $autoinc_key_idx ? (
            autoinc_key_idx => $autoinc_key_idx,
        ) : (),

        $rule->{dont_delete} ? (dont_delete => $rule->{dont_delete}) : (),
        $rule->{only_delete} ? (only_delete => $rule->{only_delete}) : (),
        );
}


=head2 my @cols = $tbl->schema_cols()

    список столбцов таблицы из db_schema (по которым составлялись sql запросы)

=cut
sub schema_cols {
    my ($self) = @_;
    return @{$self->_cols_schema};
}


=head2 my @cols = $tbl->shard_cols($shard)

    кешированное получение списка столбцов таблицы в указанном шарде

=cut
sub shard_cols {
    my ($self, $shard) = @_;
    if (!exists $self->_cols_by_shards->{$shard}) {
        $self->_cols_by_shards->{$shard} = 
            get_one_column_sql(PPC(shard => $shard), "SHOW COLUMNS FROM $self->{name}") || [];
    }
    return @{$self->_cols_by_shards->{$shard}};
}


=head2 $tbl->iterate_data($keys_dict, $old_shard, $resharder, $cb)

    выборка и обработка данных (вызов колбэка $cb)

=cut
sub iterate_data {
    my $self = shift;
    my ($keys_dict, $old_shard, $resharder, $cb) = @_;
    my $log = $resharder->logger;

    my $t1 = Time::HiRes::time;

    my $selected_cnt = 0;
    my @buffer_cb;
    for my $select_sql_tmpl (@{$self->select_sqls}) {
        for my $ids_chunk (chunks($keys_dict->{$self->key_type}, $SELECT_IDS_CHUNK_SIZE)) {
            my $select_sql = sprintf $select_sql_tmpl, join(',', map {int($_)} @$ids_chunk);
            $log->out("SELECT: $select_sql");
            my $sth = exec_sql(PPC(shard => $old_shard), $select_sql);
            my @buffer;
            while(my $row = $sth->fetchrow_arrayref) {
                $selected_cnt++;
                push @buffer, [@$row];
                if (@buffer >= $INSERT_CHUNK_ROWS) {
		            push @buffer_cb, @buffer;
                    @buffer = ();
                }
            }
            if (@buffer) {
                push @buffer_cb, @buffer;
                @buffer = ();
            }
        }
    }
    if (@buffer_cb) {
        $cb->(\@buffer_cb);
    }

    $log->out(sprintf "STATS: %s - selected: %d, ela: %.3f", $self->{name}, $selected_cnt, Time::HiRes::time() - $t1);
}


=head2 my $del_stats = $tbl->copy_data($keys_dict, $old_shard, $new_shard, $resharder)

    копирование данных таблицы между шардами
    возвращаем статистику по скопированным данным для передачи в delete_copied_data

=cut
sub copy_data {
    my $self = shift;
    my ($keys_dict, $old_shard, $new_shard, $resharder) = @_;
    my $data_log = $resharder->data_logger;

    state $json;
    $json = JSON->new();

    my %delete_stats;
    $self->iterate_data(
        $keys_dict, $old_shard, $resharder,
        sub {
            my ($buffer) = @_;
            for my $row (@$buffer) {
                if (defined(my $delete_data = $self->_get_data_for_delete($row))) {
                    $delete_stats{$delete_data}++;
                }
                splice @$row, $self->autoinc_key_idx, 1 if defined $self->autoinc_key_idx;
            }
            $data_log->out(map {"ROW: $self->{name}: " . $json->encode($_)} @$buffer) if $resharder->log_data;
            relaxed times => $SLEEP_COEF, sub {
                do_mass_insert_sql(PPC(shard => $new_shard), $self->insert_sql, $buffer) if !$resharder->simulate && !$self->only_delete;
            };
        }
        );

    return \%delete_stats;
}


=head2 $tbl->delete_copied_data($del_stats, $old_shard, $resharder)

    удаление скопированных данных из исходного шарда

=cut
sub delete_copied_data {
    my $self = shift;
    my ($del_stats, $old_shard, $resharder) = @_;

    my $log = $resharder->logger;

    # делим id на чанки, чтобы за запрос удалялось не сильно больше $DELETE_CHUNK_ROWS
    my @del_chunks;
    my $cur = [];
    my $cur_sum = 0;
    for my $id (sort keys %$del_stats) {
        if (@$cur && $cur_sum + $del_stats->{$id} > $DELETE_CHUNK_ROWS) {
            push @del_chunks, $cur;
            $cur = [];
            $cur_sum = 0;
        }
        $cur_sum += $del_stats->{$id};
        push @$cur, $id;
    }
    if (@$cur) {
        push @del_chunks, $cur;
        $cur = [];
        $cur_sum = 0;
    }

    # собственно удаление
    for my $del_data_chunk (@del_chunks) {
        my $delete_sql;
        if (@{ $self->delete_keys_idx } > 1) {
            $delete_sql = sprintf('DELETE FROM %s WHERE (%s)',
                                  $self->name,
                                  join(' OR ', @$del_data_chunk),
                                  );
        } else {
            $delete_sql = sprintf('DELETE FROM %s WHERE %s',
                                  $self->name,
                                  sql_condition({$self->_cols_schema->[ $self->delete_keys_idx->[0] ] => $del_data_chunk}),
                                  );
        }

        $log->out("DELETE $self->{name}: $delete_sql");
        do_sql(PPC(shard => $old_shard), $delete_sql) if !$resharder->simulate;
    }
}

=head3 $tbl->_get_data_for_delete($row)

    Получить необходимые данные для последующего удаления из исходного шарда

=cut
sub _get_data_for_delete {
    my $self = shift;
    my ($row) = @_;

    if (!$self->delete_keys_idx) {
        return undef;
    } elsif (@{ $self->delete_keys_idx } > 1) {
        return sql_condition([_AND => [ map { $self->_cols_schema->[$_] => $row->[$_] } @{ $self->delete_keys_idx } ]])
    } else {
        return $row->[ $self->delete_keys_idx->[0] ];
    }
}

1;
