package Direct::ResponseHelper;

=head1 NAME

    Direct::ResponseHelper

=head1 DESCRIPTION

  	Все функции, связанные с возвратом на клиентскую часть каких-то данных.

=cut

use strict;
use warnings;
use feature "state";

use Settings;
use Yandex::HTTP qw/http_fetch/;
use Yandex::Trace;
use Yandex::TVM2;
use JSON qw/decode_json/;
use HashingTools;

use Client::ClientFeatures;

use base qw/Exporter/;
our @EXPORT = qw /
    detect_respond_format

    respond_to
    error_to

    error
    error_js
    error_json
    error_jsonp
    error_plain
    error_tmpl
    error_bem
    message
    respond_bem
    respond_data
    respond_accel_redirect
    respond_http_error
    respond_json
    respond_jsonp
    respond_json_or_jsonp
    respond_stream
    respond_template
    respond_template_or_bem
    respond_text
    set_response_csp_header
    set_response_header

    redirect
    redirect_to_https
    redirect_to_check_sauth
/;

use Carp;

use JSON;
use Data::Dumper;
use Encode;
use Apache2::Const qw/ :common /;
use URI::Escape qw/uri_escape_utf8/;
use LWP::UserAgent;
use HTTP::Request;
use XML::LibXML qw//;
use TextTools qw//;
use Plack::Util;
use YAML::XS;

use Yandex::I18n;
use Yandex::Validate qw/is_valid_int/;
use Yandex::Trace;

use Direct::HttpResponse;
use Direct::StaticFilesHash qw/get_static_file_hashsums/;
use Direct::Template;
use DoCmd::CheckBySchema;
use EnvTools;
use List::Util qw/pairmap/;
use List::MoreUtils qw/any none/;
use LogTools;
use HttpTools;
use TextTools qw//;

# путь до файла с содержимым заголовков Content-Security-Policy
our %CSP_HEADERS_FILE;

# список ключей для подстановки (заполняется в Settings)
our %CSP_HEADER_PLACEHOLDER;


=head2 $AJAX_CAPTCHA_STATUS

    http-статус, выдаваемый для показа капчи для ajax-запросов

=cut
our $AJAX_CAPTCHA_STATUS = 429;


=head2 $AJAX_CAPTCHA_STATUS_MESSAGE

    http-статус-сообщение, выдаваемое для показа капчи для ajax-запросов

=cut
our $AJAX_CAPTCHA_STATUS_MESSAGE = "Too Many Requests";

my $template;

# Соответствие MIME типа и формата
my %RESPOND_MIME = (
    'text/html'        => 'html',
    'application/json' => 'json',
    'text/plain'       => 'text',
);

# Обработчики ответа для соответствующего формата
my %RESPOND_HANDLERS = (
    bem   => \&respond_bem,
    html  => \&respond_template_or_bem,
    json  => \&respond_json,
    jsonp => \&respond_jsonp,
    text  => \&respond_text,
);

# Обработчики сообщения об ошибке для соответствующего формата
my %RESPOND_ERROR_HANDLERS = (
    html  => sub { my ($r, $error, $hash, $extra) = @_; return error($error, $hash, $extra); },
    json  => sub { my ($r, $vars) = @_; return error_json($r, $vars); },
    text  => sub { my ($r, $error) = @_; return error_plain($error); },

    # Auto formats
    _json_auto => sub { my ($r, $error) = @_; return error_json($r, {error => $error}); },
);

=head2 detect_respond_format

    Определяет формат, в котором нужно отдать ответ пользователю

    Формат определяется исходя из:
        1) Наличия X-Requested-With заголовка
        2) Accept заголовка
        3) GET/POST параметра `format`

    Если X-Requested-With равен XMLHttpRequest, то формат определяется как `json`
    Для Accept заголовка формат определяется в соответствии с %RESPOND_MIME хешем
    Для остальных случаев учитывается явно переданный через GET/POST параметр `format`
    Если формат не удалось определить или он не задан - возвращяется html

=cut

sub detect_respond_format {
    my $r = shift;

    my $format;
    if (http_get_header($r, 'X-Requested-With') eq 'XMLHttpRequest') {
        $format = 'json';
    } elsif (my $accept = http_get_header($r, 'Accept')) {
        for (map { /(\w+\/[^; ]+)/ } (split ',', $accept)) {
            if (exists $RESPOND_MIME{$_}) {
                $format = $RESPOND_MIME{$_};
                last;
            }
        }
    } else {
        if ($r->can('param')) {
            $format = $r->param('format');
        } else {
            $format = {parse_form()}->{format};
        }
    }

    return $format || 'html';
}

=head2 respond_to

    Возвращяет ответ в формате, предпочитаемым пользователем.

    Предпочитаемый формат определяется через функцию detect_respond_format().
    Далее на основе доступных обработчиков (в %RESPOND_HANDLERS) и определенных разработчиком поддерживаемых форматов -
    определяется наилучший доступный.

    Используется следующий приоритет форматов: [user defined] => any

    Входные параметры:
        $r      => Apache или Plack request
        %params => список поддерживаемых форматов ответа, вида:
            format1 => ARRAYREF | CODEREF,
            format2 => ARRAYREF | CODEREF,
            ...

    Формат `any` должен быть задан только в виде CODEREF.
    Остальные форматы должны быть заданы в виде ARRAYREF или CODEREF.

    Если формат задан в виде ARRAYREF, он будет передан в существующий обработчик.
    Если формат задан в виде CODEREF, результат работы будет передан в существующий обработчик.

    Если обработчик не определен, но формат под него задан, то он (формат) должен быть CODEREF и содержать внутри себя respond_ функцию.

    Пример использования в контроллере:

        return respond_to($r,
            json => [$struct_to_serialize],
            html => [$template, $name, $vars],
            any => sub {
                my $format = shift;
                return respond_template_or_bem(...);
            },
        );

=cut

sub respond_to {
    my ($r, %params) = @_;

    croak "No \$request defined" unless $r;
    croak "No any respond_to() format specified" unless %params;

    my $format = detect_respond_format($r);

    if (exists $params{$format}) {
        if (exists $RESPOND_HANDLERS{$format}) {
            return $RESPOND_HANDLERS{$format}->($r, $params{$format}->()) if ref($params{$format}) eq 'CODE';
            return $RESPOND_HANDLERS{$format}->($r, @{$params{$format}}) if ref($params{$format}) eq 'ARRAY';
            croak "Invalid respond_to() arguments specified: format '$format' should be CODEREF or ARRAYREF";
        } else {
            return $params{$format}->() if ref($params{$format}) eq 'CODE';
            croak "Invalid respond_to() arguments specified: format '$format' should be CODEREF";
        }
    } elsif (exists $params{any}) {
        return $params{any}->($format) if ref($params{any}) eq 'CODE';
        croak "Invalid respond_to() arguments specified: format 'any' should be CODEREF";
    }

    die Direct::HttpResponse->new(body => 'Bad Request', status => 400);
}

=head2 error_to

    Возвращяет сообщение об ошибке в формате, предпочитаемым пользователем.

    Предпочитаемый формат определяется через функцию detect_respond_format().
    Далее на основе доступных обработчиков (в %RESPOND_ERROR_HANDLERS) и определенных разработчиком поддерживаемых форматов -
    определяется наилучший доступный.

    Используется следующий приоритет форматов: [user defined] => auto => any

    Входные параметры (1):
        $r      => Apache или Plack request
        $text или %params =>
            $text => текстовое сообщение об ошибке, соответствует записи:
                auto => $text

            %params => список поддерживаемых форматов ответа, вида:
                format1 => ARRAYREF | CODEREF,
                format2 => ARRAYREF | CODEREF,
                ...

    Формат `auto` должен быть задан только в виде SCALAR(TEXT).
    Формат `any` должен быть задан только в виде CODEREF.
    Остальные форматы должны быть заданы в виде ARRAYREF или CODEREF.

    Если формат задан в виде ARRAYREF, он будет передан в существующий обработчик.
    Если формат задан в виде CODEREF, результат работы будет передан в существующий обработчик.

    Формат `any` - самостоятельный и должен внутри себя возвращять программый код для возврата ошибки.
    Формат `auto` - еще более самостоятельный, автоматически определяется в каком виде и как выдать пользователю сообщение об ошибке.

    Пример использования в контроллере:

        return error_to($r,
            json => [$struct_to_serialize],
            html => [$error_msg, $hash],
            any => sub {
                my $format = shift;
                return error_tmpl($r, $template, $file, $vars, $error);
            },
        );

        return error_to($r, $error_msg); # Аналогично записи: return error_to($r, auto => $error_msg);

=cut

sub error_to {
    my $r = shift;
    my %params = @_ == 1 ? (auto => $_[0]) : @_;

    croak "No \$request defined" unless $r;
    croak "No any error_to() format specified" unless %params;

    my $format = detect_respond_format($r);

    if (exists $params{$format}) {
        if (exists $RESPOND_ERROR_HANDLERS{$format}) {
            return $RESPOND_ERROR_HANDLERS{$format}->($r, $params{$format}->()) if ref($params{$format}) eq 'CODE';
            return $RESPOND_ERROR_HANDLERS{$format}->($r, @{$params{$format}}) if ref($params{$format}) eq 'ARRAY';
            croak "Invalid error_to() arguments specified: format '$format' should be CODEREF or ARRAYREF";
        } else {
            return $params{$format}->() if ref($params{$format}) eq 'CODE';
            croak "Invalid error_to() arguments specified: format '$format' should be CODEREF";
        }
    } elsif (exists $params{auto}) {
        # croak "Invalid error_to() arguments specified: format 'auto' should be SCALAR(TEXT)" if ref($params{auto});
        # _${format}_auto => $format => html
        for ("_${format}_auto", $format, "html") {
            return $RESPOND_ERROR_HANDLERS{$_}->($r, $params{auto}) if exists $RESPOND_ERROR_HANDLERS{$_};
        }
    } elsif (exists $params{any}) {
        return $params{any}->($format) if ref($params{any}) eq 'CODE';
        croak "Invalid error_to() arguments specified: format 'any' should be CODEREF";
    }

    die Direct::HttpResponse->new(body => 'Bad Request', status => 400);
}

=head2 respond_data

    Отсылает данные с указанным content_type.

    Параметры позиционные:
        $r -- Apache2::RequestUtil->request;
        $data -- данные для отсылки
        $content_type - часть заголовка content-type или имя предопределенного формата
            ":xls" - Microsoft Excel
            ":xlsx" - Microsoft Excel 2007+
        $filename -- имя файла (не обязательно)

=cut

{
my %content_types = (
    ':xls' => 'application/vnd.ms-excel',
    ':xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
);
sub respond_data($$$;$) {
    my ($r, $data, $content_type, $filename) = @_;

    $content_type = $content_types{$content_type} if exists $content_types{$content_type};
    if ($r && $r->isa('Plack::Request')) {
        # IE don't allow to download a file over https without cache
        $r->env->{no_cache} = 0;
        return [ 200,
            [
             ($filename ? ('Content-Disposition' => qq/attachment; filename="$filename"/) : ()),
             'Content-Type' => $content_type,
             ],
            [$data]
            ];
    } else {
        $r->headers_out->add('Content-Disposition' => qq/attachment; filename="$filename"/) if $filename;
        $r->content_type($content_type);
        # IE don't allow to download a file over https without cache
        $r->no_cache(0) if http_server_scheme($r) eq 'https';
        $r->write($data);
        return OK;
    }
}
}

=head2 respond_accel_redirect

    Отвечает nginx'у так, чтобы тот сделал внутренний редирект на $url в параметрах

    Параметры позиционные:
        $r   -- Apache или Plack request
        $url -- здесь хранится файл
=cut
sub respond_accel_redirect($$) {
    my ($r, $url) = @_;

    if ($r && $r->isa('Plack::Request')) {
        # IE don't allow to download a file over https without cache
        $r->env->{no_cache} = 0;

        return [
            200,                          # status
            ['X-Accel-Redirect' => $url],   # headers
            ['']                            # data
        ];
    } else {
        # IE don't allow to download a file over https without cache
        $r->headers_out->add('X-Accel-Redirect' => $url);
        $r->no_cache(0) if http_server_scheme($r) eq 'https';
        return OK;
    }

}

=head2 respond_text

    Отсылает текст с указанным content_type, если не указан -- 'text/plain'.

    Параметры позиционные:
        $r -- Apache2::RequestUtil->request;
        $text -- данные для отсылки
        $content_type

=cut
sub respond_text($$;$$) {
    my ($r, $text, $content_type, $status_code) = @_;

    $content_type ||= 'text/plain';

    if ($r && $r->isa('Plack::Request')) {
        my $resp = [
            $status_code || 200,
            ['Content-Type' => $content_type],
            [Encode::is_utf8($text) ? Encode::encode_utf8($text) : $text],
        ];
        # проставляем в ответ заголовки, которые накапливали ранее
        for my $header ( @{$r->env->{"direct.headers_out"}||[]} ){
            Plack::Util::header_set($resp->[1], @$header);
        }
        return $resp;
    } else {
        $r->content_type($content_type);
        $r->status($status_code) if $status_code;
        print $text;
        return OK;
    }
}

=head2 respond_bem

    Передает данные в bem-шаблонизатор и отсылает получившийся html

    respond_bem($r, $reqid, $vars, source => "data3")

=cut
sub respond_bem($$$;%) {
    my ($r, $reqid, $vars, %opt) = @_;

    my $isFastShowDna = $vars->{is_fast_show_dna};
    unless ($isFastShowDna) {
        $vars->{features_enabled_for_operator_all} //= Client::ClientFeatures::get_operator_features();
        $vars->{features_enabled_for_client_all} //= Client::ClientFeatures::get_client_features();
    }

    $vars = check_output_data($r, $vars);
    my $html = Direct::Template::process_bem($reqid, $vars, %opt);
    return respond_text($r, $html, 'text/html');
}

=head2 respond_template

    Обрабатывает шаблон и отсылает получившийся html

    Параметры позиционные:
        $r -- Apache2::RequestUtil->request;
        $template -- TT-объект
        $name -- имя шаблона
        $vars -- переменные для шаблона

=cut
{
    my %skip_bem_header = map {($_ => 1)} qw{
    };

sub respond_template($$$;$) {
    my ($r, $template, $name, $vars) = @_;

    $vars->{features_enabled_for_operator_all} //= Client::ClientFeatures::get_operator_features();
    $vars->{features_enabled_for_client_all} //= Client::ClientFeatures::get_client_features();

    $vars = check_output_data($r, $vars);
    my $t = Yandex::Trace::new_profile('response:respond_template');

    # DIRECT-55725: для tt2 переопределяем csp-заголовки
    my $reqid = Yandex::Trace::trace_id();
    my $cmd = $vars->{cmd} || Direct::Template::dynamic_predefine($reqid)->{cmd} || '-';
    set_response_csp_header($r, $reqid, $cmd, Direct::Template::get_csp_nonce($reqid), for => 'data');

    # предварительно генерируем bem-шапку
    if (!$skip_bem_header{$name}) {
        $vars->{header_html} = Direct::Template::process_bem($reqid, $vars,
            source => 'data3', bundle => 'head',
            safe_json => 1,
        );
    }

    my $ret;
    $vars->{static_file_hashsums} = get_static_file_hashsums(source => 'data3');
    $template->process($name, $vars, \$ret) || die $template->error();

    return respond_text($r, $ret, 'text/html');
}
}

sub respond_template_or_bem {
    my ($r, $reqid, $vars, $template, $name) = @_;

    unless (is_beta()) {
        return respond_template($r, $template, $name, $vars);
    }

    my $cookie_val = get_cookie( $r, 'bemhtml' );
    my $form_val = $r->param('bemhtml');

    if ( $cookie_val || $form_val ) {
        return respond_bem($r, $reqid, $vars);
    }

    return respond_template($r, $template, $name, $vars);
}

=head2 respond_json

    Отсылает указанные данные в виде json (в ответ на ajax-запрос, например).
    TODO перейти на encode_json(???)

    Параметры позиционные:
        $r -- Apache2::RequestUtil->request;
        $data -- данные для отсылки, НЕ СЕРИАЛИЗОВАННЫЕ. Сериализация в json делается внутри функции.

=cut

sub respond_json($$) {
    my ($r, $data) = @_;

    $data = check_output_data($r, $data);
    my $response = ref $data ? to_json($data, is_beta() ? {pretty => 1} : ()) : $data;

    return respond_text($r, $response, 'application/json; charset=utf-8');
}

=head2 respond_jsonp

Функция отсылает данные в формате JSONP
Параметры
    $r - Apache2::RequestUtil->request
    $data - данные для отправки
    $callback - имя функции
Опционально:
    dont_send => 1 - не отсылать ответ, вместо этого вернуть его из функции

=cut

sub respond_jsonp($$$;%) {
    my ($r, $data, $callback, %opt) = @_;
    my $response = ref $data ? to_json($data) : $data;

    my $reqid = Yandex::Trace::trace_id();
    my $nonce = Direct::Template::get_csp_nonce($reqid);
    $nonce = TextTools::js_quote($nonce);

    $callback = TextTools::js_quote($callback);
    $callback = "'$callback'";
    my $html = qq{<!DOCTYPE html>
<html><head><title></title></head>
<body><script nonce="${nonce}">
parent[$callback] && parent[$callback](
    $response
);
</script></body></html>
};

    return $html if $opt{dont_send};
    return respond_text($r, $html, 'text/html');
}

=head2 respond_json_or_jsonp

Вызывает respond_json или respond_jsonp, в зависимости от того, задан ли $callback.

    respond_json_or_jsonp($r, $data, $callback, %opts);

=cut
sub respond_json_or_jsonp {
    my ($r, $data, $callback, %opt) = @_;
    if ($callback) {
        return respond_jsonp($r, $data, $callback, %opt);
    } else {
        return respond_json($r, $data);
    }
}

=head2 respond_http_error

    Отсылает ошибку с указанным HTTP-статусом

    Принимает позиционные параметры:
        $r -- Apache2::RequestUtil->request;
        $status_code -- код ошибки (404, например)

    respond_http_error($r, 404);

=cut

sub respond_http_error {
    my ($r, $status_code) = @_;

    die "invalid status_code: $status_code" unless is_valid_int($status_code, 100);

    if ($r && $r->isa('Plack::Request')) {
        return [$status_code, [], ['']];
    } else {
        $r->status($status_code);
        return OK;
    }
}

=head2 error

error - print error page
        No html tags allowed in $error
=cut

sub error {
    my ($error, $hash, $extra) = @_;

    my $vars = {
        features_enabled_for_operator_all => Client::ClientFeatures::get_operator_features(),
        features_enabled_for_client_all => Client::ClientFeatures::get_client_features(),
        cmd => 'error',
        error  => $error,
        hash => $hash,
        watermark => 'direct_error_message_page',
    };

    print STDERR "$extra\n" if defined $extra;

    die Direct::HttpResponse->new(body_sub => sub {
        my $reqid = shift;
        return Direct::Template::process_bem($reqid, $vars, source=>'data3');
    });
}



=head2 error_json

    выдаем ошибку для ajax-запросов, в json
    error_json($r, {error => 'some error'});

=cut

sub error_json($$) {
    my ($r, $vars) = @_;

    die Direct::HttpResponse->new(body => to_json($vars), type => 'application/json');
}

=head2 message

    Вывести сообщение, допускается html

=cut
sub message
{
    my ($msg, $hash) = @_;

    my $vars = {
        features_enabled_for_operator_all => Client::ClientFeatures::get_operator_features(),
        features_enabled_for_client_all => Client::ClientFeatures::get_client_features(),
        cmd => 'error',
        msg  => $msg,
        hash => $hash,
    };

    die Direct::HttpResponse->new(body_sub => sub {
        my $reqid = shift;
        return Direct::Template::process_bem($reqid, $vars, source => 'data3');
    });
}

=head2 error_plain

    error_plain - print error as plain text

=cut

sub error_plain {
    my $error = shift;
    die Direct::HttpResponse->new(body => $error);
}

=head2 error_tmpl

    error_tmpl - print templated error page

=cut

sub error_tmpl($$$$$) {
    my ( $r, $template, $file, $vars, $error ) = @_;

    $vars->{error} = $error;

    if ($vars->{current_url} && $vars->{current_url} =~ /cmd=ajax/ && $vars->{error}) {
        die Direct::HttpResponse->new(body => $vars->{error}, type => 'text/plain');
    } else {
        return respond_template($r, $template, $file, $vars);
    }
}

=head2 error_bem($type, $error, $params)

    Вывести страницу ошибки (BEM версия)

    Параметры:
        $type - тип ошибки, передается в BEM шаблонизатор как параметр cmd
        $error - текст ошибки
        $params - hashref, произвольные параметры для BEM шаблона

=cut

sub error_bem {
    my ($type, $error, $params) = @_;

    my $vars = {
        features_enabled_for_operator_all => Client::ClientFeatures::get_operator_features(),
        features_enabled_for_client_all => Client::ClientFeatures::get_client_features(),
        cmd => $type,
        error => $error,
        %$params,
    };

    die Direct::HttpResponse->new(body_sub => sub {
        my $reqid = shift;
        return Direct::Template::process_bem($reqid, $vars, source=>'data3');
    });
}

=head2 error_js

    Вернуть ошибку из cmd_ajax* метода

=cut

sub error_js {
    my ($error) = @_;
    die Direct::HttpResponse->new(body => to_json( { error => $error } ), type => 'application/x-javascript; charset=utf-8');
}

=head2 error_jsonp

    Отослать ошибку в формате JSONP
    Параметры
        $error - текст ошибки
        $callback - имя функции

=cut

sub error_jsonp($$) {
    my ($error, $callback) = @_;
    die Direct::HttpResponse->new(
        body => respond_jsonp(undef, { error => $error }, $callback, dont_send => 1),
        type => 'text/html'
        );
}


=head2 set_response_csp_header($r, $reqid, $cmd, $csp_nonce, %O)

    Установка заголовка Content-Security-Policy

    Опции:
        for

=cut
sub set_response_csp_header($$$$) {
    my ($r, $reqid, $cmd, $csp_nonce, %O) = @_;

    state $headers_cache;
    my $set = $O{for} || 'data3';
    my $cache = $headers_cache->{$set} ||= {};

    # 20 minutes
    if (!$cache->{last_headers_read} || time - $cache->{last_headers_read} > 20 * 60) {
        $cache->{headers} = YAML::XS::LoadFile($CSP_HEADERS_FILE{$set});
        croak "empty CSP headers $CSP_HEADERS_FILE{$set}" unless $cache->{headers};
        $cache->{last_headers_read} = time;
    }

    my $yandex_domain = HttpTools::yandex_domain($r);
    my $cookies = HttpTools::get_all_cookies($r);
    my %vars = (
        cmd => $cmd,
        reqid => $reqid,
        csp_nonce => $csp_nonce,
        yandex_domain => $yandex_domain,
        yandex_login => $cookies->{yandex_login} // '',
        yandexuid => $cookies->{yandexuid},
        %CSP_HEADER_PLACEHOLDER,
    );

    if (is_beta() && $cookies->{'local-dna'}) {
        return;
    }

    for my $head (@{$cache->{headers}}) {
        next if is_production() && $reqid % 100 > $head->{enable_on};
        my @directives;
        for my $directive (sort keys %{ $head->{directives} }) {
            push @directives, sprintf "%s %s", $directive, join ' ', map {
                TextTools::process_text_template($_, %vars)
            } @{ $head->{directives}->{$directive} };
        }

        my $header_name = TextTools::process_text_template($head->{head}, %vars);

        push @directives, "report-uri ".TextTools::process_text_template($head->{'report-uri'}, %vars);

        if ($r->isa('Plack::Request')) {
            # Plack: сохраняем заголовки, потом в respond_text добави их в ответ
            # если ответ может отдаваться не только respond_text-ом -- надо поддерживать обработку direct.headers_out и в других местах
            # если захочется проставлять заголовки не только из set_response_csp_header:
            #  сделать функцию типа set_headers($r, [$name => $value, ...]), и внутри нее смотреть, что такое $r
            # см. также set_response_header
            push @{$r->env->{"direct.headers_out"}}, [$header_name => join '; ', @directives];
        } elsif ($r->isa('Apache2::RequestRec')) {
            # чистый apache: проставляем заголовки через apache-евый метод
            $r->headers_out->set($header_name => join '; ', @directives);
        } else {
            die "Only mod_perl or Plack are supported for now";
        }
    }
}


=head2 set_response_header

    Установка заголовка ответа
    Параметры позиционные:
    $r -- объект-реквест (plack/apache)
    $header_name
    $header_value

    если при обработке одного запроса вызывать несколько раз с одним и тем же заголовком -- действует последнее значение

=cut
sub set_response_header
{
    my ($r, $header_name, $header_value) = @_;
    if ($r->isa('Plack::Request')) {
        # Plack: сохраняем заголовки, потом в respond_text добави их в ответ
        # если ответ может отдаваться не только respond_text-ом -- надо поддерживать обработку direct.headers_out и в других местах
        push @{$r->env->{"direct.headers_out"}}, [$header_name => $header_value];
    } elsif ($r->isa('Apache2::RequestRec')) {
        # чистый apache: проставляем заголовки через apache-евый метод
        $r->headers_out->set($header_name => $header_value);
    } else {
        die "Only mod_perl or Plack are supported for now";
    }
}

=head2 respond_stream

Генерация ответа в режиме стриминга - не держим данные в памяти, сразу отправляем.

Callback-функция может быть вызвана в любой момент, как внутри вызова respond_stream(), так и в любом другом
месте по выбору http middleware, т.е. к этому моменту может так случится, что будет размотан стек и освобождены
всё ещё нужные ресурсы.

TODO Не работает с Plack::Request. По идее в этом случае надо по другому заполнить исходящие заголовки, и
вернуть $stream_sub как результат. Вопрос только в том, что именно $stream_sub получить в качестве аргументов,
и не надо ли будет это ещё в какую-нибудь обёртку завернуть.

    my $response_for_http_middleware = respond_stream(
        $r, $content_type, $filename,
        sub {
            my ( $writer ) = @_;   # $writer - ссылка на функцию, отправляющую данные ответа
            $writer->("Field name\n");
            my $sth = prepare_some_heavy_database_request();
            while (my $row = $sth->fetchrow_hashref) {
                $writer->($row->{field} . "\n");
            }
        }
    );

=cut
sub respond_stream {
    my ($r, $content_type, $filename, $stream_sub) = @_;

    die "You could add RFC5987 support here if you want" unless $filename =~ /^[-_.0-9a-z]+$/i;

    if ($r->isa('Plack::Request')) {
        return sub {
            my ($respond) = @_;
            my $writer = $respond->([
                200,
                [
                    'Content-Type', $content_type,
                    'X-Accel-Buffering', 'no',
                    'Content-disposition', "attachment;filename=$filename",
                ],
            ]);
            $stream_sub->(sub { $writer->write(@_) });
            $writer->close;
        };
    } elsif ($r->isa('Apache2::RequestRec')) {
        $r->content_type($content_type);
        $r->headers_out->set('X-Accel-Buffering', 'no');
        $r->headers_out->set('Content-disposition', "attachment;filename=$filename");
        $stream_sub->(sub { $r->print(@_) } );
        return;
    } else {
        die "Only mod_perl or Plack are supported for now";
    }
}

=head2 redirect

    Редирект на URL

    redirect($r, $href, $opt, $anchor);

    Параметры:
        $r -- объект Apache2::RequestRec
        $href -- начало URL, куда редиректим (вместе с протоколом)
        $opt -- ссылка на хеш с параметрами, которые будут добавлены к $href; необязательный параметр
                специально рассматривается ключ uid_url этого хеша, который добавляется в конец полученной ссылки
        $anchor -- якорь, который будет добавлен к ссылке (#$anchor); необязательный параметр

    redirect($r, $SCRIPT, {cmd => 'showCamp', cid => $cid, uid_url => $FORM{uid_url}}, $bid);

=cut

sub redirect {
    my ( $r, $href, $opt, $anchor ) = @_;

    # во избежание http response splitting
    my $DEFAULT_REDIRECT = 'https://direct.yandex.ru/';
    $href =~ s/\s+/ /g;
    $href = $DEFAULT_REDIRECT unless $href =~ /\S/;
    my $uid_url;
    if ($opt) {
        $href .= $opt->{url_suffix} if $opt->{url_suffix};
        $uid_url = $opt->{uid_url};
        my @additions;
        for my $k (sort keys %$opt) {
            next if $k eq 'uid_url';
            next if $k eq 'url_suffix';
            my @vals = grep {defined} ref $opt->{$k} ? @{$opt->{$k}} : $opt->{$k};
            push @additions, map {uri_escape_utf8($k)."=".uri_escape_utf8($_)} @vals;
        }
        if (@additions) {
            $href .= ($href =~ /\?/ ? '&' : '?').join('&', @additions);
        }
    }
    # добавляем uid_url (&ulogin=<login> или &uid=<uid>), если требуется
    if (defined $uid_url) {
        $href .= $uid_url;
    }
    # добавляем якорь, если надо
    if (defined $anchor) {
        $href .= "#".uri_escape_utf8($anchor);
    }

    # проверяем редирект, разрешаем редиректы только на яндексовые домены
    # (пока только логгируем)
    my $is_good_redirect;
    my $href_uri = URI->new($href);
    if ($href_uri->scheme) {
        $is_good_redirect = $href_uri->host =~ /^ (.*\.|) yandex ((\.com)? \.[a-z]+)? \.? $/ix ? 1 : 0;
    } else {
        $is_good_redirect = 1;
    }

    # делаем редирект
    LogTools::log_redirects({location => $href, is_good => $is_good_redirect});
    $href = $DEFAULT_REDIRECT unless $is_good_redirect;
    $r->headers_out->set(Location => $href);
    $r->status(302);
    return 302;
}


#======================================================================

=head2 redirect_to_https

    редиректит на https если находимся на http
    возвр. 1 если редирект был

    использование:
    return if redirect_to_https($r, \%FORM, $SCRIPT, [qw/cmd save cl_login cl_name cl_phone cl_email use_deleted/], $is_beta_flag);

=cut

sub redirect_to_https
{
    my ($r, $FORM, $SCRIPT, $copy_form_keys, $is_beta) = @_;

    if ($SCRIPT =~ /^http:/) {
        my $https_script = $SCRIPT;
        $https_script =~ s/^http:/https:/;
        $https_script =~ s[:\d+/][/] if $is_beta; # remove port for beta
        redirect($r, $https_script, { map {$_ => $FORM->{$_}} grep {defined $FORM->{$_}} @$copy_form_keys });
        return 1;
    } else {
        return 0;
    }
}

#======================================================================

=head2 redirect_to_check_sauth

    проверяем платежный пароль
    возвр. 1 если редирект был
    работает только на https!
    см. https://wiki.yandex-team.ru/passport/checksauth

    использование:
    return if redirect_to_check_sauth($r, \%FORM, $SCRIPT, $UID, [qw/cmd save cl_login cl_name cl_phone cl_email use_deleted/]);
    return if redirect_to_check_sauth($r, \%FORM, \$SCRIPT, $UID, [qw/cmd save cl_login cl_name cl_phone cl_email use_deleted/]);

=cut

sub redirect_to_check_sauth
{
    my ($r, $FORM, $SCRIPT_arg, $UID, $client_id, $copy_form_keys) = @_;

    my $SCRIPT = ref($SCRIPT_arg) ? $$SCRIPT_arg : $SCRIPT_arg;

    my $new_authentication_enabled = Client::ClientFeatures::has_new_master_token_authentication_feature($client_id);

    if ($new_authentication_enabled) {
        # менять алгоритм хеширования нужно синхронно с аналогичным изменением в java коде (см. DirectCookieAuthProvider)
        my $direct_session_id = md5_hex_utf8(get_cookie($r, 'Session_id'));

        my $trace_id = Yandex::Trace::current_span_id();
        my $trace_header = join(',', map {$_ // 0} $trace_id, Yandex::Trace::generate_traceid(), $trace_id, 0);
        my $ticket = eval { Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{intapi}) } or die "Cannot get ticket for $Settings::TVM2_APP_ID{intapi}: $@";

        my %opts = (
            timeout => 5,
            num_attempts => 2,
            ipv6_prefer => 1,
            headers => {
                'Content-Type' => 'application/json',
                'X-Ya-Service-Ticket' => $ticket,
                'X-Yandex-Trace' => $trace_header,
                'Direct-Session-Id' => $direct_session_id,
            },
        );

        my $response_raw = http_fetch('GET', $Settings::CHECK_AUTH_FOR_SMS_JAVA_HTTP_URL, [operatorUid => $UID], %opts);
        my $response_decoded = decode_json($response_raw);

        if ($response_decoded->{isUserAuthenticated} eq $JSON::XS::true) {
            return 0;
        } else {
            my $retpath = build_retpath_for_check_sauth($r, $FORM, $SCRIPT, $UID, $client_id, $copy_form_keys);
            redirect($r,  $SCRIPT, {cmd => 'smsAuthentication', retpath => $retpath});
            return 1;
        }
    } else {
        my $lwp = LWP::UserAgent->new(max_redirect => 0);
        $lwp->default_header('Cookie' => $ENV{HTTP_COOKIE});

        my $response = $lwp->request(HTTP::Request->new('GET', 'https://sauth.yandex.ru/passport?mode=checksauth'));

        if ($response->code() == 500) {
            my $retpath = build_retpath_for_check_sauth($r, $FORM, $SCRIPT, $UID, $client_id, $copy_form_keys);
            redirect($r, 'https://sauth.yandex.ru/cgi-bin/Reg.pl', {mode => 'sauth', retpath => $retpath});
            return 1;

        } elsif ($response->code() == 302) {

            # проверяем что UID в xml (в теле ответа паспорта) такой же как и у нас ($UID)
            my $xml_doc = XML::LibXML->new()->parse_string($response->content());
            my $uid_from_sauth = XML::LibXML::XPathContext->new($xml_doc)->find("/page/uid")->string_value();
            die "checksauth return uid: $uid_from_sauth != $UID (\$UID)" if $UID != $uid_from_sauth;

            # подменяем домен в $SCRIPT, если нам передали ссылку на него
            if (ref($SCRIPT_arg)) {
                my $original_yandex_domain = $FORM->{original_yandex_domain};
                if ($original_yandex_domain && $original_yandex_domain =~ /^yandex\.(?:ru|ua|com\.tr|com|kz|by)$/) {
                    $$SCRIPT_arg =~ s/\Qyandex.ru\E/$original_yandex_domain/;
                }
            }

            return 0;

        } else {
            die("check https://sauth.yandex.ru/passport?mode=checksauth failed: " . Dumper({UID => $UID, FORM => $FORM, response_code => $response->code()}));
        }
    }
}

=head2 build_retpath_for_check_sauth

    Строит URL для редиректа со страницы авторизации

=cut

sub build_retpath_for_check_sauth {
    my ($r, $FORM, $SCRIPT, $UID, $client_id, $copy_form_keys) = @_;

    my $retpath = "$SCRIPT?";
    # при проверке платёжного пароля паспорт выставляет куку Secure_session_id только на домен .yandex.ru
    # чтобы в дальнейшем нам её прислали, редиректим на direct.yandex.ru, а затем возвращаемся обратно на исходный домен
    my $yandex_domain = yandex_domain($r);
    if ($yandex_domain && $yandex_domain ne 'yandex.ru') {
        $retpath =~ s/\Q$yandex_domain\E/yandex.ru/;
        $retpath .= '&original_yandex_domain=' . uri_escape_utf8($yandex_domain);
    }
    for (grep {defined $FORM->{$_}} @$copy_form_keys) {
        $retpath .= "&$_=" . uri_escape_utf8($FORM->{$_});
    }
    my $redirlevel = defined $FORM->{redirlevel} && $FORM->{redirlevel} =~ /^\d+$/ ? $FORM->{redirlevel} : 0;
    if ($redirlevel > 10) {
        die "https://sauth.yandex.ru/cgi-bin/Reg.pl redirect level exceeded " . Dumper({UID => $UID, FORM => $FORM, redirlevel => $redirlevel});
    } else {
        $redirlevel++;
    }
    $retpath .= "&redirlevel=$redirlevel";

    return $retpath;
}

=head2 check_output_data
    Проверяет структуру данных на соответствие выходной схеме для текущей cmd.
    Текущая cmd и заданная для нее конфигурация проверки определяются по контексту.
    Если не удалось определить cmd или для нее не задана конфигурация проверки - проверка не выполняется.

    Если в конфигурации задан жесткий уровень проверки и структура данных не соответствует схеме -
    она будет переопределена структурой ошибок валидации.

=cut

sub check_output_data {
    my ($r, $data) = @_;

    my ($cmd, $conf, $errors);
    eval {
        no warnings 'once';
        $cmd = $DoCmd::CURRENT_CMD;
        $conf = $DoCmd::Base::CheckBySchema{$cmd} if $cmd;
        1;
    } || warn $@;

    return $data unless $cmd && $conf && ref $data;

    ($data, $errors) =  DoCmd::CheckBySchema::check($cmd, $conf, output => $data);

    return error_to(
        $r,
            html => [iget('Структура данных ответа неверна'), {cmd => $cmd, %$errors}],
            json => [{cmd => $cmd, %$errors}],
        ) if $errors;

    return $data;
}


1;
