package Direct::RetargetingConditionIntAPIClient;

use JSON;
use Settings;
use Yandex::HTTP;
use Yandex::I18n;
use Yandex::Trace;
use Yandex::TVM2;

use Direct::Modern;

our $INTAPI_REQUEST_TIMEOUT = 15;

our @EXPORT = qw/
    create_retargeting_condition
    update_retargeting_condition
    validate_create_retargeting_condition
    validate_update_retargeting_condition
/;


=head2 create_retargeting_condition($ClientID, $retargeting_condition)

Создает переданное условие ретаргетинга через ручку intapi-java
Возвращает ($push_result, $push_error). $push_result->{ret_cond_id} содержит id созданного объекта

=cut

sub create_retargeting_condition {
    my ($ClientID, $retargeting_condition) = @_;
    return _send_retargeting_condition('create', JSON::false, $ClientID, $retargeting_condition);
}

=head2 update_retargeting_condition($ClientID, $retargeting_condition)

Обновляет переданное условие ретаргетинга через ручку intapi-java
Возвращает $push_error

=cut

sub update_retargeting_condition {
    my ($ClientID, $retargeting_condition) = @_;
    return (_send_retargeting_condition('update', JSON::false, $ClientID, $retargeting_condition))[1];
}

=head2 validate_create_retargeting_condition($ClientID, $retargeting_condition)

Выполняет валидацию создания переданного условия ретаргетинга через ручку intapi-java
Возвращает $push_error

=cut

sub validate_create_retargeting_condition {
    my ($ClientID, $retargeting_condition) = @_;
    return (_send_retargeting_condition('create', JSON::true, $ClientID, $retargeting_condition))[1];
}

=head2 validate_update_retargeting_condition($ClientID, $retargeting_condition)

Выполняет валидацию обновления переданного условия ретаргетинга через ручку intapi-java
Возвращает $push_error

=cut

sub validate_update_retargeting_condition {
    my ($ClientID, $retargeting_condition) = @_;
    return (_send_retargeting_condition('update', JSON::true, $ClientID, $retargeting_condition))[1];
}

sub _send_retargeting_condition {
    my ($method, $validation_only, $ClientID, $retargeting_condition) = @_;

    my $profile = Yandex::Trace::new_profile(
        $validation_only ? 'java_retargeting:condition_validate' : 'java_retargeting:condition_save');

    state ($log, $json);
    $log //= Yandex::Log->new(
        no_log => 1,
        use_syslog => 1,
        syslog_prefix => 'INTAPI-JAVA',
        log_file_name => "RetargetingConditionIntAPIClient.log",
    );
    $json //= JSON->new->allow_unknown(1)->allow_blessed(1)->convert_blessed(1);

    my $url = Yandex::HTTP::make_url(
        $Settings::DIRECT_INTAPI_URL . 'retargeting/condition/' . $method,
        { client_id => [ $ClientID ], validation_only => $validation_only ? 'true' : 'false' }
    );

    my $body = encode_json({
        id => $retargeting_condition->{ret_cond_id},
        name => $retargeting_condition->{condition_name},
        type => $retargeting_condition->{type},
        description => $retargeting_condition->{condition_desc},
        conditions => $retargeting_condition->{groups}
    });

    my $trace_id = Yandex::Trace::current_span_id();
    my $trace_header = join(',', map {$_ // 0} $trace_id, Yandex::Trace::generate_traceid(), $trace_id, 0);
    my $ticket = eval{Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{intapi})} or $log->die("Cannot get ticket for $Settings::TVM2_APP_ID{intapi}: $@");

    $log->out(['REQUEST:', 'POST', $url, 'BODY:', $body, 'X-Yandex-Trace:', $trace_header]);

    my $response = Yandex::HTTP::http_parallel_request(POST => { 1 => {
            url => $url,
            body => $body
        }},
        timeout => $INTAPI_REQUEST_TIMEOUT,
        log => $log,
        headers => {
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'Accept-Language' => Yandex::I18n::current_lang(),
            'X-Ya-Service-Ticket' => $ticket,
            'X-Yandex-Trace' => $trace_header
        },
    )->{1};

    my ($result, $error);
    if ($response->{is_success}) {
        utf8::decode($response->{content});
        my $content = $json->decode($response->{content});
        if (!$content->{success}) {
            $error = $content;
        } else {
            $result = $content->{id} if $response->{content};
        }
    } else {
        $error = {
            server_status => $response->{headers}->{Status},
            server_error => 'Server returned an error'
        };
    }

    $log->out(['RESPONSE:', $result // $response->{content}, ( $error ? ( $response->{headers}->{Reason}) : () ) ]);

    return $result, $error;
}

1;
